import { IPSAppView } from '@ibiz-template/model';
import {
  IModal,
  IModalData,
  IOpenViewUtil,
  safeRedirect,
  ViewMode,
} from '@ibiz-template/runtime';
import { generateRoutePath } from '@ibiz-template/vue-util';
import { UrlHelper } from '@ibiz-template/core';
import router from '@/router';
import {
  getDrawerPlacement,
  openViewDrawer,
  openViewModal,
  openViewPopover,
} from '../overlay-view-util/overlay-view-util';

/**
 * 打开视图方式工具类
 *
 * @description 此实现类挂载在 ibiz.openViewUtil
 * @author chitanda
 * @date 2022-08-16 20:08:54
 * @export
 * @class OpenViewUtil
 * @implements {IOpenViewUtil}
 */
export class OpenViewUtil implements IOpenViewUtil {
  root(
    appView: IPSAppView,
    context: IContext,
    params?: IParams | undefined,
  ): void {
    const { path } = generateRoutePath(
      appView,
      router.currentRoute,
      context,
      params,
    );

    router.push({ path });
  }

  /**
   * 模态打开视图
   *
   * @author lxm
   * @date 2022-09-12 01:09:06
   * @param {IPSAppView} appView
   * @param {(IContext | undefined)} [context]
   * @param {(IParams | undefined)} [params]
   * @returns {*}  {Promise<IModalData>}
   */
  async modal(
    appView: IPSAppView,
    context: IContext,
    params?: IParams | undefined,
    _opts?: IData,
  ): Promise<IModalData> {
    // 获取视图path
    const modelPath = appView.modelPath!;

    // 设置默认的modal参数
    const opts = {
      width: appView.width || '80%',
      height: appView.height || '80%',
      footerHide: true,
    };

    const modal: IModal = { mode: ViewMode.MODAL };
    return openViewModal(
      {
        context,
        params,
        modelPath,
        modal,
        ..._opts,
      },
      opts,
    );
  }

  async popover(
    appView: IPSAppView,
    event: MouseEvent,
    context: IContext,
    params?: IParams | undefined,
  ): Promise<IModalData> {
    // 获取视图path
    const modelPath = appView.modelPath!;

    const modal: IModal = { mode: ViewMode.POPOVER };
    return openViewPopover(
      event,
      {
        context,
        params,
        modelPath,
        modal,
      },
      { autoClose: true, placement: 'bottom' },
    );
  }

  /**
   * 抽屉打开视图
   *
   * @author lxm
   * @date 2022-09-15 15:09:50
   * @param {IPSAppView} appView
   * @param {(IContext | undefined)} [context]
   * @param {(IParams | undefined)} [params]
   * @returns {*}  {Promise<IModalData>}
   */
  async drawer(
    appView: IPSAppView,
    context: IContext,
    params?: IParams | undefined,
  ): Promise<IModalData> {
    // 获取视图path
    const modelPath = appView.modelPath!;
    const placement = getDrawerPlacement(appView.openMode);

    // 设置默认的modal参数
    const opts = {
      width: appView.width || '800',
      height: appView.height || '600',
      placement,
    };

    const modal: IModal = { mode: ViewMode.DRAWER };
    return openViewDrawer(
      {
        context,
        params,
        modelPath,
        modal,
      },
      opts,
    );
  }

  async custom(
    appView: IPSAppView,
    context: IContext,
    params?: IParams | undefined,
  ): Promise<IModalData> {
    ibiz.log.warn('openUserCustom', appView, context, params);
    throw new Error();
  }

  /**
   * @description 独立程序弹出
   * @param {IPSAppView | string} appView
   * @param {IContext} context
   * @param {IParams} [params] (srfopenmode: 打开方式，srfopentarget：打开目标)
   * @returns {*}  {Promise<void>}
   * @memberof OpenViewUtil
   */
  async popupApp(
    appView: IPSAppView | string,
    context: IContext,
    params?: IParams,
  ): Promise<void> {
    const { srfopenmode = 'open', srfopentarget = '_blank' } = params || {};
    let url = '';
    if (typeof appView === 'string') {
      url = appView;
    } else {
      const { path } = await generateRoutePath(
        appView,
        router.currentRoute,
        context,
        params,
      );
      url = `${UrlHelper.routeBase}${path}`;
    }
    if (srfopenmode === 'open') {
      safeRedirect(url, { type: srfopentarget });
    }
    // 其他方式待补充
  }
}
