import { FormController } from '@ibiz-template/controller';
import { IBizContext } from '@ibiz-template/core';
import {
  FormDetailModel,
  FormGroupPanelModel,
  FormTabPanelModel,
  FormTabPageModel,
  FormModel,
  FormItemModel,
} from '@ibiz-template/model';
import { useForceTogether, useNamespace } from '@ibiz-template/vue-util';
import {
  defineComponent,
  getCurrentInstance,
  toRef,
  computed,
  PropType,
} from 'vue';
import '@/styles/components/widgets/form/form.scss';

/**
 * 根据类型绘制表单成员
 *
 * @author lxm
 * @date 2022-08-23 16:08:31
 * @param {FormDetailModel} detail 成员模型
 * @param {FormController} controller 表单控制器
 * @returns {*}
 */
function renderByDetailType(
  detail: FormDetailModel,
  controller: FormController,
) {
  const commonProps = {
    modelData: detail,
    controller: computed(() => controller.details[detail.source.name]),
  };
  switch (detail.source.detailType) {
    // 分页部件
    case 'TABPANEL':
      return (
        <form-tab-panel key={detail.id} props={{ ...commonProps }}>
          {(detail as FormTabPanelModel).children.map(page => {
            return (
              <form-tab-page
                key={page.id}
                props={{
                  modelData: page,
                  controller: controller.details[page.source.name],
                  caption: page.source.caption,
                }}
              >
                {(page as FormTabPageModel).children.map(item =>
                  renderByDetailType(item, controller),
                )}
              </form-tab-page>
            );
          })}
        </form-tab-panel>
      );
    // 分组面板
    case 'GROUPPANEL':
      return (
        <form-group-panel key={detail.id} props={{ ...commonProps }}>
          {(detail as FormGroupPanelModel).children.map(item =>
            renderByDetailType(item, controller),
          )}
        </form-group-panel>
      );
    // 表单项
    case 'FORMITEM':
      if ((detail as FormItemModel).source.hidden) {
        return null;
      }
      return (
        <app-form-item
          key={detail.id}
          props={{ ...commonProps }}
        ></app-form-item>
      );
    // 表单按钮
    case 'BUTTON':
      return (
        <form-button key={detail.id} props={{ ...commonProps }}></form-button>
      );
    // 关系界面
    case 'DRUIPART':
      return (
        <form-druipart
          key={detail.id}
          props={{ ...commonProps }}
        ></form-druipart>
      );
    // 直接内容
    case 'RAWITEM':
      return (
        <form-raw-item
          key={detail.id}
          props={{ ...commonProps }}
        ></form-raw-item>
      );
    default:
      return <div>暂未支持的表单项类型: {detail.source.detailType}</div>;
  }
}

export const FormControl = defineComponent({
  name: 'FormControl',
  props: {
    modelData: {
      type: FormModel,
      required: true,
    },
    context: IBizContext,
    controller: {
      type: Object as PropType<FormController>,
      required: true,
    },
  },
  setup(props) {
    const { proxy } = getCurrentInstance()!;
    useForceTogether(proxy, props.controller);

    const ns = useNamespace('form');

    const c = toRef(props, 'controller');

    return { ns, c };
  },
  render(_h) {
    return (
      <control-layout modelData={this.c.model}>
        {this.c.complete && (
          <div class={this.ns.b()}>
            <form-page modelData={this.c.model}>
              {this.$props.modelData.children.map(page => {
                return (
                  <form-page-item
                    key={page.id}
                    caption={page.source.caption}
                    model-data={page}
                    controller={this.c.details[page.source.name]}
                  >
                    {page.children.map(detail =>
                      renderByDetailType(detail, this.c),
                    )}
                  </form-page-item>
                );
              })}
            </form-page>
          </div>
        )}
      </control-layout>
    );
  },
});
