import { computed, defineComponent, ref, watch } from 'vue';
import { debounce, floor } from 'lodash-es';
import {
  getEditorEmits,
  getInputProps,
  useNamespace,
} from '@ibiz-template/vue-util';
import '@ibiz-template/theme/style/components/editor/ibiz-input/ibiz-input.scss';
import xss from 'xss';

const fomatFloat = function (value: number, n: number) {
  const f = value;
  let s = f.toString();
  if (n <= 0) {
    return s;
  }
  const rs = s.indexOf('.');
  if (rs < 0) {
    s += '.';
  }
  for (let i = s.length - s.indexOf('.'); i <= n; i++) {
    s += '0';
  }
  return s;
};

export const IBizInput = defineComponent({
  name: 'IBizInput',
  props: getInputProps(),
  emits: getEditorEmits(),
  setup(props, { emit }) {
    const ns = useNamespace('input');
    const c = props.controller;
    const editorModel = c.model;
    const inputRef = ref();
    let autoSize: boolean | { minRows: Number; maxRows?: Number } = {
      minRows: 2,
    };

    // 文本域默认行数，仅在 textarea 类型下有效
    const rows = ref(2);
    if (editorModel.editorType === 'TEXTAREA_10') {
      rows.value = 10;
      autoSize = {
        minRows: 10,
      };
    }

    // 类型
    const type = computed(() => {
      if (c.model.precision !== -1) {
        return 'number';
      }
      switch (editorModel.editorType) {
        case 'TEXTBOX':
          return 'text';
        case 'PASSWORD':
          return 'password';
        case 'TEXTAREA':
        case 'TEXTAREA_10':
          return 'textarea';
        default:
          return 'string';
      }
    });

    if (editorModel.editorParams) {
      if (editorModel.editorParams.autosize) {
        try {
          autoSize = JSON.parse(editorModel.editorParams.autosize);
        } catch {
          autoSize = false;
        }
      }
    }
    const currentVal = ref<string>('');

    const getInputValue = (value: string | number) => {
      if (type.value === 'number' || !ibiz.config.enableXSS) {
        return value;
      }
      const result = xss(value as string);
      if (result !== value) {
        currentVal.value = result;
        inputRef.value?.setCurrentValue?.(result);
        ibiz.message.warning('输入值存在不规范格式，已自动调整！');
      }
      return result;
    };

    watch(
      () => props.value,
      (newVal, oldVal) => {
        if (newVal !== oldVal) {
          if (!newVal) {
            currentVal.value = '';
          } else {
            currentVal.value = newVal;
          }
        }
      },
      { immediate: true },
    );

    let isDebounce = false;
    let awaitSearch: () => void;
    let blurCacheValue: string | undefined;
    // 防抖值变更回调
    const debounceChange = debounce(
      (e: IData) => {
        // 拦截掉blur触发后change
        if (blurCacheValue !== e.target.value) {
          emit('change', getInputValue(e.target.value));
        }
        blurCacheValue = undefined;
        isDebounce = false;
        if (awaitSearch) {
          awaitSearch();
        }
      },
      300,
      { leading: true },
    );

    const matchNum = (e: IData) => {
      const newValue = e.target.value;
      const regexString = `^(\\d+\\.?\\d{0,${c.model.precision}})$`;
      const match = new RegExp(regexString).exec(newValue);
      return match;
    };

    // 值变更
    const handleChange = (e: IData) => {
      isDebounce = true;
      if (c.model.precision !== -1) {
        const match = matchNum(e);
        if (match) {
          debounceChange(e);
        }
      } else {
        debounceChange(e);
      }
    };

    const handleKeyUp = (e: KeyboardEvent) => {
      if (e && e.code === 'Enter') {
        if (isDebounce) {
          awaitSearch = () => {
            inputRef.value.$el.dispatchEvent(e);
          };
        }
      }
    };

    /**
     * blur时马上抛值变更
     * @author lxm
     * @date 2023-03-06 06:36:23
     */
    const handleBlur = () => {
      blurCacheValue = inputRef.value.currentValue;
      if (c.model.precision !== -1) {
        const num = floor(Number(blurCacheValue), c.model.precision);
        const number = blurCacheValue ? fomatFloat(num, c.model.precision) : '';
        currentVal.value = number;
        const input = inputRef.value?.$refs?.input;
        if (input) {
          input.value = number;
        }
        inputRef.value?.setCurrentValue?.(number);
        emit('change', number);
      } else {
        emit('change', getInputValue(blurCacheValue as string));
      }
    };

    // 自动聚焦
    if (props.autoFocus) {
      watch(inputRef, newVal => {
        if (newVal) {
          let input = newVal.$el.getElementsByTagName('input')[0];
          if (!input) {
            input = newVal.$el.getElementsByTagName('textarea')[0];
          }
          input.focus();
        }
      });
    }

    return {
      ns,
      rows,
      type,
      currentVal,
      handleChange,
      handleKeyUp,
      handleBlur,
      inputRef,
      autoSize,
      c,
    };
  },
  render(h) {
    return (
      <div
        class={[
          this.ns.b(),
          this.ns.is('textarea', Object.is(this.type, 'textarea')),
          this.disabled ? this.ns.m('disabled') : '',
          this.readonly ? this.ns.m('readonly') : '',
        ]}
      >
        {[
          this.readonly && this.currentVal,
          !this.readonly &&
            h(
              'IInput',
              {
                ref: 'inputRef',
                class: [
                  this.ns.b('input'),
                  this.ns.is(
                    'disabledresize',
                    this.c.editorParams.disabledresize === 'true',
                  ),
                ],
                props: {
                  ...this.c.customProps,
                  value: this.currentVal,
                  clearable: true,
                  placeholder: this.controller.placeHolder,
                  type: this.type,
                  rows: this.rows,
                  disabled: this.disabled,
                  autosize: this.autoSize,
                },
                on: {
                  'on-change': this.handleChange,
                  'on-blur': this.handleBlur,
                },
                nativeOn: {
                  keyup: this.handleKeyUp,
                },
              },
              [
                this.controller.model.unitName &&
                  h('i', { class: this.ns.e('unit'), slot: 'suffix' }, [
                    this.controller.model.unitName,
                  ]),
              ],
            ),
        ]}
      </div>
    );
  },
});
