import keys from './raw-material-keys';
import { IContext, IEntityLocalDataService, IParams } from 'ibiz-core';
import { GlobalService } from '../../service';
import { IRawMaterial } from '../interface';

/**
 * RawMaterialDTOdto辅助类
 *
 * @export
 * @class RawMaterialDTOHelp
 */
export class RawMaterialDTOHelp {

    /**
     * 获取数据服务
     *
     * @param {IContext} context 应用上下文对象
     * @returns {*}
     * @memberof RawMaterialDTOHelp
     */
    public static async getService(context: IContext): Promise<IEntityLocalDataService<IRawMaterial>> {
        return new GlobalService().getService('RawMaterial',context);
    }

    /**
     * DTO转化成数据对象
     *
     * @param {IContext} context 应用上下文对象
     * @param {IParams} source dto对象
     * @returns {*}
     * @memberof RawMaterialDTOHelp
     */
    public static async ToDataObj(context: IContext, source: IParams) {
        const _data: any = {};
        // 建立时间
        _data.createdate = source.createdate;
        // 建立人
        _data.createman = source.createman;
        // 原材料标识
        _data.rawmaterialid = source.rawmaterialid;
        // 原材料名称
        _data.rawmaterialname = source.rawmaterialname;
        // 更新时间
        _data.updatedate = source.updatedate;
        // 更新人
        _data.updateman = source.updateman;
        // 处理预置字段（以srf开头）
        if(source && Object.keys(source).length >0){
            Object.keys(source).forEach((key:string) =>{
                if(key.startsWith('srf')){
                    _data[key] = source[key];
                }
            })
        }
        return _data;
    }

    /**
     * 转化数组(dto转化成数据对象)
     *
     * @param {IContext} context 应用上下文对象
     * @param {any[]} data 数据对象
     * @returns {any[]}
     * @memberof RawMaterialDTOHelp
     */
    public static async ToDataObjArray(context: IContext, data: any[]) {
        const _data: any[] = [];
        if (data && Array.isArray(data) && data.length > 0) {
            for (let i = 0; i < data.length; i++) {
                const tempdata = await this.ToDataObj(context, data[i]);
                _data.push(tempdata);
            }
        }
        return _data;
    }

    /**
     * 数据对象转化成DTO
     *
     * @param {IContext} context 应用上下文对象
     * @param {*} source 数据对象
     * @returns {*}
     * @memberof RawMaterialDTOHelp
     */
    public static async ToDto(context: IContext, source: IParams) {
        const _data: any = {};
        const excludes: string[] = [];
        // 建立时间
        _data.createdate = source.createdate;
        // 建立人
        _data.createman = source.createman;
        // 原材料标识
        _data.rawmaterialid = source.rawmaterialid;
        // 原材料名称
        _data.rawmaterialname = source.rawmaterialname;
        // 更新时间
        _data.updatedate = source.updatedate;
        // 更新人
        _data.updateman = source.updateman;
        // 附加额外数据
        if (source && Object.keys(source).length > 0 && keys) {
            Object.keys(source).forEach((key: string) => {
                if (!keys.includes(key) && !excludes.includes(key)) {
                    _data[key] = source[key];
                }
            })
        }
        return _data;
    }

    /**
     * 转化数组(数据对象转化成dto)
     *
     * @param {IContext} context 应用上下文对象
     * @param {any[]} data 
     * @returns {any[]}
     * @memberof RawMaterialDTOHelp
     */
    public static async ToDtoArray(context: IContext, data: any[]) {
        const _data: any[] = [];
        if (data && Array.isArray(data) && data.length > 0) {
            for (let i = 0; i < data.length; i++) {
                const tempdata = await this.ToDto(context, data[i]);
                _data.push(tempdata);
            }
        }
        return _data;
    }

    /**
     * 处理响应dto对象
     *
     * @param {*} context 应用上下文对象
     * @param {*} data 响应dto对象
     * @returns {*}
     * @memberof RawMaterialDTOHelp
     */
    public static async set(context: IContext, data: any) {
        const _data: IParams = await this.ToDataObj(context, data);
        return _data;
    }

    /**
     * 处理请求数据对象
     *
     * @param {*} context 应用上下文对象
     * @param {*} data 数据对象
     * @returns {*}
     * @memberof RawMaterialDTOHelp
     */
    public static async get(context: IContext, data: any = {}) {
        return await this.ToDto(context, data);
    }

    /**
     * 获取缓存数据
     *
     * @param {*} context 应用上下文对象
     * @param {*} srfkey 数据主键
     * @returns {*}
     * @memberof RawMaterialDTOHelp
     */
    public static async getCacheData(context: IContext, srfkey: string) {
        const targetService: IEntityLocalDataService<IRawMaterial> = await this.getService(context);
        const result =  await targetService.getLocal(context, srfkey);
        if(result){
            return await this.ToDto(context,result);
        }
    }

    /**
     * 获取缓存数组
     *
     * @param {*} context 应用上下文对象
     * @returns {any[]}
     * @memberof RawMaterialDTOHelp
     */
    public static async getCacheDataArray(context: IContext) {
        const targetService: IEntityLocalDataService<IRawMaterial> = await this.getService(context);
        const result =  await targetService.getLocals(context);
        if(result && result.length >0){
            return await this.ToDtoArray(context,result);
        }else{
            return [];
        }
    }

    /**
     * 设置缓存数据
     *
     * @param {*} context 应用上下文对象
     * @param {any} data 数据
     * @returns {any[]}
     * @memberof RawMaterialDTOHelp
     */
    public static async setCacheData(context: IContext, data: any) {
        const targetService: IEntityLocalDataService<IRawMaterial> = await this.getService(context);
        const _data: any = await this.set(context, data);
        await targetService.addLocal(context, _data);
    }

    /**
     * 设置缓存数组
     *
     * @param {*} context 应用上下文对象
     * @param {any[]} data 数据
     * @returns {any[]}
     * @memberof RawMaterialDTOHelp
     */
    public static async setCacheDataArray(context: IContext, data: any[]) {
        if (data && data.length > 0) {
            const targetService: IEntityLocalDataService<IRawMaterial> = await this.getService(context);
            for (let i = 0; i < data.length; i++) {
                const _data = await this.set(context, data[i]);
                await targetService.addLocal(context, _data);
            }
        }
    }


}
