import { IPSCodeListEditor, IPSEditor } from '@ibiz/dynamic-model-api';
import { AppServiceBase, ModelTool, Util } from 'ibiz-core';
import { Vue, Prop, Emit } from 'vue-property-decorator';
import { Watch } from '../../../decorators/vue-lifecycleprocessing';

/**
 * editor解析器基类
 *
 * @export
 * @class EditorBase
 * @extends {Vue}
 */
export class EditorBase extends Vue {

    /**
     * 编辑器值
     *
     * @type {*}
     * @memberof EditorBase
     */
    @Prop() value!: any;

    /**
     * 编辑器模型
     *
     * @type {*}
     * @memberof EditorBase
     */
    @Prop() editorInstance!: IPSEditor;

    /**
     * 外层部件容器模型
     *
     * @type {*}
     * @memberof EditorBase
     */
    @Prop() containerCtrl!: any;

    /**
     * 父级项模型（表单项，表格项）
     *
     * @type {*}
     * @memberof EditorBase
     */
    @Prop() parentItem!: any;
    
    /**
     * 应用上下文
     *
     * @type {*}
     * @memberof EditorBase
     */
    @Prop() public context!: any;

    /**
     * 视图参数
     *
     * @type {*}
     * @memberof EditorBase
     */
    @Prop() public viewparams!: any;

    /**
     * 上下文data数据(form里的data，表格里的row)
     *
     * @type {*}
     * @memberof EditorBase
     */
    @Prop() public contextData?: any;

    /**
     * 是否禁用
     *
     * @type {*}
     * @memberof EditorBase
     */
    @Prop({default: false}) public disabled!: boolean;

    /**
     * 编辑器状态(表单里的formState)
     *
     * @type {*}
     * @memberof EditorBase
     */    
    @Prop() public contextState?: any;
    
    /**
     * 表单服务
     *
     * @type {*}
     * @memberof EditorBase
     */    
    @Prop() public service?: any;

    /**
     * 值格式化
     *
     * @type {boolean}
     * @memberof AppDefaultEditor
     */
    @Prop() public valueFormat?:any
    
    /**
     * 编辑器组件名称
     *
     * @type {*}
     * @memberof EditorBase
     */
    public editorComponentName!: string;

    /**
     * 动态样式表
     *
     * @type {*}
     * @memberof EditorBase
     */
    get dynaClass():any{
        if(this.editorInstance.dynaClass){
            const context = this.context;
            const viewparams = this.viewparams;
            const data = this.contextData;
            return eval(this.editorInstance.dynaClass);
        }
        return {};
    }

    /**
     * 自定义样式的对象
     *
     * @type {*}
     * @memberof EditorBase
     */
    public customStyle: any = {};

    /**
     * 设置自定义props
     *
     * @type {*}
     * @memberof EditorBase
     */
    public customProps: any = {};

    /**
     * 编辑器是否初始化完成
     * 
     * @type {boolean}
     * @memberof EditorBase
     */
    public editorIsLoaded: boolean = false;
    
    /**
     * 编辑器change事件
     *
     * @param {*} value
     * @memberof EditorBase
     */
    @Emit('change')
    public editorChange(value: any): void {}

    /**
     * 编辑器blur事件
     *
     * @param {*} event
     * @memberof EditorBase
     */
    @Emit('blur')
    public editorBlur(event: any): any {
        return { name: this.editorInstance.name, value: this.value, event: event };
    }

    /**
     * 编辑器enter事件
     *
     * @param {*} event
     * @memberof EditorBase
     */
    @Emit('enter')
    public editorEnter(event: any): any {
        return { name: this.editorInstance.name, value: this.value, event: event };
    }

    /**
     * 生命周期-created
     *
     * @memberof EditorBase
     */
    created() {}

    /**
     * editorJsonStr值变化
     *
     * @param {*} newVal
     * @param {*} oldVal
     * @memberof EditorBase
     */
    @Watch('editorInstance', { immediate: true})
    public onEditorInstanceChange(newVal: any, oldVal: any) {
        if (newVal && newVal != oldVal) {
            this.initEditorBase();
        }
    }

    /**
     * 编辑器初始化(基类)
     *
     * @memberof EditorBase
     */
    public async initEditorBase() {
        this.editorChange = this.editorChange.bind(this);
        this.editorBlur = this.editorBlur.bind(this);
        this.editorEnter = this.editorEnter.bind(this);
        this.editorComponentName = AppServiceBase.getInstance().getAppComponentService().getEditorComponents(this.editorInstance.editorType,this.editorInstance.editorStyle);
        this.setCustomStyle();
        this.setCustomProps();
        await this.initEditor();
        this.setEditorParams();
        this.editorIsLoaded = true;
    }

    /**
     * 编辑器初始化
     *
     * @memberof EditorBase
     */
    public async initEditor() {
        try {
            // 加载编辑器实体
            await (this.editorInstance as any)?.getPSAppDataEntity?.()?.fill();
            // 加载编辑器代码表
            await (this.editorInstance as IPSCodeListEditor)?.getPSAppCodeList?.()?.fill();
        } catch (error) {
            console.error(error)
        }
    }

    /**
     * 设置编辑器的自定义高宽
     *
     * @memberof EditorBase
     */
    public setCustomStyle() {
        let { editorWidth, editorHeight } = this.editorInstance;
        this.customStyle = {};
        if (!Util.isEmpty(editorWidth) && editorWidth != 0) {
            this.customStyle.width = editorWidth > 1 ? editorWidth + "px" : editorWidth * 100 + "%";
        }
        if (!Util.isEmpty(editorHeight) && editorHeight != 0) {
            this.customStyle.height = editorHeight > 1 ? editorHeight + "px" : editorHeight * 100 + "%";
        }
    }

    /**
     * 设置编辑器的自定义参数
     * 
     * @memberof EditorBase
     */
    public setCustomProps() {
        this.customProps = {
            placeholder: this.editorInstance.placeHolder,
        };
        // 只读状态
        if (this.editorInstance.readOnly) {
            Object.assign(this.customProps, {
                readonly: this.editorInstance.readOnly,
            });
        }
    }

    /**
     * 设置编辑器导航参数
     * 
     * @param keys 编辑器参数key
     * @memberof EditorBase
     */
    public setEditorParams() {
        let _this = this;
        if(!this.editorInstance.editorParams){
            return 
        }
        Object.assign(this.customProps, {
            localContext: ModelTool.getNavigateContext(this.editorInstance),
            localParam: ModelTool.getNavigateParams(this.editorInstance)
        });
        for (const key in this.editorInstance.editorParams) {
          let param: any;
          if (key == 'uploadparams' || key == 'exportparams' || key == 'readonly') {
            param = eval('(' + this.editorInstance.editorParams[key] + ')');
          }else {
            param = this.editorInstance.editorParams[key];
          }
            if(key.indexOf('.') != -1) {
                let splitArr: Array<any> = key.split('.');
                switch (splitArr[0]) {
                    case "SRFNAVPARAM":
                        Object.assign(this.customProps.localParam, { [splitArr[1]]: param });
                        break;
                    case "SRFNAVCTX":
                        Object.assign(this.customProps.localContext, { [splitArr[1]]: param });
                        break;
                }
            } else {
                if(param) {
                    this.customProps[key] = param;
                }
            }
        }
    }

    /**
     * @description 处理boolean
     * @param {string} value
     * @return {*} 
     * @memberof EditorBase
     */
    public handleBoolean(value: string) {
        return Object.is("string", typeof (value)) ? Object.is("true", value) ? true : false : value;
    }

    /**
     * @description 处理number
     * @param {string} value
     * @return {*} 
     * @memberof EditorBase
     */
    public handleNumber(value: string) {
        return Object.is("string", typeof (value)) ? +value : value;
    }

    /**
     * 绘制内容
     *
     * @returns {*}
     * @memberof EditorBase
     */
    public render(): any {
        return <div>{this.editorInstance ? this.$t('app.editor.nooutput') : this.$t('app.editor.noexist')}</div>;
    }
}
