import {
  defineComponent,
  getCurrentInstance,
  onUnmounted,
  ref,
  toRaw,
  watch,
} from 'vue';
import { ModelUtil } from '@ibiz-template/model';
import {
  parseRouteViewData,
  useRoute,
  useRouter,
} from '@ibiz-template/vue-util';
import { Route } from 'vue-router';
import { IBizContext } from '@ibiz-template/core';
import { ViewMode } from '@ibiz-template/runtime';

export default defineComponent({
  name: 'RouterShell',
  props: {
    level: {
      type: Number,
      default: 0,
    },
  },
  setup(props, ctx) {
    const { proxy } = getCurrentInstance()!;
    const route = useRoute(proxy) as Route;
    const router = useRouter(proxy);
    const viewData = ref<{
      viewPath?: string;
      viewType?: string;
      context?: IContext;
      params?: IParams;
      srfnav?: string;
    }>({});
    const isLoaded = ref(false);

    // 销毁视图上下文
    onUnmounted(() => {
      if (viewData.value.context) {
        const context = toRaw(viewData.value).context;
        if (context) context.destroy();
      }
    });

    // 视图初始化事件，往上抛
    const onNeuronInit = (...args: IData[]) => {
      ctx.emit('neuronInit', ...args);
    };

    const viewModal = {
      mode: ViewMode.ROUTE,
      level: props.level,
    };

    // 根据应用模型解析视图参数
    const calcViewData = async () => {
      const service = await ModelUtil.getModelService();
      const appModel = service.app;
      if (appModel) {
        // 获取视图
        try {
          const _viewData = parseRouteViewData(appModel, route, props.level);
          // 路由绘制的视图，添加一个上下文isRouter
          _viewData.context!.isRouter = true;
          const _context = IBizContext.create(_viewData.context);

          viewData.value = {
            ..._viewData,
            context: _context,
          };
          // 确定视图组件
          isLoaded.value = true;
          ctx.emit('viewFound', { modelPath: viewData.value.viewPath });
        } catch (error) {
          router.push({ name: `404View${props.level}` });
        }
      }
    };
    calcViewData();

    watch(
      () => route.params.view1,
      () => {
        if (props.level === 1) {
          // 第一级路由壳监测到view1变化后，就重新计算一遍viewData
          calcViewData();
        }
      },
    );

    return {
      route,
      viewData,
      isLoaded,
      onNeuronInit,
      viewModal,
    };
  },
  render(h) {
    if (!this.isLoaded) {
      return null;
    }
    return h('ViewShell', {
      props: {
        context: this.viewData.context,
        params: this.viewData.params,
        modelPath: this.viewData.viewPath,
      },
      on: {
        neuronInit: this.onNeuronInit,
      },
    });
  },
});
