import { useClickOutside, useNamespace } from '@ibiz-template/vue-util';
import { defineComponent, PropType, Ref, ref, toRefs, watch } from 'vue';
import { UploadEditorController } from '@ibiz-template/controller';
import {
  FileInfo,
  openImagePreview,
  useIBizUpload,
} from '../use/use-ibiz-upload';
import '@ibiz-template/theme/style/components/editor/ibiz-grid-file-upload/ibiz-grid-file-upload-popover.scss';

export const IBizGridFileUploadPopover = defineComponent({
  name: 'IBizGridFileUploadPopover',
  props: {
    uploadUrl: {
      type: String,
      required: true,
    },
    downloadUrl: {
      type: String,
      required: true,
    },
    value: {
      type: Array<{
        name: string;
        id: string;
        url: string;
      }>,
      required: true,
    },
    width: {
      type: Number,
      default: 300,
    },
    controller: {
      type: Object as PropType<Ref<UploadEditorController>>,
      required: true,
    },
    data: {
      type: Object as PropType<Ref<IData>>,
      required: true,
    },
  },
  emits: {
    close: (_result: { isModified: boolean; resultFiles: FileInfo[] }) => true,
  },
  setup(props, { emit }) {
    const ns = useNamespace('grid-file-upload-popover');

    // 使用文件上传逻辑
    const { downloadUrl, value, uploadUrl } = toRefs(props);
    const { fileList, uploadState, selectFile } = useIBizUpload({
      downloadUrl,
      value,
      uploadUrl,
      controller: props.controller,
      data: props.data,
    });

    const componentRef = ref();
    const isModified = ref(false);

    // 点击外部处理
    const outsideFns = useClickOutside(componentRef, async _evt => {
      emit('close', {
        isModified: isModified.value,
        resultFiles: fileList.value,
      });
    });

    // 上传完毕后数据发生修改,上传过程中暂停外部点击事件
    watch(uploadState, state => {
      if (state === 'done') {
        isModified.value = true;
        outsideFns.proceed();
      } else if (state === 'loading') {
        outsideFns.pause();
      }
    });

    /**
     * 删除文件
     *
     * @author lxm
     * @date 2022-11-18 16:11:10
     * @param {FileInfo} file
     */
    const deleteFile = (file: FileInfo) => {
      const index = fileList.value.findIndex(item => item.id === file.id);
      fileList.value.splice(index, 1);
      isModified.value = true;
    };

    const onImageClick = async (file: FileInfo) => {
      outsideFns.pause();
      await openImagePreview(file);
      outsideFns.proceed();
    };

    return {
      ns,
      componentRef,
      fileList,
      deleteFile,
      selectFile,
      onImageClick,
    };
  },
  render() {
    return (
      <div
        ref='componentRef'
        style={{ width: `${this.width}px` }}
        class={this.ns.b()}
      >
        <div class={this.ns.b('list')}>
          {this.fileList.map(file => {
            return (
              <div class={this.ns.be('list', 'item')}>
                {file.isImage && (
                  <div class={this.ns.be('list', 'preview')}>
                    <img
                      onClick={() => this.onImageClick(file)}
                      class={this.ns.be('list', 'preview-image')}
                      src={file.url}
                    />
                  </div>
                )}
                <div class={this.ns.be('list', 'info')}>
                  <a
                    class={this.ns.be('list', 'file-name')}
                    href={file.url || ''}
                    download={file.name}
                  >
                    {file.name}
                  </a>
                  <div class={this.ns.be('list', 'actions')}>
                    <i-icon
                      class={this.ns.be('list', 'action')}
                      onClick={() => this.deleteFile(file)}
                      type='md-close'
                    />
                  </div>
                </div>
              </div>
            );
          })}
        </div>
        <div onClick={() => this.selectFile()} class={this.ns.e('add-button')}>
          <i-icon type='md-add' />
          添加文件
        </div>
        <div class={this.ns.e('description')}>
          {this.controller.uploadDescription}
        </div>
      </div>
    );
  },
});
