import { defineComponent, ref } from 'vue';
import { IndexViewModel } from '@ibiz-template/model';
import { useNamespace } from '@ibiz-template/vue-util';
import { Icon, Layout } from 'view-design';
import '@/styles/components/layout/app-layout/app-layout.scss';
import { Namespace } from '@ibiz-template/core';

function renderLogo(ns: Namespace, model: IndexViewModel) {
  return (
    <div class={ns.e('logo')}>
      {model.appIconPath ? (
        <img class={ns.e('logo-img')} src={model.appIconPath}></img>
      ) : null}
      <div class={ns.e('logo-caption')}>{model.caption}</div>
    </div>
  );
}

export const AppLayout = defineComponent({
  name: 'AppLayout',
  props: {
    model: {
      type: IndexViewModel,
      required: true,
    },
    // 视图是否完成加载
    isComplete: {
      type: Boolean,
      default: false,
    },
  },
  setup(props, { emit }) {
    const ns = useNamespace('layout');

    // 菜单收缩变化
    const collapseChange = ref(false);

    const collapseMenus = () => {
      collapseChange.value = !collapseChange.value;
      emit('onCollapseChange', collapseChange.value);
    };

    const onBackClick = (event: MouseEvent) => {
      event.stopPropagation();
      emit('backClick');
    };

    return { ns, collapseChange, collapseMenus, onBackClick };
  },
  render() {
    return this.isComplete ? (
      <Layout
        class={[this.ns.b(), this.ns.is('collapse', this.collapseChange)]}
      >
        <sider
          hide-trigger
          class={[this.ns.b('nav')]}
          width={this.collapseChange ? 80 : 256}
          value={this.collapseChange}
        >
          {renderLogo(this.ns, this.model)}
          {this.$slots.menu}
        </sider>
        <layout class={[this.ns.b('content')]}>
          <i-header class={this.ns.b('header')}>
            <div class={this.ns.be('header', 'left')}>
              {renderLogo(this.ns, this.model)}
              <div class={this.ns.be('header', 'icon')}>
                {!this.collapseChange ? (
                  <Icon
                    type='ios-arrow-back'
                    on-click={() => this.collapseMenus()}
                  />
                ) : (
                  <Icon
                    type='ios-arrow-forward'
                    on-click={() => this.collapseMenus()}
                  />
                )}
              </div>
              <div
                title='后退'
                class={this.ns.be('header', 'back-icon')}
                onClick={this.onBackClick}
              >
                <Icon type='ios-arrow-dropleft' />
              </div>

              {this.model.source.mainMenuAlign === 'TOP' ? (
                <i-menu
                  class={this.ns.be('header', 'menu')}
                  mode='horizontal'
                  active-name='1'
                ></i-menu>
              ) : null}
            </div>
            <div class={this.ns.be('header', 'right')}>
              <app-user />
            </div>
          </i-header>
          <i-content class={this.ns.be('content', 'main')}>
            {this.$slots.default}
          </i-content>
          {/* <i-footer class={this.ns.b('footer')}>Footer</i-footer> */}
        </layout>
      </Layout>
    ) : null;
  },
});
