import { PickerEditorController } from '@ibiz-template/controller';
import { ref, watch, Ref, defineComponent, PropType } from 'vue';
import { useNamespace } from '@ibiz-template/vue-util';
import '@/styles/components/editor/ibiz-picker/ibiz-picker.scss';
import { isEmpty } from 'lodash-es';

export const IBizPicker = defineComponent({
  name: 'IBizPicker',
  props: {
    value: {
      type: String,
    },
    controller: {
      type: PickerEditorController,
    },
    data: {
      type: Object as PropType<IData>,
      required: true,
    },
  },
  setup(props, { emit }) {
    const ns = useNamespace('picker');

    const c = props.controller!;

    // 当前值
    const curValue: Ref<Array<string> | string | null> = ref('');

    // 实体数据集
    const items: Ref<IData[]> = ref([]);

    // 是否显示所有数据
    const isShowAll = ref(true);

    watch(
      () => props.value,
      newVal => {
        if (newVal || newVal === null) {
          curValue.value = newVal;
          if (newVal === null) {
            curValue.value = '';
          }
          const value = props.data[c.valueItem];
          const index = items.value.findIndex((item: IData) =>
            Object.is(item[c.keyName], value),
          );
          if (index !== -1) {
            return;
          }
          // items里匹配不到当前值项值时，生成自身的选项
          items.value = [];
          if (!isEmpty(newVal) && !isEmpty(value)) {
            items.value.push({ [c.textName]: newVal, [c.keyName]: value });
          }
        }
      },
      { immediate: true },
    );

    // 处理视图关闭，往外抛值
    const handleOpenViewClose = (result: IData[]) => {
      const item: IData = {};
      if (result && Array.isArray(result)) {
        Object.assign(item, result[0]);
      }
      if (c.valueItem) {
        emit('change', item[c.keyName], c.valueItem);
      }
      emit('change', item[c.textName]);
    };

    // 打开数据选择视图
    const openPickUpView = async () => {
      const res = await c.openPickUpView(props.data);
      if (res) {
        handleOpenViewClose(res);
      }
    };

    // 打开数据链接视图
    const openLinkView = async () => {
      const res = await c.openLinkView(props.data);
      if (res) {
        handleOpenViewClose(res);
      }
    };

    // 往外抛值
    const onACSelect = (item: IData) => {
      console.log('onACSelect');
      if (c.valueItem) {
        emit('change', item[c.keyName], c.valueItem);
      }
      emit('change', item[c.textName]);
      isShowAll.value = true;
    };

    // 搜索
    const onSearch = async (query: string) => {
      console.log('picker-auto-query', query);
      if (c.model.appDataEntity) {
        const res = await c.getServiceData(query, props.data);
        if (res) {
          items.value = res.data as IData[];
        }
      }
    };

    // 清除
    const onClear = () => {
      if (c.valueItem) {
        emit('change', null, c.valueItem);
      }
      emit('change', null);
    };

    // 聚焦
    const onFocus = (e: IData) => {
      const query = isShowAll.value ? '' : e.target.value;
      onSearch(query);
    };
    return {
      ns,
      c,
      curValue,
      items,
      openPickUpView,
      openLinkView,
      onACSelect,
      onSearch,
      onClear,
      onFocus,
    };
  },
  render() {
    return (
      <div class={this.ns.b()}>
        {this.c.noAC ? (
          <i-input
            value={this.curValue}
            clearable
            placeholder={this.c.placeHolder}
            on-on-clear={this.onClear}
          >
            {this.$slots.append}
            {!this.$slots.append && this.c.pickupView ? (
              <i-button
                key='ios-search'
                icon='ios-search'
                on-click={this.openPickUpView}
              ></i-button>
            ) : null}
            {!this.$slots.append && this.c.linkView ? (
              <i-button
                key='ios-search-outline'
                icon='ios-search-outline'
                on-click='openLinkView'
              ></i-button>
            ) : null}
          </i-input>
        ) : (
          <div class={this.ns.e('autocomplete')}>
            <auto-complete
              value={this.curValue}
              placeholder={this.c.placeHolder}
              placement='bottom'
              clearable
              transfer-class-name={this.ns.e('transfer')}
              on-on-focus={this.onFocus}
              on-on-search={this.onSearch}
              on-on-clear={this.onClear}
            >
              {this.items.map(item => {
                return (
                  <div
                    class={this.ns.e('transfer-item')}
                    on-click={() => {
                      this.onACSelect(item);
                    }}
                  >
                    {item[this.c.textName]}
                  </div>
                );
              })}
            </auto-complete>
            <div class={this.ns.e('btns')}>
              {this.c.pickupView ? (
                <i-button
                  icon='ios-search'
                  on-click={this.openPickUpView}
                ></i-button>
              ) : null}
              {this.c.linkView ? (
                <i-button
                  icon='ios-search-outline'
                  on-click={this.openLinkView}
                ></i-button>
              ) : null}
            </div>
          </div>
        )}
      </div>
    );
  },
});
export default IBizPicker;
