import { CoreConst } from '@ibiz-template/core';
import {
  defineComponent,
  getCurrentInstance,
  onMounted,
  reactive,
  ref,
} from 'vue';
import { setCookie } from 'qx-util';
import { useNamespace, useRoute } from '@ibiz-template/vue-util';
import router from '@/router';
import '@/styles/components/views/login-view/login-view.scss';

interface LoginData {
  username: string;
  password: string;
}

const rules = {
  username: [
    {
      required: true,
      message: '请输入账号',
      trigger: 'blur',
    },
  ],
  password: [
    {
      required: true,
      message: '请输入密码',
      trigger: 'blur',
    },
    {
      type: 'string',
      min: 6,
      message: '密码长度不能少于6位',
      trigger: 'blur',
    },
  ],
};

export default defineComponent({
  setup() {
    const ns = useNamespace('login-view');

    const loginData = reactive<LoginData>({
      username: 'sy_admin',
      password: 'syadmin@2022',
    });

    const formRef = ref<IData | null>(null);

    const instance = getCurrentInstance()!;
    const route = useRoute(instance.proxy);
    const ru = (route.query.ru as string) || '/';

    onMounted(() => {
      setTimeout(() => {
        const el = document.querySelector('.app-loading-x') as HTMLDivElement;
        if (el) {
          el.style.display = 'none';
        }
      }, 300);
    });

    const onClick = () => {
      formRef.value!.validate(async (valid: boolean) => {
        if (valid) {
          try {
            const res = await ibiz.auth.v7login(
              loginData.username,
              loginData.password,
            );
            if (res.ok) {
              const { data } = res;
              if (data && data.token) {
                setCookie(CoreConst.TOKEN, data.token, 7, true);
                router.push({ path: ru });
                return;
              }
            }
            ibiz.notification.error({
              title: res.data?.message || '登录失败',
            });
          } catch (error) {
            ibiz.notification.error({
              title: (error as IData).response?.data?.message || '登录失败',
            });
          }
        }
      });
    };

    return () => (
      <div class={ns.b()}>
        <div class={ns.b('box')}>
          <header class={ns.b('box-header')}>
            <img src='/img/login-header.png' />
          </header>
          <main class={ns.b('box-main')}>
            <img
              class={ns.be('box-main', 'avatar')}
              src='/img/login-avatar.png'
            />
            <div class={ns.b('box-main-content')}>
              <i-form ref={formRef} props={{ model: loginData, rules }}>
                <form-item prop='username'>
                  <i-input
                    type='text'
                    value={loginData.username}
                    on-on-change={(evt: InputEvent) => {
                      const { value } = evt.target as HTMLInputElement;
                      loginData.username = value;
                    }}
                    placeholder='请输入账号'
                    size='large'
                  >
                    <icon type='md-person' slot='prefix'></icon>
                  </i-input>
                </form-item>
                <form-item prop='password'>
                  <i-input
                    type='password'
                    value={loginData.password}
                    on-on-change={(evt: InputEvent) => {
                      const { value } = evt.target as HTMLInputElement;
                      loginData.password = value;
                    }}
                    placeholder='请输入密码'
                    size='large'
                    password
                  >
                    <icon type='ios-unlock' slot='prefix'></icon>
                  </i-input>
                </form-item>
                <form-item>
                  <i-button shape='circle' long on-click={onClick}>
                    登录
                  </i-button>
                </form-item>
              </i-form>
            </div>
          </main>
        </div>
      </div>
    );
  },
});
