import { computed, defineComponent, onMounted, ref } from 'vue';
import {
  IPSDETBGroupItem,
  IPSDETBUIActionItem,
  IPSDEToolbarItem,
  ToolbarModel,
} from '@ibiz-template/model';
import { ToolbarNeuron } from '@ibiz-template/controller';
import { useNamespace } from '@ibiz-template/vue-util';
import '@/styles/components/common/view-toolbar/view-toolbar.scss';

const btnContent = (item: IPSDEToolbarItem, viewMode: string) => {
  const image = item.getPSSysImage();
  if (viewMode === 'EMBED') {
    if (image) {
      return [<app-icon icon={image} />, item.caption];
    }
    return [<img src='undefined' />, item.caption];
  }
  return [<app-icon icon={image} />, item.caption];
};

export const ViewToolbar = defineComponent({
  name: 'ViewToolbar',
  props: {
    modelData: {
      type: ToolbarModel,
      required: true,
    },
    viewMode: {
      type: String,
      required: true,
    },
  },
  setup(props, { emit }) {
    const ns = useNamespace('view-toolbar');
    const neuron = new ToolbarNeuron({});
    emit('neuronInit', neuron);
    // 正在执行的工具栏项
    const doingToolbarItem = ref<string>('');

    onMounted(async () => {
      await neuron.evt.asyncEmit('mounted');
    });

    const handleClick = async (item: IPSDEToolbarItem, _event: MouseEvent) => {
      doingToolbarItem.value = item.id;
      try {
        await neuron.evt.asyncEmit(
          'itemClick',
          item as IPSDETBUIActionItem,
          _event,
        );
      } finally {
        doingToolbarItem.value = '';
      }
    };

    const btnSize = computed(() => {
      return props.viewMode === 'EMBED' ? 'small' : 'default';
    });

    return { ns, doingToolbarItem, handleClick, btnSize };
  },
  render() {
    return (
      <div class={[this.ns.b(), this.ns.m(this.viewMode.toLowerCase())]}>
        {this.modelData!.source.getPSDEToolbarItems()?.map(item => {
          if (item.itemType === 'SEPERATOR') {
            return (
              <div key={item.id} class={this.ns.b('item')}>
                |
              </div>
            );
          }
          if (item.itemType === 'RAWITEM') {
            return (
              <div key={item.id} class={this.ns.b('item')}>
                {btnContent(item, this.viewMode)}
              </div>
            );
          }
          if (item.itemType === 'DEUIACTION') {
            return (
              <div key={item.id} class={this.ns.b('item')}>
                <i-button
                  title={item.tooltip}
                  size={this.btnSize}
                  loading={
                    !!this.doingToolbarItem && this.doingToolbarItem === item.id
                  }
                  disabled={!!this.doingToolbarItem}
                  on-click={(e: MouseEvent) => this.handleClick(item, e)}
                >
                  {btnContent(item, this.viewMode)}
                </i-button>
              </div>
            );
          }
          if (item.itemType === 'ITEMS') {
            return (
              <div key={item.id} class={this.ns.b('item')}>
                <dropdown trigger='click' placement='bottom-start'>
                  <i-button title={item.tooltip} size={this.btnSize}>
                    {btnContent(item, this.viewMode)}
                  </i-button>
                  <dropdown-menu slot='list'>
                    {(item as IPSDETBGroupItem)
                      .getPSDEToolbarItems()!
                      .map(item2 => {
                        return (
                          <dropdown-item
                            key={item2.id}
                            nativeOn-click={(e: MouseEvent) =>
                              this.handleClick(item2, e)
                            }
                          >
                            {btnContent(item2, this.viewMode)}
                          </dropdown-item>
                        );
                      })}
                  </dropdown-menu>
                </dropdown>
              </div>
            );
          }
          return null;
        })}
      </div>
    );
  },
});
