import { CreateElement } from 'vue';
import { Emit, Prop, Watch } from 'vue-property-decorator';
import { throttle, Util } from 'ibiz-core';
import { DataViewControlBase } from '../../../widgets';

/**
 * 数据视图部件基类
 *
 * @export
 * @class AppDataViewBase
 * @extends {DataViewControlBase}
 */
export class AppDataViewBase extends DataViewControlBase {

    /**
     * 部件动态参数
     *
     * @memberof AppDataViewBase
     */
    @Prop() public declare dynamicProps: any;

    /**
     * 部件静态参数
     *
     * @memberof AppDataViewBase
     */
    @Prop() public declare staticProps: any;

    /**
     * 监听动态参数变化
     *
     * @param {*} newVal
     * @param {*} oldVal
     * @memberof AppDataViewBase
     */
    @Watch('dynamicProps', {
        immediate: true,
    })
    public onDynamicPropsChange(newVal: any, oldVal: any) {
        if (newVal && !Util.isFieldsSame(newVal, oldVal)) {
            super.onDynamicPropsChange(newVal, oldVal);
        }
    }

    /**
     * 监听静态参数变化
     *
     * @param {*} newVal
     * @param {*} oldVal
     * @memberof AppDataViewBase
     */
    @Watch('staticProps', {
        immediate: true,
    })
    public onStaticPropsChange(newVal: any, oldVal: any) {
        if (newVal && !Util.isFieldsSame(newVal, oldVal)) {
            super.onStaticPropsChange(newVal, oldVal);
        }
    }

    /**
     * 部件事件
     * 
     * @param 抛出参数 
     * @memberof AppDataViewBase
     */
    @Emit('ctrl-event')
    public ctrlEvent({ controlname, action, data }: { controlname: string, action: string, data: any }): void { }



    /**
     * 绘制分组
     *
     * @param {*} h
     * @return {*} 
     * @memberof AppDataViewBase
     */
    public renderGroup(h: any) {
        return this.groupData.map((group: any, index: number) => {
            return (
                <el-collapse-item>
                    <template slot='title'>
                        <div style='margin: 0 0 0 12px;'>
                            <b>{group.group}</b>
                        </div>
                    </template>
                    {this.hasChildrenRender(h, group)}
                </el-collapse-item>
            );
        });
    }

    /**
     * 绘制有子成员项
     *
     * @param {*} h
     * @param {*} group
     * @return {*} 
     * @memberof AppDataViewBase
     */
    public hasChildrenRender(h: any, group: any) {
        if (group.children.length > 0) {
            const { cardColXS, cardColSM, cardColMD, cardColLG } = this.controlInstance;
            return group.children.map((groupChild: any, index: number) => {
                return (
                    <a href={groupChild.starturl}>
                        <i-col style='min-height: 170px;margin-bottom: 10px;' xs={cardColXS} sm={cardColSM} md={cardColMD} lg={cardColLG}>
                            {this.renderCard(groupChild)}
                        </i-col>
                    </a>
                );
            });
        } else {
            return (
                <div v-else class='item-nodata'>
                    {this.$t('app.commonwords.nodata')}
                </div>
            );
        }
    }

    /**
     * 渲染数据视图项行为
     *
     * @param {*} item
     * @return {*} 
     * @memberof AppDataViewBase
     */
    public renderDataViewItemAction(item: any) {
        return Object.keys(this.actionModel).map((key: string) => {
            if (item[key].visabled) {
                const action = this.actionModel[key];
                return (
                    <el-button type='text' disabled={item[key].disabled}
                        on-click={($event: any) => {
                            this.handleActionClick(item, $event, action, action);
                        }}>
                        {action.showIcon ? <i class={action.cssClass} style='margin-right:2px;'></i> : ''}
                        <span>
                            {action.showCaption ? this.$tl(action.lanResTag, action?.caption ? action.caption : '') : ''}
                        </span>
                    </el-button>
                )
            }
        });
    }

    /**
     * 渲染数据视图项
     *
     * @param {*} item
     * @return {*} 
     * @memberof AppDataViewBase
     */
    public renderDataViewItem(item: any) {
        return (
            <div class='data-view-item'>
                <div class='single-card-default'>
                    <div class='item-title'>
                        <span>{item.srfmajortext}</span>
                    </div>
                    {item.content && <div class='item-content'>{item.content}</div>}
                </div>
                {this.actionModel && Object.keys(this.actionModel).length ? <div class="data-view-item-action">
                    {this.renderDataViewItemAction(item)}
                </div> : null}
            </div>
        )
    }

    /**
     * 渲染卡片
     *
     * @param {*} item
     * @return {*} 
     * @memberof AppDataViewBase
     */
    public renderCard(item: any) {
        const style: any = {
            width: this.controlInstance.cardWidth > 0 ? `${this.controlInstance.cardWidth}px` : false,
            height: this.controlInstance.cardHeight > 0 ? `${this.controlInstance.cardHeight}px` : false,
        };
        return (
            <el-card
                shadow='always'
                body-style={style}
                class={[item.srfchecked === 1 ? 'isselected' : false, 'single-card-data']}
                nativeOnClick={() => throttle(this.handleClick, [item], this)}
                nativeOnDblclick={() => throttle(this.handleDblClick, [item], this)}
            >
                {this.controlInstance.getItemPSLayoutPanel()
                    ? this.renderItemPSLayoutPanel(item)
                    : this.renderDataViewItem(item)}
            </el-card>
        );
    }

    /**
     * 绘制数据视图内容
     *
     * @memberof AppDataViewBase
     */
    public renderDataViewContent(h: CreateElement) {
        if (!this.isEnableGroup) {
            const { cardColXS, cardColSM, cardColMD, cardColLG } = this.controlInstance;
            return this.items.map((item: any, index: number) => {
                return (
                    <i-col style='margin-bottom: 10px' xs={cardColXS} sm={cardColSM} md={cardColMD} lg={cardColLG}> 
                        {this.renderCard(item)}
                    </i-col>
                );
            });
        } else {
            return <el-collapse>{this.renderGroup(h)}</el-collapse>;
        }
    }

    /**
     * @description 绘制加载更多
     * @return {*} 
     * @memberof AppDataViewBase
     */
     renderLoadMore(){
        return (
            <div onclick={(event: MouseEvent)=>{this.loadMore(event)}} class='load-more'>{this.$t('app.dataview.loadmore')}</div>
        )
    }

    /**
     * 绘制
     * 
     * @memberof AppDataViewBase
     */
    public render(h: CreateElement) {
        if (!this.controlIsLoaded) {
            return null;
        }
        const { controlClassNames } = this.renderOptions;
        return (
            <div class={{ ...controlClassNames, "app-data-view": true }} style="height: 100%;">
                {this.renderSortBar(h)}
                <row class="data-view-container" gutter={20} type="flex" justify="start" style={`margin: 0px; ${this.items.length ? '' : 'display: none;'}`}>
                    {this.renderDataViewContent(h)}
                    {Object.is(this.totalRecord,this.items.length) ? null : this.renderLoadMore()}
                </row>
                {this.selections.length > 0 ? this.renderBatchToolbar() : null}
                {!this.ctrlLoadingService?.isLoading ? this.renderEmptyDataTip() : this.renderLoadDataTip()}
                <el-backtop target={`#${this.controlId}`}></el-backtop>
            </div>
        );
    }

    /**
     * 销毁视图回调
     *
     * @memberof AppDataViewBase
     */
    public destroyed() {
        this.ctrlDestroyed();
    }
}