import { computed, defineComponent, ref, watch } from 'vue';
import { debounce, isNaN } from 'lodash-es';
import {
  getEditorEmits,
  getInputProps,
  useNamespace,
} from '@ibiz-template/vue-util';
import '@ibiz-template/theme/style/components/editor/ibiz-input/ibiz-input.scss';

const fomatFloat = function (value: number, n: number) {
  const f = Math.round(value * 10 ** n) / 10 ** n;
  let s = f.toString();
  const rs = s.indexOf('.');
  if (rs < 0) {
    s += '.';
  }
  for (let i = s.length - s.indexOf('.'); i <= n; i++) {
    s += '0';
  }
  return s;
};

export const IBizInput = defineComponent({
  name: 'IBizInput',
  props: getInputProps(),
  emits: getEditorEmits(),
  setup(props, { emit }) {
    const ns = useNamespace('input');
    const c = props.controller;
    const editorModel = c.model;
    const inputRef = ref();
    let autoSize: boolean | { minRows: Number; maxRows?: Number } = {
      minRows: 2,
    };

    // 文本域默认行数，仅在 textarea 类型下有效
    const rows = ref(2);
    if (editorModel.editorType === 'TEXTAREA_10') {
      rows.value = 10;
      autoSize = {
        minRows: 10,
      };
    }

    // 类型
    const type = computed(() => {
      if (c.model.precision !== -1) {
        return 'number';
      }
      switch (editorModel.editorType) {
        case 'TEXTBOX':
          return 'text';
        case 'PASSWORD':
          return 'password';
        case 'TEXTAREA':
        case 'TEXTAREA_10':
          return 'textarea';
        default:
          return 'string';
      }
    });

    if (editorModel.editorParams) {
      if (editorModel.editorParams.autosize) {
        try {
          autoSize = JSON.parse(editorModel.editorParams.autosize);
        } catch {
          autoSize = false;
        }
      }
    }
    const currentVal = ref<string>('');

    watch(
      () => props.value,
      (newVal, oldVal) => {
        if (newVal !== oldVal) {
          if (!newVal) {
            currentVal.value = '';
          } else {
            currentVal.value = newVal;
          }
        }
      },
      { immediate: true },
    );

    let isDebounce = false;
    let awaitSearch: () => void;
    let blurCacheValue: string | undefined;
    // 防抖值变更回调
    const debounceChange = debounce(
      (e: IData) => {
        // 拦截掉blur触发后change
        if (blurCacheValue !== e.target.value) {
          if (c.model.precision !== -1) {
            const num = Number(e.target.value);
            if (!isNaN(num)) {
              const number = fomatFloat(num, c.model.precision);
              currentVal.value = e.target.value; // 1.001
              setTimeout(() => {
                currentVal.value = number;
                // 1.00 1.001
                if (number !== props.value) {
                  emit('change', number);
                }
              });
            }
          } else {
            emit('change', e.target.value);
          }
        }
        blurCacheValue = undefined;
        isDebounce = false;
        if (awaitSearch) {
          awaitSearch();
        }
      },
      300,
      { leading: true },
    );
    // 值变更
    const handleChange = (e: IData) => {
      isDebounce = true;
      debounceChange(e);
    };

    const handleKeyUp = (e: KeyboardEvent) => {
      if (e && e.code === 'Enter') {
        if (isDebounce) {
          awaitSearch = () => {
            inputRef.value.$el.dispatchEvent(e);
          };
        }
      }
    };

    /**
     * blur时马上抛值变更
     * @author lxm
     * @date 2023-03-06 06:36:23
     */
    const handleBlur = () => {
      blurCacheValue = inputRef.value.currentValue;
      const number = fomatFloat(Number(blurCacheValue), c.model.precision);
      if (c.model.precision !== -1) {
        emit('change', number);
      } else {
        emit('change', blurCacheValue);
      }
    };

    // 自动聚焦
    if (props.autoFocus) {
      watch(inputRef, newVal => {
        if (newVal) {
          let input = newVal.$el.getElementsByTagName('input')[0];
          if (!input) {
            input = newVal.$el.getElementsByTagName('textarea')[0];
          }
          input.focus();
        }
      });
    }

    return {
      ns,
      rows,
      type,
      currentVal,
      handleChange,
      handleKeyUp,
      handleBlur,
      inputRef,
      autoSize,
    };
  },
  render() {
    return (
      <div
        class={[
          this.ns.b(),
          this.ns.is('textarea', Object.is(this.type, 'textarea')),
          this.disabled ? this.ns.m('disabled') : '',
          this.readonly ? this.ns.m('readonly') : '',
        ]}
      >
        {this.readonly ? (
          this.currentVal
        ) : (
          <i-input
            ref='inputRef'
            value={this.currentVal}
            placeholder={this.controller.placeHolder}
            type={this.type}
            rows={this.rows}
            on-on-change={this.handleChange}
            on-on-blur={this.handleBlur}
            nativeOnkeyup={this.handleKeyUp}
            class={this.ns.b('input')}
            disabled={this.disabled}
            autosize={this.autoSize}
          >
            {this.controller.model.unitName && (
              <i class={this.ns.e('unit')} slot='suffix'>
                {this.controller.model.unitName}
              </i>
            )}
          </i-input>
        )}
      </div>
    );
  },
});
