import { IBizContext } from '@ibiz-template/core';
import { IPSAppView } from '@ibiz-template/model';
import { IModalData, IOpenViewUtil } from '@ibiz-template/runtime';
import { generateRoutePath } from '@ibiz-template/vue-util';
import router from '@/router';
import { AppModal } from './app-modal/app-modal';
import { AppDrawer } from './app-drawer/app-drawer';

/**
 * 打开视图方式工具类
 *
 * @description 此实现类挂载在 ibiz.openViewUtil
 * @author chitanda
 * @date 2022-08-16 20:08:54
 * @export
 * @class OpenViewUtil
 * @implements {IOpenViewUtil}
 */
export class OpenViewUtil implements IOpenViewUtil {
  root(
    appView: IPSAppView,
    context?: IBizContext | undefined,
    _params?: IParams | undefined,
  ): void {
    const path = generateRoutePath(
      appView,
      router.currentRoute,
      context,
      _params,
    );

    router.push({ path });
  }

  /**
   * 模态打开视图
   *
   * @author lxm
   * @date 2022-09-12 01:09:06
   * @param {IPSAppView} appView
   * @param {(IBizContext | undefined)} [context]
   * @param {(IParams | undefined)} [params]
   * @returns {*}  {Promise<IModalData>}
   */
  async modal(
    appView: IPSAppView,
    context?: IBizContext | undefined,
    params?: IParams | undefined,
  ): Promise<IModalData> {
    // 获取视图组件名和path
    const modelPath = appView.modelPath!;

    // 设置默认的modal参数
    const opts = {
      width: appView.width || '80%',
      height: appView.height || '80%',
      footerHide: true,
    };

    const res = await AppModal.openModal(
      'ViewShell',
      {
        context,
        params,
        modelPath,
      },
      opts,
    );
    return res;
  }

  async popover(
    appView: IPSAppView,
    context?: IBizContext | undefined,
    params?: IParams | undefined,
  ): Promise<IModalData> {
    ibiz.log.warn('openPopover', appView, context, params);
    throw new Error();
  }

  /**
   * 抽屉打开视图
   *
   * @author lxm
   * @date 2022-09-15 15:09:50
   * @param {IPSAppView} appView
   * @param {(IBizContext | undefined)} [context]
   * @param {(IParams | undefined)} [params]
   * @returns {*}  {Promise<IModalData>}
   */
  async drawer(
    appView: IPSAppView,
    context?: IBizContext | undefined,
    params?: IParams | undefined,
  ): Promise<IModalData> {
    // 获取视图组件名和path
    const modelPath = appView.modelPath!;
    const placement = AppDrawer.getPlacement(appView.openMode);

    // 设置默认的modal参数
    const opts = {
      width: appView.width || '800',
      height: appView.height || '600',
      placement,
    };

    const res = await AppDrawer.openDrawer(
      'ViewShell',
      {
        context,
        params,
        modelPath,
      },
      opts,
    );
    return res;
  }

  async custom(
    appView: IPSAppView,
    context?: IBizContext | undefined,
    params?: IParams | undefined,
  ): Promise<IModalData> {
    ibiz.log.warn('openUserCustom', appView, context, params);
    throw new Error();
  }
}
