import { ref, Ref, watch, defineComponent } from 'vue';
import {
  getDataPickerProps,
  getEditorEmits,
  useNamespace,
} from '@ibiz-template/vue-util';
import { isNil } from 'ramda';
import '@ibiz-template/theme/style/components/editor/ibiz-picker-dropdown/ibiz-picker-dropdown.scss';

export const IBizPickerDropDown = defineComponent({
  name: 'IBizPickerDropDown',
  props: getDataPickerProps(),
  emits: getEditorEmits(),
  setup(props, { emit }) {
    const ns = useNamespace('picker-dropdown');

    const c = props.controller;

    // 当前值
    const curValue: Ref<Array<string> | string | null> = ref('');

    // 实体数据集
    const items: Ref<IData[]> = ref([]);

    watch(
      () => props.value,
      newVal => {
        if (newVal || newVal === null) {
          curValue.value = newVal;
          const value = props.data[c.valueItem];
          const index = items.value.findIndex((item: IData) =>
            Object.is(item[c.keyName], value),
          );
          if (index !== -1) {
            return;
          }
          // items里匹配不到当前值项值时，生成自身的选项
          items.value = [];
          if (!isNil(newVal) && !isNil(value)) {
            items.value.push({ [c.textName]: newVal, [c.keyName]: value });
          }
        }
      },
      { immediate: true },
    );

    // 获取关联数据项值
    const refValue = ref('');

    watch(
      () => props.data[c.valueItem],
      (newVal, oldVal) => {
        if (newVal !== oldVal) {
          refValue.value = newVal;
        }
      },
      { immediate: true, deep: true },
    );

    // 加载中
    const loading: Ref<boolean> = ref(false);

    // 往外抛值
    const onACSelect = async (item: IData) => {
      // 处理回填数据
      const dataItems = await c.calcFillDataItems(item);
      if (dataItems.length) {
        dataItems.forEach(dataItem => {
          emit('change', dataItem.value, dataItem.name);
        });
      }

      // 处理值项和本身的值
      if (c.valueItem) {
        emit('change', item[c.keyName], c.valueItem);
      }
      emit('change', item[c.textName]);
    };

    // 值变更
    const onSelect = (select: string | Array<string>) => {
      const index = items.value.findIndex(item =>
        Object.is(item[c.keyName], select),
      );
      if (index >= 0) {
        onACSelect(items.value[index]);
      }
    };

    // 在搜索中时，再次触发搜索记录搜索值，等待上次搜索触发完成后再次搜索
    let waitQuery: string | null = null;

    // 搜索
    const onSearch = async (query: string) => {
      if (c.model.appDataEntity && loading.value === false) {
        loading.value = true;
        try {
          const res = await c.getServiceData(query, props.data!);
          if (res) {
            items.value = res.data as IData[];
          }
        } finally {
          loading.value = false;
          if (waitQuery != null) {
            const selfQuery = waitQuery;
            waitQuery = null;
            await onSearch(selfQuery);
          }
        }
      } else {
        waitQuery = query;
      }
    };

    // 下拉打开
    const onOpenChange = (isOpen: boolean) => {
      emit('operate', isOpen);
      if (isOpen) {
        items.value = [];
        onSearch('');
      }
    };

    // 清除
    const onClear = () => {
      // 清空回填数据
      const dataItems = c.model.deACMode?.dataItems;
      if (dataItems?.length) {
        dataItems.forEach(dataItem => {
          emit('change', null, dataItem.name);
        });
      }
      if (c.valueItem) {
        emit('change', null, c.valueItem);
      }
      emit('change', null);
    };
    return {
      ns,
      c,
      refValue,
      loading,
      items,
      onOpenChange,
      onClear,
      onSelect,
      onSearch,
    };
  },
  render() {
    if (this.readonly) {
      return (
        <div class={(this.ns.b(), this.ns.m('readonly'))}>{this.value}</div>
      );
    }
    return (
      <div class={[this.ns.b(), this.disabled ? this.ns.m('disabled') : '']}>
        {this.readonly ? (
          this.value
        ) : (
          <i-select
            value={this.refValue}
            filterable
            remote
            allow-clear
            clearable
            transfer
            loading={this.loading}
            placeholder={this.c.placeHolder}
            remote-method={this.onSearch}
            on-on-open-change={this.onOpenChange}
            on-on-change={this.onSelect}
            on-on-clear={this.onClear}
            disabled={this.disabled}
          >
            {this.items.map((item, index) => {
              return (
                <i-option value={item[this.c.keyName]} key={index}>
                  {item[this.c.textName]}
                </i-option>
              );
            })}
          </i-select>
        )}
      </div>
    );
  },
});
