import {
  defineComponent,
  getCurrentInstance,
  onMounted,
  PropType,
  reactive,
  ref,
} from 'vue';
import { useNamespace } from '@ibiz-template/vue-util';
import { ViewNeuron } from '@ibiz-template/controller';
import { isNumber } from 'lodash-es';
import { IModal, IModalData, ViewMode } from '@ibiz-template/runtime';
import { AppDrawerOptions } from '@/interface';
import { useUIStore } from '@/store';
import '@ibiz-template/theme/style/components/util/drawer/drawer.scss';

export const AppDrawerComponent = defineComponent({
  props: {
    componentName: String,
    componentProps: Object as PropType<IData>,
    opts: {
      type: Object as PropType<AppDrawerOptions>,
      default: () => ({}),
    },
    resolve: { type: Function, required: true },
    reject: { type: Function, required: true },
  },
  setup(props) {
    const ns = useNamespace('drawer');

    const isShow = ref(false);
    const { proxy } = getCurrentInstance()!;

    const { zIndex } = useUIStore();
    const drawerZIndex = zIndex.increment();

    // 处理自定义样式
    const customStyle = reactive<IData>({});
    const { width, height } = props.opts;
    if (width) {
      customStyle.width = isNumber(width) ? `${width}px` : width;
    }
    if (height) {
      customStyle.height = isNumber(height) ? `${height}px` : height;
    }

    // 执行抽屉关闭操作
    const doDrawerClose = () => {
      setTimeout(() => {
        zIndex.decrement();
        proxy.$destroy();
        document.body.removeChild(proxy.$el);
      }, 300);
    };

    // 抽屉自身的操作触发的关闭(如点右上角的×)
    const onVisibleChange = (state: boolean) => {
      if (!state) {
        props.resolve({ ok: false, data: {} });
        doDrawerClose();
      }
    };

    // modal对象
    const modal: IModal = { mode: ViewMode.DRAWER };

    // 抽屉标题
    const drawerCaption = ref('');

    // 视图神经元接收
    const onNeuronInit = (neuron: ViewNeuron) => {
      neuron.evt.on('setTitle', title => {
        drawerCaption.value = title;
      });
      neuron.evt.on('closeView', (res?: IModalData) => {
        props.resolve(res);
        doDrawerClose();
      });
    };

    onMounted(() => {
      isShow.value = true;
    });

    return {
      ns,
      modal,
      isShow,
      drawerZIndex,
      customStyle,
      drawerCaption,
      onNeuronInit,
      onVisibleChange,
    };
  },
  render(h) {
    return (
      <i-drawer
        value={this.isShow}
        on-input={(val: boolean) => {
          this.isShow = val;
        }}
        width={this.opts.width}
        height={this.opts.height}
        class={[this.ns.b()]}
        style={{ [this.ns.cssVarBlockName('z-index')]: this.drawerZIndex }}
        styles={this.customStyle}
        placement={this.opts.placement || 'right'}
        on-on-visible-change={this.onVisibleChange}
      >
        {h(this.componentName, {
          props: {
            ...this.componentProps,
            modal: this.modal,
          },
          on: {
            neuronInit: this.onNeuronInit,
          },
        })}
      </i-drawer>
    );
  },
});
