import { ref, watch, defineComponent } from 'vue';
import { useNamespace } from '@ibiz-template/vue-util';
import { isNil } from 'ramda';
import './code-list.scss';

export const CodeList = defineComponent({
  props: {
    codeListItems: {
      type: Array<IData>,
    },
    codeList: {
      type: Object,
      required: true,
    },
    value: {
      type: [String, Number],
    },
  },
  setup(props) {
    const ns = useNamespace('code-list');
    const items = ref<IData[]>([]);

    watch(
      () => props.value,
      (newVal, oldVal) => {
        if (newVal !== oldVal) {
          if (isNil(newVal) || newVal === '') {
            // 空值置空
            items.value = [];
            // 非空值解析代码表
          } else if (
            props.codeList.orMode === 'NUM' &&
            props.codeListItems?.length
          ) {
            const numVal =
              typeof newVal === 'string' ? parseInt(newVal, 10) : newVal;
            // 数值或模式
            items.value = props
              .codeListItems!.filter(
                // eslint-disable-next-line no-bitwise
                item => numVal & item.value,
              )
              .map(item => ({ text: item.text, color: item.color }));
          } else {
            // 字符串或模式
            let values: string[] | number[];
            if (typeof newVal === 'string') {
              values = newVal.split(',');
            } else {
              values = [newVal];
            }
            items.value = values.map(val => {
              const findItem = props.codeListItems?.find(
                // eslint-disable-next-line eqeqeq
                item => item.value == val,
              );
              return {
                text: findItem?.text || val,
                color: findItem?.color,
                textCls: findItem?.textCls,
              };
            });
          }
        }
      },
      { immediate: true },
    );

    return { items, ns };
  },
  render() {
    if (this.items.length === 0) {
      return this.codeList.emptyText;
    }
    return (
      <span class={this.ns.b()}>
        {this.items.map((item, index) => {
          return [
            index !== 0 ? this.codeList?.textSeparator : null,
            <span
              class={[this.ns.e('item'), item.textCls ? item.textCls : null]}
              style={this.ns.cssVarBlock({
                'item-color': `${item.color}`,
              })}
            >
              {item.text}
            </span>,
          ];
        })}
      </span>
    );
  },
});
