import { IBizContext } from '@ibiz-template/core';
import { GridModel } from '@ibiz-template/model';
import { useGridController, useNamespace } from '@ibiz-template/vue-util';
// import { Table } from 'view-design';
import { defineComponent, getCurrentInstance, PropType } from 'vue';
import { AppGridPagination } from '@/components/common';
import '@/styles/components/widgets/grid/grid.scss';
import {
  useAppGridPagination,
  useITableColumns,
  useITableEvent,
} from './grid-control.util';

export const GridControl = defineComponent({
  props: {
    modelData: GridModel,
    context: IBizContext,
    params: { type: Object as PropType<IParams>, default: () => ({}) },
    /**
     * 表格行数据默认激活模式
     * - 0 不激活
     * - 1 单击激活
     * - 2 双击激活(默认值)
     *
     * @type {(number | 0 | 1 | 2)}
     */
    gridRowActiveMode: { type: Number, default: 2 },
  },
  setup(props) {
    const { proxy } = getCurrentInstance()!;

    const ns = useNamespace('grid');
    const c = useGridController(
      proxy,
      props.modelData!,
      props.context!,
      props.params,
    );

    const [columns] = useITableColumns(c);
    const { onRowClick, onDbRowClick, onSelectionChange, onSortChange } =
      useITableEvent(c);
    const { onPageChange, onPageReset, onPageSizeChange } =
      useAppGridPagination(c);

    // iView表格单击行选中没有，必须手动调用选中方法
    const onUIRowClick = (data: IData, index: number) => {
      const gridInstance: IData | undefined = proxy.$refs.grid;
      if (gridInstance) {
        if (gridInstance.toggleSelect) {
          gridInstance.toggleSelect(index);
        }
        if (gridInstance.highlightCurrentRow) {
          gridInstance.highlightCurrentRow(index);
        }
      }
      onRowClick(data);
    };

    return {
      c,
      ns,
      columns,
      onDbRowClick,
      onUIRowClick,
      onSelectionChange,
      onSortChange,
      onPageChange,
      onPageSizeChange,
      onPageReset,
    };
  },
  render() {
    if (!this.c.complete) {
      return;
    }

    // 绘制作用域插槽,绘制行编辑开启开关
    const columnSlots: IData = {
      rowEdit: ({ index }: IData) => {
        if (this.c.rows[index]) {
          return (
            <i-button
              icon='md-settings'
              on-click={async (event: MouseEvent) => {
                event.stopPropagation();
                const rowController = this.c.rows[index];
                await this.c.toggleRowEdit(rowController);
              }}
              nativeOn-dblclick={(e: MouseEvent) => {
                e.stopPropagation();
              }}
            ></i-button>
          );
        }
      },
    };
    // 属性列自定义
    this.c.model.fieldColumns.forEach(item => {
      columnSlots[item.codeName] = ({ row, index }: IData) => {
        if (this.c.rows[index]) {
          return (
            <grid-column
              key={row.srfkey + item.codeName}
              field-column={this.c.fieldColumns[item.codeName]}
              edit-item={this.c.editItems[item.codeName]}
              row={this.c.rows[index]}
            ></grid-column>
          );
        }
      };
    });
    // 操作列自定义
    this.c.model.uaColumns.forEach(item => {
      columnSlots[item.codeName] = ({ row, index }: IData) => {
        if (this.c.rows[index]) {
          return (
            <grid-ua-column
              key={row.srfkey + item.codeName}
              ua-column={this.c.uaColumns[item.codeName]}
              row={this.c.rows[index]}
            ></grid-ua-column>
          );
        }
      };
    });

    return (
      <control-layout modelData={this.c.model}>
        <div
          class={[
            this.ns.b(),
            this.ns.is('show-header', !this.c.model.source.hideHeader),
            this.ns.is('enable-page', this.c.model.source.enablePagingBar),
          ]}
        >
          <i-table
            ref={'grid'}
            class={this.ns.b('content')}
            show-header={!this.c.model.source.hideHeader}
            highlight-row
            data={this.c.items}
            columns={this.columns}
            on-on-row-click={this.onUIRowClick}
            on-on-row-dblclick={this.onDbRowClick}
            on-on-selection-change={this.onSelectionChange}
            on-on-sort-change={this.onSortChange}
            scopedSlots={columnSlots}
          ></i-table>
          {this.c.model.source.enablePagingBar && (
            <AppGridPagination
              total={this.c.total}
              curPage={this.c.curPage}
              size={this.c.size}
              on-change={this.onPageChange}
              on-page-size-change={this.onPageSizeChange}
              on-page-reset={this.onPageReset}
            ></AppGridPagination>
          )}
        </div>
      </control-layout>
    );
  },
});
