import Vue from 'vue';
import { RuntimeError } from '@ibiz-template/core';
import { DefectModelError } from '@ibiz-template/model';
import { IPluginFactory } from '@ibiz-template/runtime';
import { IPSAppPFPluginRef, IPSSysPFPlugin } from '@ibiz/dynamic-model-api';
import {
  RemotePluginConfig,
  RemotePluginItem,
} from '../remote-plugin-item/remote-plugin-item';

/**
 * 插件工具类
 *
 * @author chitanda
 * @date 2022-10-21 16:10:29
 * @export
 * @class PluginFactory
 */
export class PluginFactory implements IPluginFactory {
  /**
   * 是否已经加载过文件缓存
   *
   * @author chitanda
   * @date 2022-10-31 14:10:17
   * @protected
   * @type {Map<string, boolean>}
   */
  protected cache: Map<string, boolean> = new Map();

  /**
   * 插件缓存
   *
   * @author chitanda
   * @date 2022-10-31 14:10:28
   * @protected
   * @type {Map<string, RemotePluginItem>}
   */
  protected pluginCache: Map<string, RemotePluginItem> = new Map();

  /**
   * 加载插件
   *
   * @author chitanda
   * @date 2022-10-31 14:10:13
   * @param {IPSSysPFPlugin} plugin
   * @return {*}  {Promise<boolean>}
   */
  async loadPlugin(plugin: IPSSysPFPlugin): Promise<boolean> {
    if (plugin.runtimeObject === true) {
      const pluginRef = plugin as unknown as IPSAppPFPluginRef;
      if (pluginRef) {
        return this.loadPluginRef(pluginRef);
      }
    }
    return false;
  }

  /**
   * 加载应用饮用插件
   *
   * @author chitanda
   * @date 2022-10-31 16:10:57
   * @param {IPSAppPFPluginRef} pluginRef
   * @return {*}  {Promise<boolean>}
   */
  async loadPluginRef(pluginRef: IPSAppPFPluginRef): Promise<boolean> {
    if (ibiz.env.dev) {
      ibiz.log.warn(
        `开发模式，远程插件[${pluginRef.name}]不加载。开发测试请在本地引入!`,
      );
      return true;
    }
    if (this.pluginCache.has(pluginRef.rTObjectName)) {
      return true;
    }
    let config: IParams = {};
    try {
      config = JSON.parse(pluginRef.rTObjectRepo);
    } catch (err) {
      throw new DefectModelError(
        pluginRef,
        `插件[${pluginRef.name}]参数格式异常请检查`,
      );
    }
    const remotePlugin = new RemotePluginItem(
      pluginRef.rTObjectName,
      config as unknown as RemotePluginConfig,
    );
    if (remotePlugin) {
      await this.loadScript(remotePlugin.script);
      if (remotePlugin.styles instanceof Array) {
        await this.loadStyles(remotePlugin.styles);
      } else {
        await this.loadStyles([remotePlugin.styles]);
      }
      this.pluginCache.set(pluginRef.rTObjectName, remotePlugin);
      return true;
    }
    return false;
  }

  /**
   * 加载插件
   *
   * @author chitanda
   * @date 2022-10-31 14:10:10
   * @protected
   * @param {string} scriptUrl
   * @return {*}  {Promise<void>}
   */
  protected async loadScript(scriptUrl: string): Promise<void> {
    if (scriptUrl) {
      if (this.cache.has(scriptUrl)) {
        return;
      }
      const url = this.parseUrl(scriptUrl);
      const module = await System.import(url);
      if (module) {
        if (module.default) {
          Vue.use(module.default);
        } else {
          throw new RuntimeError(
            `远程插件加载失败, 远程插件未找到[default]默认导出`,
          );
        }
        this.cache.set(scriptUrl, true);
      } else {
        throw new RuntimeError(`远程插件加载失败, 未找到文件`);
      }
    }
  }

  /**
   * 加载插件样式文件
   *
   * @author chitanda
   * @date 2022-10-31 12:10:23
   * @protected
   * @param {(string[])} styles
   * @return {*}  {Promise<void>}
   */
  protected async loadStyles(styles: string[]): Promise<void> {
    if (styles && styles.length > 0) {
      const all = styles.map(styleUrl => {
        if (this.cache.has(styleUrl)) {
          return false;
        }
        return new Promise((resolve, reject) => {
          const url = this.parseUrl(styleUrl);
          const linkDom = document.createElement('link');
          linkDom.setAttribute('type', 'text/css');
          linkDom.setAttribute('rel', 'stylesheet');
          linkDom.setAttribute('href', url);
          linkDom.onload = resolve;
          linkDom.onerror = reject;
          document.head.appendChild(linkDom);
        });
      });
      await Promise.all(all);
    }
  }

  /**
   * 编译请求文件地址
   *
   * @author chitanda
   * @date 2022-10-31 14:10:19
   * @protected
   * @param {string} script
   * @return {*}  {string}
   */
  protected parseUrl(script: string): string {
    if (script.startsWith('http://') || script.startsWith('https://')) {
      return script;
    }
    return `${ibiz.env.pluginBaseUrl}${script}`;
  }
}
