/* eslint-disable no-param-reassign */
import { GridController } from '@ibiz-template/controller';
import { GridModel } from '@ibiz-template/model';
import { ControlVO } from '@ibiz-template/service';
import { computed, ref } from 'vue';

/**
 * 生成iViewTable要用的columns
 *
 * @author lxm
 * @date 2022-08-31 19:08:34
 * @export
 * @param {GridController} c
 * @returns {*}  {IData[]}
 */
export function generateIViewColumns(c: GridController): IData[] {
  const gridModel: GridModel = c.model;
  const columns: IData[] = gridModel.columns.map(column => {
    const width = column.source.widthUnit === 'STAR' ? undefined : column.width;
    return {
      title: column.title,
      width,
      minWidth: column.width,
      align: column.source.align?.toLowerCase() || 'center',
      slot: column.codeName,
      key: column.codeName,
      ellipsis: true,
      tooltip: false, // todo 表格提示用title
      resizable: true,
      sortable: !c.noSort && column.source.enableSort ? 'custom' : false,
    };
  });

  // 多选的时候给第一列添加选择列
  if (!c.singleSelect) {
    columns.splice(0, 0, { type: 'selection', width: 60 });
  }

  return columns;
}

/**
 * 获取iview的Table使用的columns响应式变量
 *
 * @author lxm
 * @date 2022-08-18 19:08:22
 * @export
 * @param {GridController} c
 * @returns [columns, originColumns 原始]
 */
export function useITableColumns(c: GridController) {
  // 原始iview用列数据
  const originColumns = ref<IData[]>([]);

  c.nerve.self.evt.on('created', () => {
    originColumns.value = generateIViewColumns(c);
  });

  // 实际iview使用的columns
  const columns = computed(() => {
    // 先浅拷贝
    const copy = [...originColumns.value];

    // 如果没有自适应列，深拷贝最后一个对象，改成自适应
    const flexColumn = copy.find(column => !!column.width);
    if (flexColumn) {
      // 修改自适应列的索引，默认最后一个
      let index = copy.length - 1;
      if (copy[index].key === 'rowEdit') {
        index -= 1;
      }
      copy[index] = {
        ...copy[index],
        width: undefined,
      };
    }
    return copy;
  });
  return [columns, originColumns];
}

/**
 * 适配iview的table的事件
 *
 * @author lxm
 * @date 2022-09-05 21:09:42
 * @export
 * @param {GridController} c
 * @returns {*}
 */
export function useITableEvent(c: GridController) {
  const getOriginData = (data: IData): ControlVO => {
    return c.items.find(item => item.srfkey === data.srfkey)!;
  };

  function onRowClick(data: IData) {
    const origin = getOriginData(data);
    // 单选的时候选中效果靠这个回调
    if (c.singleSelect) {
      c.onSelectionChange([origin]);
    }
    c.onRowClick(origin);
  }

  function onDbRowClick(data: IData) {
    c.onDbRowClick(getOriginData(data));
  }

  function onSelectionChange(selection: IData[]) {
    // 单选的时候选中效果靠点击事件
    if (!c.singleSelect) {
      const origins = selection.map(item => getOriginData(item));
      c.onSelectionChange(origins);
    }
  }

  function onSortChange(opts: {
    column: IData;
    key: string;
    order: 'asc' | 'desc';
  }) {
    const { key, order } = opts;
    c.setSort(key, order);
    c.load();
  }
  return { onRowClick, onDbRowClick, onSelectionChange, onSortChange };
}

/**
 * 使用表格分页组件
 *
 * @author lxm
 * @date 2022-09-06 17:09:09
 * @export
 * @param {GridController} c
 * @returns {*}
 */
export function useAppGridPagination(c: GridController) {
  function onPageChange(page: number) {
    if (!page || page === c.curPage) {
      return;
    }
    c.curPage = page;
    c.load();
  }

  function onPageSizeChange(size: number) {
    if (!size || size === c.size) {
      return;
    }
    c.size = size;

    // 当page为第一页的时候切换size不会触发pageChange，需要自己触发加载
    if (c.curPage === 1) {
      c.load();
    }
  }

  function onPageReset() {
    c.curPage = 1;
    c.load();
  }
  return { onPageChange, onPageSizeChange, onPageReset };
}
