import { toastController } from '@ionic/core';
import { Dialog } from 'vant';
import store from '../../../../../src/store';

/**
 * 消息提示
 *
 * @export
 * @class AppNoticeService
 */
export class AppNoticeService {

    /**
     * store
     */
    public store: any = null;

    /**
     * 唯一实例
     *
     * @private
     * @static
     * @type {AppNoticeService}
     * @memberof AppNoticeService
     */
    private static readonly instance: AppNoticeService = new AppNoticeService();


    /**
     * Creates an instance of AppNoticeService.
     * @memberof AppNoticeService
     */
    constructor() {
        this.initBasicData();
        if (AppNoticeService.instance) {
            return AppNoticeService.instance;
        }
    }

    /**
     * 初始化基础数据
     * 
     * @memberof AppDrawer
     */
    private initBasicData() {
        this.store = store;
    }


    /**
     * 消息提示
     *
     * @param {string} message
     * @param {number} [time]
     * @returns {Promise<any>}
     * @memberof AppNoticeService
     */
    public async info(message: string, time?: number): Promise<any> {
        const type = 'secondary';
        return this.createToast(type, message, time);
    }

    /**
     * 成功提示
     *
     * @param {string} message
     * @param {number} [time]
     * @returns {Promise<any>}
     * @memberof AppNoticeService
     */
    public async success(message: string, time?: number): Promise<any> {
        const type = 'success';
        return this.createToast(type, message, time);
    }

    /**
     * 警告提示
     *
     * @param {string} message
     * @param {number} [time]
     * @returns {Promise<any>}
     * @memberof AppNoticeService
     */
    public async warning(message: string, time?: number): Promise<any> {
        const type = 'warning';
        return this.createToast(type, message, time);
    }

    /**
     * 错误提示
     *
     * @param {string} message
     * @param {number} [time]
     * @returns {Promise<any>}
     * @memberof AppNoticeService
     */
    public async error(message: string, time?: number): Promise<any> {
        const type = 'danger';
        return this.createToast(type, message, time);
    }


    /**
     * 确认操作
     *
     * @param {string} title 标题
     * @param {string} message 信息
     * @param {*} xData 当前this指向
     * @return {*}  {Promise<boolean>}
     * @memberof AppNoticeService
     */
    public async confirm(title: string, message: string, xData: any): Promise<boolean> {
        return new Promise(async (resolve) => {
            if (this.store && this.store.state && this.store.state.noticeStatus) {
                Dialog.confirm({
                    title: title ? title : '标题',
                    message: message,
                    confirmButtonText: xData.$t('app.commonWords.ok'),
                    cancelButtonText: xData.$t('app.commonWords.cancel')
                })
                    .then(() => {
                        if (this.store && this.store.commit) {
                            this.store.commit('setNoticeStatus', true);
                        }
                        resolve(true);
                    })
                    .catch(() => {
                        if (this.store && this.store.commit) {
                            this.store.commit('setNoticeStatus', true);
                        }
                        resolve(false);
                    });
                if (this.store && this.store.commit) {
                    this.store.commit('setNoticeStatus', false);
                }
            }
        });
    }

    /**
     * 创建对象
     *
     * @private
     * @param {string} type
     * @param {string} message
     * @param {number} [time]
     * @memberof AppNoticeService
     */
    private async createToast(type: string, message: string, time?: number) {
        const toast = await toastController.create({
            position: 'top',
            color: type ? type : 'primary',
            duration: time ? time : 2000,
            message: message,
        });
        await toast.present();
    }

    /**
     * 获取实例
     *
     * @static
     * @returns {AppNoticeService}
     * @memberof AppNoticeService
     */
    public static getInstance(): AppNoticeService {
        return this.instance;
    }

}