import { Body, Controller, Delete, Get, Param, Post, Put, Req, Res } from '@nestjs/common';
import { FastifyRequest, FastifyReply } from 'fastify';
import { NetService, ControllerBase } from '@/core';
import { Environment } from '@/environment';

@Controller()
export class BookController extends ControllerBase {
  protected readonly http: NetService = new NetService('book');

  @Post('/books')
  async create(
    @Req() request: FastifyRequest,
    @Res() response: FastifyReply,
    @Body() body: any,
  ): Promise<any> {
    if (Environment.EnableRuntime) {
      const url = `/book/create`;
      this.callAPI(request, response, url, body);
      return;
    }
    const url = this.parseUrl(request.url);
    const res = await this.http.post(request, response, url, body);
    return this.parseResponse(request, response, res);
  }

  @Get('/books/:book')
  async get(
    @Req() request: FastifyRequest,
    @Res() response: FastifyReply,
    @Param('book') book: string,
  ): Promise<any> {
    if (Environment.EnableRuntime) {
      const url = `/book/get/${encodeURIComponent(book)}`;
      this.callAPI(request, response, url);
      return;
    }
    const url = this.parseUrl(request.url);
    const res = await this.http.get(request, response, url);
    return this.parseResponse(request, response, res);
  }

  @Delete('/books/:book')
  async remove(
    @Req() request: FastifyRequest,
    @Res() response: FastifyReply,
    @Param('book') book: string,
  ): Promise<any> {
    if (Environment.EnableRuntime) {
      const url = `/book/remove/${encodeURIComponent(book)}`;
      this.callAPI(request, response, url);
      return;
    }
    const url = this.parseUrl(request.url);
    const res = await this.http.delete(request, response, url);
    return this.parseResponse(request, response, res);
  }

  @Put('/books/:book')
  async update(
    @Req() request: FastifyRequest,
    @Res() response: FastifyReply,
    @Param('book') book: string,
    @Body() body: any,
  ): Promise<any> {
    if (Environment.EnableRuntime) {
      const url = `/book/update/${encodeURIComponent(book)}`;
      this.callAPI(request, response, url, body);
      return;
    }
    const url = this.parseUrl(request.url);
    const res = await this.http.put(request, response, url, body);
    return this.parseResponse(request, response, res);
  }

  @Post('/books/checkkey')
  async checkKey(
    @Req() request: FastifyRequest,
    @Res() response: FastifyReply,
    @Body() body: any,
  ): Promise<any> {
    if (Environment.EnableRuntime) {
      const url = `/book/checkkey`;
      this.callAPI(request, response, url, body);
      return;
    }
    const url = this.parseUrl(request.url);
    const res = await this.http.post(request, response, url, body);
    return this.parseResponse(request, response, res);
  }

  @Get('/books/getdraft')
  async getDraft(
    @Req() request: FastifyRequest,
    @Res() response: FastifyReply,
    @Body() body: any,
  ): Promise<any> {
    if (Environment.EnableRuntime) {
      const url = `/book/getdraft`;
      this.callAPI(request, response, url);
      return;
    }
    const url = this.parseUrl(request.url);
    const res = await this.http.get(request, response, url);
    return this.parseResponse(request, response, res);
  }

  @Post('/books/fetchdefault')
  async fetchDefault(
    @Req() request: FastifyRequest,
    @Res() response: FastifyReply,
    @Body() body: any,
  ): Promise<any> {
    if (Environment.EnableRuntime) {
      const url = `/book/fetchdefault`;
      this.callFetchAPI(request, response, url, body);
      return;
    }
    const url = this.parseUrl(request.url);
    const res = await this.http.post(request, response, url, body);
    return this.parseResponse(request, response, res);
  }

  @Delete('/books/batch')
  async removeBatch(
    @Req() request: FastifyRequest,
    @Res() response: FastifyReply,
    @Body() body: any
  ): Promise<any> {
    if (Environment.EnableRuntime) {
      let keys = '';
      if (body && body instanceof Array) {
        keys = body.join(',');
      }
      const url = `/book/remove/${keys}`;
      delete request.headers['content-length'];
      this.callAPI(request, response, url);
      return;
    }
    const url = this.parseUrl(request.url);
    const res = await this.http.delete(request, response, url, { data: body });
    return this.parseResponse(request, response, res);
  }

  @Post('/books/batch')
  async saveBatch(
    @Req() request: FastifyRequest,
    @Res() response: FastifyReply,
    @Body() body: any
  ): Promise<any> {
    if (Environment.EnableRuntime) {
      const url = `/book/save`;
      this.callAPI(request, response, url, body);
      return;
    }
    const url = this.parseUrl(request.url);
    const res = await this.http.post(request, response, url, body);
    return this.parseResponse(request, response, res);
  }
}
