import schema from 'async-validator';
import { debounceTime, distinctUntilChanged } from 'rxjs/operators';
import { FormButtonModel, FormDruipartModel, FormGroupPanelModel, FormIFrameModel, FormItemModel, FormPageModel, FormPartModel, FormRawItemModel, FormTabPageModel, FormTabPanelModel, FormUserControlModel, ModelTool, Util, Verify, ViewTool, EditFormControlInterface, IParams } from 'ibiz-core';
import { FormControlBase } from './form-control-base';
import { AppFormService } from '../ctrl-service';
import moment from 'moment';
import { AppCenterService, AppViewLogicService, CodeListTranslator } from '../app-service';
import { IPSAppDEUIAction, IPSDEEditForm, IPSDEEditFormItem, IPSDEFDCatGroupLogic, IPSDEFDLogic, IPSDEFDSingleLogic, IPSDEFIUpdateDetail, IPSDEFormButton, IPSDEFormDetail, IPSDEFormDRUIPart, IPSDEFormGroupPanel, IPSDEFormItem, IPSDEFormItemUpdate, IPSDEFormItemVR, IPSDEFormPage, IPSDEFormTabPage, IPSDEFormTabPanel, IPSUIActionGroupDetail, IPSDEFormItemEx, IPSCodeListEditor } from '@ibiz/dynamic-model-api';

/**
 * 编辑表单部件基类
 *
 * @export
 * @class EditFormControlBase
 * @extends {FormControlBase}
 */
export class EditFormControlBase extends FormControlBase implements EditFormControlInterface {

    /**
     * 表单的模型对象
     *
     * @type {*}
     * @memberof FormControlBase
     */
    public declare controlInstance: IPSDEEditForm;

    /**
     * 是否自动加载
     *
     * @type {*}
     * @memberof EditFormControlBase
     */
    public isAutoLoad?: any;

    /**
     * 是否可以编辑（主要用于工作流提交不需要先更新，直接提交数据）
     *
     * @type {*}
     * @memberof EditFormControlBase
     */
    public isEditable?: any;

    /**
     * 是否默认保存
     *
     * @type {*}
     * @memberof EditFormControlBase
     */
    public isAutoSave?: any;

    /**
     * 是否显示导航栏
     *
     * @type {*}
     * @memberof EditFormControlBase
     */
    public showFormNavBar?: boolean;

    /**
     * 部件行为--submit
     *
     * @type {*}
     * @memberof EditFormControlBase
     */
    public WFSubmitAction?: any;

    /**
     * 部件行为--start
     *
     * @type {*}
     * @memberof EditFormControlBase
     */
    public WFStartAction?: any;

    /**
     * 部件行为--update
     *
     * @type {*}
     * @memberof EditFormControlBase
     */
    public updateAction: any;

    /**
     * 部件行为--remove
     *
     * @type {*}
     * @memberof EditFormControlBase
     */
    public removeAction: any;

    /**
     * 部件行为--create
     *
     * @type {*}
     * @memberof EditFormControlBase
     */
    public createAction?: any;

    /**
     * 部件行为--create
     *
     * @type {*}
     * @memberof EditFormControlBase
     */
    public searchAction?: any;

    /**
     * 主键表单项名称
     *
     * @protected
     * @type {string}
     * @memberof EditFormControlBase
     */
    public majorKeyItemName: string = "";

    /**
     * 主信息属性映射表单项名称
     *
     * @type {string}
     * @memberof EditFormControlBase
     */
    public majorMessageItemName: string = "";

    /**
      * 当前执行的行为逻辑
      *
      * @type {string}
      * @memberof EditFormControlBase
      */
    public currentAction: string = "";

    /**
     * 工作流审批意见控件绑定值
     *
     * @memberof EditFormControlBase
     */
    public srfwfmemo: string = "";

    /**
     * 关系界面数量
     *
     * @type {number}
     * @memberof EditFormControlBase
     */
    public drCount: number = 0;

    /**
      * 关系界面计数器
      *
      * @type {number}
      * @memberof EditFormControlBase
      */
    public drcounter: number = 0;

    /**
      * 需要等待关系界面保存时，第一次调用save参数的备份
      *
      * @type {number}
      * @memberof EditFormControlBase
      */
    public drsaveopt: any = {};

    /**
      * 表单保存回调存储对象
      *
      * @type {any}
      * @memberof EditFormControlBase
      */
    public saveState: any;

    /**
     * 表单项校验错误提示信息
     * 
     *  @memberof  EditFormControlBase
     */
    public errorMessages: Array<any> = [];

    /**
     * 保存时的显示处理提示
     * 
     * @memberof  EditFormControlBase
     */
    public showResultInfo: boolean = true;

    /**
     * 表单分组锚点数据集合
     * 
     * @memberof  EditFormControlBase
     */
    public groupAnchorDatas: any[] = [];

    /**
     * 监听静态参数变化
     *
     * @param {*} newVal
     * @param {*} oldVal
     * @memberof EditFormControlBase
     */
    public onStaticPropsChange(newVal: any, oldVal: any) {
        this.isAutoLoad = this.staticProps.isautoload;
        this.isEditable = this.staticProps.iseditable;
        super.onStaticPropsChange(newVal, oldVal);
    }

    /**
     * 表单部件模型数据加载
     *
     * @memberof EditFormControlBase
     */
    public async ctrlModelLoad() {
        await super.ctrlModelLoad();
        // 加载关系界面
        let allFormDetails: IPSDEEditFormItem[] = ModelTool.getAllFormDetails(this.controlInstance);
        for (let formDetail of allFormDetails) {
            if (formDetail.detailType == 'DRUIPART') {
                let refView = (formDetail as any)?.getPSAppView();
                if (refView) {
                    await refView.fill(true);
                }
            }
        }
    }

    /**
     * 部件模型数据初始化实例
     *
     * @memberof EditFormControlBase
     */
    public async ctrlModelInit(args?: any) {
        await super.ctrlModelInit();
        if (!(this.Environment && this.Environment.isPreviewMode)) {
            this.service = new AppFormService(this.controlInstance, this.context, { localSourceTag: this.localSourceTag });
            await this.service.loaded();
        }
        this.showFormNavBar = this.controlInstance.showFormNavBar;
        this.isAutoSave = this.controlInstance.enableAutoSave;
        this.loaddraftAction = this.controlInstance.getGetDraftPSControlAction?.()?.actionName;
        this.updateAction = this.controlInstance.getUpdatePSControlAction?.()?.getPSAppDEMethod?.()?.codeName || "Update";
        this.removeAction = this.controlInstance.getRemovePSControlAction?.()?.actionName;
        this.loadAction = this.controlInstance.getGetPSControlAction?.()?.actionName;
        this.createAction = this.controlInstance.getCreatePSControlAction?.()?.getPSAppDEMethod?.()?.codeName || "Create";
        this.WFSubmitAction = (this.controlInstance as any).getWFSubmitPSControlAction?.()?.actionName;
        this.WFStartAction = (this.controlInstance as any).getWFStartPSControlAction?.()?.actionName;
        // 初始化data
        this.controlInstance.getPSDEFormItems()?.forEach((formItem: IPSDEFormItem) => {
            this.$set(this.data, formItem.id, null);
        });
        // 初始化表单成员运行时模型
        this.initDetailsModel();
        // 初始化静态值规则
        this.initRules();
    }

    /**
     * 编辑表单初始化
     *
     * @memberof EditFormControlBase
     */
    public ctrlInit(args?: any) {
        super.ctrlInit(args);
        if (this.isAutoLoad) {
            this.autoLoad({ srfkey: this.context[this.appDeCodeName.toLowerCase()] });
        }
        if (this.viewState) {
            this.viewStateEvent = this.viewState.subscribe(({ tag, action, data }: { tag: string, action: string, data: any }) => {
                if (!Object.is(tag, this.name)) {
                    return;
                }
                if (Object.is('save', action)) {
                    this.save(data, data.showResultInfo);
                }
                if (Object.is('remove', action)) {
                    this.remove(data);
                }
                if (Object.is('saveandexit', action)) {
                    this.saveAndExit(data);
                }
                if (Object.is('saveandnew', action)) {
                    this.saveAndNew(data);
                }
                if (Object.is('removeandexit', action)) {
                    this.removeAndExit(data);
                }
                if (Object.is('refresh', action)) {
                    this.refresh(data);
                }
                if (Object.is('panelaction', action)) {
                    this.panelAction(data.action, data.emitAction, data.data);
                }
            });
        }
        if (this.controlInstance.infoFormMode && AppCenterService.getMessageCenter()) {
            this.appStateEvent = AppCenterService.getMessageCenter().subscribe(({ name, action, data }: { name: string, action: string, data: any }) => {
                if (!this.appDeCodeName || !Object.is(name, this.appDeCodeName)) {
                    return;
                }
                if (Object.is(action, 'appRefresh')) {
                    this.refresh(data);
                }
            })
        }
        if (this.dataChang) {
            this.dataChang.pipe(debounceTime(300), distinctUntilChanged()).subscribe((data: any) => {
                this.handleDataChange();
            });
        }
    }

    /**
     * 处理dataChang下发的事件
     *
     * @memberof EditFormControlBase
     */
    public handleDataChange() {
        if (this.isAutoSave) {
            this.autoSave();
        }
        const state = !Object.is(JSON.stringify(this.oldData), JSON.stringify(this.data)) ? true : false;
        this.$store.commit('viewAction/setViewDataChange', { viewTag: this.viewtag, viewDataChange: state });
    }

    /**
     * 表单值变化
     *
     * @public
     * @param {{ name: string}} { name}
     * @returns {void}
     * @memberof EditFormControlBase
     */
    public formDataChange({ name }: { name: string }): void {
        if (this.ignorefieldvaluechange || this.checkIgnoreInput(name)) {
            return;
        }
        this.resetFormData({ name: name });
        this.formLogic({ name: name });
        this.dataChang.next(JSON.stringify(this.data));
    }

    /**
     * 加载草稿
     *
     * @param {*} opt 额外参数
     * @memberof EditFormControlBase
     */
    public async loadDraft(opt: any = {}): Promise<void> {
        let callBack: any;
        if (!this.loaddraftAction) {
            this.$throw(`${this.controlInstance.codeName}` + (this.$t('app.formpage.notconfig.loaddraftaction') as string), 'loadDraft');
            return;
        }
        this.createDefault();
        if (opt.callBack) {
            callBack = opt.callBack;
            delete opt.callBack;
        }
        const arg: any = { ...opt };
        let viewparamResult: any = Object.assign(arg, this.viewparams);
        let tempContext: any = JSON.parse(JSON.stringify(this.context));
        if (!(await this.handleCtrlEvents('onbeforeloaddraft', { action: this.loaddraftAction, navParam: viewparamResult }))) {
            return;
        }
        this.onControlRequset('loadDraft', tempContext, viewparamResult);
        try {
            const response: any = await this.service.loadDraft(this.loaddraftAction, tempContext, { viewparams: viewparamResult }, this.showBusyIndicator);
            this.onControlResponse('loadDraft', response);
            if (!response.status || response.status !== 200) {
                if (!(await this.handleCtrlEvents('onloaddrafterror', { action: this.loaddraftAction, navParam: viewparamResult, data: response?.data }))) {
                    return;
                }
                this.$throw(response, 'loadDraft');
                return;
            }
            const data = response.data;
            if (!(await this.handleCtrlEvents('onloaddraftsuccess', { action: this.loaddraftAction, navParam: viewparamResult, data: data }))) {
                return;
            }
            this.resetDraftFormStates();
            await this.onFormLoad(data, 'loadDraft');
            if (callBack && (callBack instanceof Function)) callBack();

            // 删除主键表单项的值
            this.controlInstance.getPSDEFormItems()?.find((item: IPSDEFormItem) => {
                if (item.getPSAppDEField()?.keyField && !item.hidden) {
                    data[item.name] = null;
                }
            })
            this.ctrlEvent({
                controlname: this.controlInstance.name,
                action: 'load',
                data: data,
            });
            this.$nextTick(() => {
                this.formState.next({ type: 'load', data: data });
            });
            setTimeout(() => {
                const form: any = this.$refs.form;
                if (form) {
                    form.fields.forEach((field: any) => {
                        field.validateMessage = "";
                        field.validateState = "";
                        field.validateStatus = false;
                    });
                }
            });
        } catch (error: any) {
            this.onControlResponse('loadDraft', error);
            if (!(await this.handleCtrlEvents('onloaddrafterror', { action: this.loaddraftAction, navParam: viewparamResult, data: error?.data }))) {
                return;
            }
            this.$throw(error, 'loadDraft');
        }
    }

    /**
     * 自动保存
     *
     * @param {*} opt 额外参数
     * @memberof EditFormControlBase
     */
    public autoSave(opt: any = {}): void {
        if (!this.formValidateStatus()) {
            return;
        }
        const arg: any = { ...opt };
        const data = this.getData();
        Object.assign(arg, data);
        Object.assign(arg, { srfmajortext: data[this.majorMessageItemName] });
        const action: any = Object.is(data.srfuf, '1') ? this.updateAction : this.createAction;
        if (!action) {
            let actionName: any = Object.is(data.srfuf, '1') ? "updateAction" : "createAction";
            this.$throw(`${this.controlInstance.codeName}` + (this.$t('app.formpage.notconfig.actionname') as string), 'autoSave');
            return;
        }
        Object.assign(arg, { viewparams: this.viewparams });
        let tempContext: any = JSON.parse(JSON.stringify(this.context));
        this.onControlRequset('autoSave', tempContext, arg);
        const post: Promise<any> = this.service.add(action, tempContext, arg, this.showBusyIndicator);
        post.then(async (response: any) => {
            this.onControlResponse('autoSave', response);
            if (!response.status || response.status !== 200) {
                this.$throw(response, 'autoSave');
                return;
            }
            const data = response.data;
            await this.onFormLoad(data, 'autoSave');
            this.ctrlEvent({
                controlname: this.controlInstance.name,
                action: 'save',
                data: data,
            });
            this.$nextTick(() => {
                this.formState.next({ type: 'save', action: 'autoSave', data: data });
            });
        }).catch((response: any) => {
            this.onControlResponse('autoSave', response);
            if (response && response.status && response.data) {
                this.$throw(response, 'autoSave', { dangerouslyUseHTMLString: true });
            } else {
                this.$throw((this.$t('app.commonwords.sysexception') as string), 'autoSave');
            }
        });
    }

    /**
     * 保存
     *
     * @param {*} opt 额外参数
     * @param {boolean} showResultInfo 是否显示提示信息
     * @param {boolean} isStateNext formState是否下发通知
     * @return {*}  {Promise<any>}
     * @memberof EditFormControlBase
     */
    public async save(opt: any = {}, showResultInfo: boolean = true, isStateNext: boolean = true): Promise<any> {
        return new Promise((resolve: any, reject: any) => {
            if (!this.formValidateStatus()) {
                this.$throw(this.$t('app.searchform.globalerrortip') as string, 'save', { dangerouslyUseHTMLString: true });
                return;
            }
            const arg: any = { ...opt };
            const data = this.getData();
            Object.assign(arg, this.context);
            Object.assign(arg, data);
            Object.assign(arg, { srfmajortext: data[this.majorMessageItemName] });
            this.showResultInfo = showResultInfo;
            if (isStateNext && this.drCount > 0) {
                this.drcounter = this.drCount;
                this.drsaveopt = opt;
                this.formState.next({ type: 'beforesave', data: arg });//先通知关系界面保存
                this.saveState = resolve;
                return;
            }
            if (this.viewparams && this.viewparams.copymode) {
                data.srfuf = '0';
            }
            const action: any = Object.is(data.srfuf, '1') ? this.updateAction : this.createAction;
            const actionName: any = Object.is(data.srfuf, '1') ? "updateAction" : "createAction";
            if (!action) {
                this.$throw(`${this.controlInstance.codeName}` + (this.$t('app.formpage.notconfig.actionname') as string), 'save');
                return;
            }
            Object.assign(arg, { viewparams: this.viewparams });
            let tempContext: any = JSON.parse(JSON.stringify(this.context));
            this.handleCtrlEvents('onbeforesave', { action: action, navContext: tempContext, navParam: arg }).then((beforeSaveResult: boolean) => {
                if (!beforeSaveResult) {
                    return;
                }
                this.onControlRequset('save', tempContext, arg);
                const post: Promise<any> = Object.is(data.srfuf, '1') ? this.service.update(action, tempContext, arg, this.showBusyIndicator) : this.service.add(action, tempContext, arg, this.showBusyIndicator);
                post.then((response: any) => {
                    this.onControlResponse('save', response);
                    if (!response.status || response.status !== 200) {
                        this.handleCtrlEvents('onsaveerror', { action: action, navContext: tempContext, navParam: arg, data: response?.data }).then((saveErrorResult: boolean) => { })
                        this.$throw(response, 'save');
                        return;
                    }
                    const data = response.data;
                    this.handleCtrlEvents('onsavesuccess', { action: action, navContext: tempContext, navParam: arg, data: data }).then(async (saveSuccessResult: boolean) => {
                        if (!saveSuccessResult) {
                            return;
                        }
                        this.viewparams.copymode = false;
                        await this.onFormLoad(data, 'save');
                        this.ctrlEvent({
                            controlname: this.controlInstance.name,
                            action: 'save',
                            data: data,
                        });
                        AppCenterService.notifyMessage({ name: this.controlInstance.getPSAppDataEntity()?.codeName || '', action: 'appRefresh', data: data });
                        this.$nextTick(() => {
                            this.formState.next({ type: 'save', data: data });
                        });
                        if (this.controlInstance.formFuncMode?.toLowerCase() != 'wizardform' && showResultInfo) {
                            this.$success((data.srfmajortext ? data.srfmajortext : '') + (this.$t(`app.formpage.${actionName}`)) + (this.$t('app.commonwords.success') as string), 'save');
                        }
                        resolve(response);
                    });
                }).catch((response: any) => {
                    this.onControlResponse('save', response);
                    this.handleCtrlEvents('onsaveerror', { action: action, navContext: tempContext, navParam: arg, data: response?.data }).then((saveErrorResult: boolean) => { });
                    if (response && response.status && response.data) {
                        this.$throw(response, 'save', { dangerouslyUseHTMLString: true });
                        reject(response);
                    } else {
                        this.$throw((this.$t('app.commonwords.sysexception') as string), 'save');
                        reject(response);
                    }
                    reject(response);
                });
            })
        })
    }

    /**
    * 删除
    *
    * @param {Array<any>} [opt=[]] 额外参数 
    * @param {boolean} [showResultInfo] 是否显示提示信息
    * @return {*}  {Promise<any>}
    * @memberof EditFormControlBase
    */
    public remove(opt: Array<any> = [], showResultInfo?: boolean): Promise<any> {
        return new Promise((resolve: any, reject: any) => {
            if (!this.removeAction) {
                this.$throw(`${this.controlInstance.codeName}` + (this.$t('app.formpage.notconfig.removeaction') as string), 'remove');
                return;
            }
            const arg: any = opt[0];
            const _this: any = this;
            Object.assign(arg, { viewparams: this.viewparams });
            let tempContext: any = Util.deepCopy(this.context);
            this.handleCtrlEvents('onbeforeremove', { action: this.removeAction, navParam: arg }).then((beforeRemoveResult: boolean) => {
                if (!beforeRemoveResult) {
                    return;
                }
                this.onControlRequset('remove', tempContext, arg);
                this.service.delete(_this.removeAction, tempContext, arg, showResultInfo).then((response: any) => {
                    this.onControlResponse('remove', response);
                    if (response && response.status === 200) {
                        const data = response.data;
                        this.handleCtrlEvents('onremovesuccess', { action: this.removeAction, navParam: arg, data: data }).then((removeSuccessResult: boolean) => {
                            if (!removeSuccessResult) {
                                return;
                            }
                            this.ctrlEvent({
                                controlname: this.controlInstance.name,
                                action: 'remove',
                                data: data,
                            });
                            this.formState.next({ type: 'remove', data: data });
                            this.data.ismodify = false;
                            this.$success((data.srfmajortext ? data.srfmajortext : '') + (this.$t('app.commonwords.deletesuccess') as string), 'remove');
                            AppCenterService.notifyMessage({ name: this.controlInstance.getPSAppDataEntity()?.codeName || '', action: 'appRefresh', data: data });
                            resolve(response);
                        });
                    } else {
                        this.handleCtrlEvents('onremoveerror', { action: this.removeAction, navParam: arg, data: response?.data }).then((removeErrorResult: boolean) => { });
                    }
                }).catch((error: any) => {
                    this.onControlResponse('remove', error);
                    this.handleCtrlEvents('onremoveerror', { action: this.removeAction, navParam: arg, data: error?.data }).then((removeErrorResult: boolean) => { });
                    this.$throw(error, 'remove');
                    reject(error);
                });
            });
        });
    }

    /**
     * 工作流启动
     *
     * @param {*} data  表单数据
     * @param {*} [localdata] 补充逻辑完成参数
     * @return {*}  {Promise<any>}
     * @memberof EditFormControlBase
     */
    public wfstart(data: any, localdata?: any): Promise<any> {
        return new Promise((resolve: any, reject: any) => {
            if (!this.formValidateStatus()) {
                this.$throw(this.$t('app.searchform.globalerrortip') as string, 'wfstart', { dangerouslyUseHTMLString: true });
                reject({});
                return;
            }
            const _this: any = this;
            const formData: any = this.getData();
            const copyData: any = Util.deepCopy(data[0]);
            Object.assign(formData, { viewparams: copyData });
            let tempContext: any = JSON.parse(JSON.stringify(this.context));
            this.handleCtrlEvents('onbeforesave', { action: Object.is(formData.srfuf, '1') ? this.updateAction : this.createAction, navParam: formData }).then((beforeSaveResult: boolean) => {
                if (!beforeSaveResult) {
                    return;
                }
                this.onControlRequset('save', tempContext, formData);
                const post: Promise<any> = Object.is(formData.srfuf, '1') ? this.service.update(this.updateAction, tempContext, formData, this.showBusyIndicator, true) : this.service.add(this.createAction, tempContext, formData, this.showBusyIndicator, true);
                post.then(async (response: any) => {
                    this.onControlResponse('save', response);
                    const arg: any = response.data;
                    const responseData: any = Util.deepCopy(arg);
                    // 保存完成UI处理
                    await this.onFormLoad(arg, 'save');
                    this.ctrlEvent({
                        controlname: this.controlInstance.name,
                        action: 'save',
                        data: arg,
                    });
                    this.$nextTick(() => {
                        this.formState.next({ type: 'save', data: arg });
                    });
                    // 准备工作流数据,填充未存库数据
                    let tempWFData: any = {};
                    if (copyData && Object.keys(copyData).length > 0) {
                        Object.keys(copyData).forEach((key: string) => {
                            if ((!arg.hasOwnProperty(key)) || (!arg[key] && copyData[key])) {
                                tempWFData[key] = copyData[key];
                            }
                        })
                    }
                    // 准备提交参数
                    if (this.viewparams) {
                        let copyViewParams: any = Util.deepCopy(this.viewparams);
                        if (copyViewParams.w) {
                            delete copyViewParams.w;
                        }
                        if (this.appDeKeyFieldName && copyViewParams[this.appDeKeyFieldName.toLocaleLowerCase()]) {
                            delete copyViewParams[this.appDeKeyFieldName.toLocaleLowerCase()]
                        }
                        if (this.appDeMajorFieldName && copyViewParams[this.appDeMajorFieldName.toLocaleLowerCase()]) {
                            delete copyViewParams[this.appDeMajorFieldName.toLocaleLowerCase()]
                        }
                        Object.assign(responseData, copyViewParams);
                    }
                    if (tempWFData && Object.keys(tempWFData).length > 0) {
                        Object.assign(responseData, tempWFData);
                    }
                    Object.assign(arg, { viewparams: responseData });
                    // 强制补充srfwfmemo
                    if (copyData.srfwfmemo) {
                        Object.assign(arg, { srfwfmemo: copyData.srfwfmemo });
                    }
                    let tempContext: any = JSON.parse(JSON.stringify(this.context));
                    this.handleCtrlEvents('onbeforewfstart', { action: this.WFStartAction, navParam: arg }).then((beforeWfStartResult: boolean) => {
                        if (!beforeWfStartResult) {
                            return;
                        }
                        this.onControlRequset('wfstart', tempContext, arg);
                        const result: Promise<any> = this.service.wfstart(_this.WFStartAction, tempContext, arg, this.showBusyIndicator, localdata);
                        result.then((response: any) => {
                            this.onControlResponse('wfstart', response);
                            if (!response || response.status !== 200) {
                                this.handleCtrlEvents('onwfstarterror', { action: this.WFStartAction, navParam: arg, data: response?.data }).then((wfStartErrorResult: boolean) => { });
                                this.$throw((this.$t('app.formpage.workflow.starterror') as string) + ', ' + response.data.message, 'wfstart');
                                return;
                            }
                            this.handleCtrlEvents('onwfstartsuccess', { action: this.WFStartAction, navParam: arg, data: response.data }).then((wfStartSuccessResult: boolean) => {
                                if (!wfStartSuccessResult) {
                                    return;
                                }
                                AppCenterService.notifyMessage({ name: this.controlInstance.getPSAppDataEntity()?.codeName || '', action: 'appRefresh', data: data });
                                // 工作流数据刷新
                                AppCenterService.notifyMessage({ name: 'SysTodo', action: 'appRefresh', data: data });
                                AppCenterService.notifyMessage({ name: 'WFTask', action: 'appRefresh', data: data });
                                this.$success((this.$t('app.formpage.workflow.startsuccess') as string), 'wfstart');
                                resolve(response);
                            });
                        }).catch((response: any) => {
                            this.onControlResponse('wfstart', response);
                            this.handleCtrlEvents('onwfstarterror', { action: this.WFStartAction, navParam: arg, data: response?.data }).then((wfStartErrorResult: boolean) => {
                                if (!wfStartErrorResult) {
                                    return;
                                }
                                this.$throw(response, 'wfstart');
                                reject(response);
                            });
                        });
                    });
                }).catch((response: any) => {
                    this.onControlResponse('wfstart', response);
                    this.handleCtrlEvents('onwfstarterror', { action: this.WFStartAction, data: response?.data }).then((wfStartErrorResult: boolean) => {
                        if (!wfStartErrorResult) {
                            return;
                        }
                        this.$throw(response, 'wfstart');
                        reject(response);
                    });
                })
            })
        });
    }

    /**
     * 工作流提交
     *
     * @param {*} data  表单数据
     * @param {*} [localdata] 补充逻辑完成参数
     * @return {*}  {Promise<any>}
     * @memberof EditFormControlBase
     */
    public wfsubmit(data: any, localdata?: any): Promise<any> {
        return new Promise((resolve: any, reject: any) => {
            if (!this.formValidateStatus()) {
                this.$throw(this.$t('app.searchform.globalerrortip') as string, 'wfsubmit', { dangerouslyUseHTMLString: true });
                return;
            }
            const _this: any = this;
            const arg: any = data[0];
            const copyData: any = Util.deepCopy(arg);
            Object.assign(this.viewparams, copyData);
            Object.assign(arg, { viewparams: this.viewparams });
            const submitData: Function = (arg: any, responseData: any) => {
                // 准备工作流数据,填充未存库数据
                let tempWFData: any = {};
                if (copyData && Object.keys(copyData).length > 0) {
                    Object.keys(copyData).forEach((key: string) => {
                        if ((!arg.hasOwnProperty(key)) || (!arg[key] && copyData[key])) {
                            tempWFData[key] = copyData[key];
                        }
                    })
                }
                // 准备提交参数
                if (this.viewparams) {
                    Object.assign(responseData, this.viewparams);
                }
                if (tempWFData && Object.keys(tempWFData).length > 0) {
                    Object.assign(responseData, tempWFData);
                }
                // 补充逻辑完成参数
                if (localdata && localdata.type && Object.is(localdata.type, "finish")) {
                    Object.assign(responseData, { srfwfpredefaction: "finish" });
                }
                Object.assign(arg, { viewparams: responseData });
                // 强制补充srfwfmemo
                if (copyData.srfwfmemo) {
                    Object.assign(arg, { srfwfmemo: copyData.srfwfmemo });
                }
                let tempContext: any = JSON.parse(JSON.stringify(this.context));
                this.handleCtrlEvents('onbeforewfsubmit', { action: this.WFSubmitAction, navContext: tempContext, navParam: arg }).then((beforeWFSubmitResult: boolean) => {
                    if (!beforeWFSubmitResult) {
                        return;
                    }
                    this.onControlRequset('wfsubmit', tempContext, arg);
                    const result: Promise<any> = this.service.wfsubmit(_this.WFSubmitAction, tempContext, arg, this.showBusyIndicator, localdata);
                    result.then((response: any) => {
                        this.onControlResponse('wfsubmit', response);
                        if (!response || response.status !== 200) {
                            this.handleCtrlEvents('onwfsubmiterror', { action: this.WFSubmitAction, navContext: tempContext, navParam: arg, data: response?.data }).then((wfSubmitErrorResult: boolean) => { });
                            this.$throw((this.$t('app.formpage.workflow.submiterror') as string) + ', ' + response.data.message, 'wfsubmit');
                            return;
                        }
                        this.handleCtrlEvents('onwfsubmitsuccess', { action: this.WFSubmitAction, navContext: tempContext, navParam: arg, data: response.data }).then(async (wfSubmitSuccessResult: boolean) => {
                            if (!wfSubmitSuccessResult) {
                                return;
                            }
                            await this.onFormLoad(arg, 'submit');
                            AppCenterService.notifyMessage({ name: this.controlInstance.getPSAppDataEntity()?.codeName || '', action: 'appRefresh', data: data });
                            // 工作流数据刷新
                            AppCenterService.notifyMessage({ name: 'SysTodo', action: 'appRefresh', data: data });
                            AppCenterService.notifyMessage({ name: 'WFTask', action: 'appRefresh', data: data });
                            this.$success((this.$t('app.formpage.workflow.submitsuccess') as string), 'wfsubmit');
                            resolve(response);
                        });
                    }).catch((response: any) => {
                        this.onControlResponse('wfsubmit', response);
                        this.handleCtrlEvents('onwfsubmiterror', { action: this.WFSubmitAction, navContext: tempContext, navParam: arg, data: response?.data }).then((wfSubmitErrorResult: boolean) => {
                            if (wfSubmitErrorResult) {
                                this.$throw(response, 'wfsubmit');
                                reject(response);
                            }
                        });
                    });
                })
            }
            if (this.isEditable) {
                let tempContext: any = JSON.parse(JSON.stringify(this.context));
                this.handleCtrlEvents('onbeforesave', { action: Object.is(arg.srfuf, '1') ? this.updateAction : this.createAction, navParam: arg }).then((beforeSaveResult: boolean) => {
                    if (!beforeSaveResult) {
                        return;
                    }
                    this.onControlRequset('save', tempContext, arg);
                    const post: Promise<any> = Object.is(arg.srfuf, '1') ? this.service.update(this.updateAction, tempContext, arg, this.showBusyIndicator, true) : this.service.add(this.createAction, tempContext, arg, this.showBusyIndicator, true);
                    post.then((response: any) => {
                        this.onControlResponse('save', response);
                        this.handleCtrlEvents('onsavesuccess', { action: Object.is(arg.srfuf, '1') ? this.updateAction : this.createAction, navParam: arg, data: response?.data }).then(async (saveSuccessResult: boolean) => {
                            if (!saveSuccessResult) {
                                return;
                            }
                            const responseData: any = response.data;
                            let tempResponseData: any = Util.deepCopy(response);
                            this.service.handleResponse('save', tempResponseData);
                            const _arg: any = tempResponseData.data;
                            // 保存完成UI处理
                            await this.onFormLoad(_arg, 'save');
                            this.ctrlEvent({
                                controlname: this.controlInstance.name,
                                action: 'save',
                                data: _arg,
                            });
                            this.$nextTick(() => {
                                this.formState.next({ type: 'save', data: _arg });
                            });
                            submitData(_arg, responseData);
                        })
                    }).catch((response: any) => {
                        this.handleCtrlEvents('onsaveerror', { action: Object.is(arg.srfuf, '1') ? this.updateAction : this.createAction, navParam: arg, data: response?.data }).then((saveErrorResult: boolean) => {
                            if (!saveErrorResult) {
                                return;
                            }
                            this.onControlResponse('save', response);
                            this.$throw(response, 'wfsubmit');
                            reject(response);
                        });
                    })
                });
            } else {
                const responseData: any = this.getData();
                submitData(arg, responseData);
            }
        })
    }

    /**
     * 表单刷新数据
     *
     * @param {any} args 额外参数
     * @memberof EditFormControlBase
     */
    public refresh(args?: any): void {
        let data: any = {};
        if (args && args instanceof Array && args.length > 0) {
            data = args[0];
        } else {
            data = args || {};
        }
        if (this.data.srfkey && !Object.is(this.data.srfkey, '')) {
            Object.assign(data, { srfkey: this.data.srfkey });
            this.load(data);
            return;
        }
        if (this.data.srfkeys && !Object.is(this.data.srfkeys, '')) {
            Object.assign(data, { srfkey: this.data.srfkeys });
            this.load(data);
            return;
        }
    }

    /**
     * 面板行为
     *
     * @param {string} [action] 调用的实体行为
     * @param {string} [emitAction] 抛出行为
     * @param {*} [data={}] 传入数据
     * @param {boolean} [showloading] 是否显示加载状态
     * 
     * @memberof EditFormControlBase
     */
    public panelAction(action: string, emitAction: string, data: any = {}, showloading?: boolean): void {
        if (!action || (action && Object.is(action, ''))) {
            return;
        }
        const arg: any = { ...data };
        const formdata = this.getData();
        Object.assign(arg, this.viewparams);
        Object.assign(arg, formdata);
        let tempContext: any = JSON.parse(JSON.stringify(this.context));
        if (data[this.appDeCodeName.toLowerCase()]) {
            Object.assign(tempContext, { [this.appDeCodeName.toLowerCase()]: data[this.appDeCodeName.toLowerCase()] });
        }
        this.onControlRequset('panelAction', tempContext, arg);
        const post: Promise<any> = this.service.frontLogic(action, tempContext, arg, showloading);
        post.then(async (response: any) => {
            this.onControlResponse('panelAction', response);
            if (!response.status || response.status !== 200) {
                this.$throw(response, 'panelAction');
                return;
            }
            const data = response.data;
            await this.onFormLoad(data, emitAction);
            this.ctrlEvent({
                controlname: this.controlInstance.name,
                action: emitAction,
                data: data,
            });
            this.$nextTick(() => {
                this.formState.next({ type: emitAction, data: data });
            });
        }).catch((response: any) => {
            this.onControlResponse('panelAction', response);
            this.$throw(response, 'panelAction');
        });
    }

    /**
     * 表单项更新
     *
     * @param {string} mode 界面行为名称
     * @param {*} [data={}] 请求数据
     * @param {string[]} updateDetails 更新项
     * @param {boolean} [showloading] 是否显示加载状态
     * @returns {void}
     * @memberof EditFormControlBase
     */
    public updateFormItems(mode: string, data: any = {}, updateDetails: string[], showloading: boolean = false): void {
        if (!mode || (mode && Object.is(mode, ''))) {
            return;
        }
        const arg: any = Object.assign(Util.deepCopy(this.viewparams), data);
        let tempContext: any = JSON.parse(JSON.stringify(this.context));
        this.onControlRequset('updateFormItems', tempContext, arg);
        const post: Promise<any> = this.service.frontLogic(mode, tempContext, arg, showloading);
        post.then((response: any) => {
            this.onControlResponse('updateFormItems', response);
            if (!response || response.status !== 200) {
                this.$throw((this.$t('app.formpage.updateerror') as string), 'updateFormItems');
                return;
            }
            const data = response.data;
            const _data: any = {};
            updateDetails.forEach((name: string) => {
                if (!data || !data.hasOwnProperty(name)) {
                    return;
                }
                Object.assign(_data, { [name]: data[name] });
            });
            this.setFormEnableCond(_data);
            this.fillForm(_data, 'updateFormItem').then(() => {
                this.formLogic({ name: '' });
                this.dataChang.next(JSON.stringify(this.data));
                this.$nextTick(() => {
                    this.formState.next({ type: 'updateformitem', ufimode: arg.srfufimode, data: _data });
                });
            })
        }).catch((response: any) => {
            this.onControlResponse('updateFormItems', response);
            this.$throw(response, 'updateFormItems');
        });
    }

    /**
     * 保存并退出
     *
     * @param {any[]} data 额外参数
     * @return {*}  {Promise<any>}
     * @memberof EditFormControlBase
     */
    public saveAndExit(data: any[]): Promise<any> {
        let _this = this;
        return new Promise((resolve: any, reject: any) => {
            let arg: any = {};
            if (data && data.length > 0) {
                Object.assign(arg, data[0]);
            }
            _this.currentAction = "saveAndExit";
            _this.save([arg]).then((res) => {
                if (res) {
                    _this.closeView(res.data);
                }
                resolve(res);
            }).catch((error) => {
                reject(error);
            })
        })
    }

    /**
     * 保存并新建
     *
     * @param {any[]} data 额外参数
     * @return {*}  {Promise<any>}
     * @memberof EditFormControlBase
     */
    public saveAndNew(data: any[]): Promise<any> {
        let _this = this;
        return new Promise((resolve: any, reject: any) => {
            let arg: any = {};
            if (data && data.length > 0) {
                Object.assign(arg, data[0]);
            }
            _this.currentAction = "saveAndNew";
            _this.save([arg]).then((res) => {
                _this.ResetData(res);
                _this.loadDraft({});
            }).catch((error) => {
                reject(error);
            })
        })
    }

    /**
     * 删除并退出
     *
     * @param {any[]} data 额外参数
     * @return {*}  {Promise<any>}
     * @memberof EditFormControlBase
     */
    public removeAndExit(data: any[]): Promise<any> {
        let _this = this;
        return new Promise((resolve: any, reject: any) => {
            let arg: any = {};
            if (data && data.length > 0) {
                Object.assign(arg, data[0]);
            }
            _this.remove([arg]).then((res) => {
                if (res) {
                    _this.closeView(res.data);
                }
                resolve(res);
            }).catch((error) => {
                reject(error);
            })
        })
    }

    /**
    * 关系界面数据保存完成
    *
    * @param {any} $event
    * @memberof EditFormControlBase
    */
    public drdatasaved($event: any) {
        let _this = this;
        this.drcounter--;
        if (this.drcounter === 0) {
            this.save(this.drsaveopt, this.showResultInfo, false).then((res) => {
                this.saveState(res);
                this.drsaveopt = {};
                if (Object.is(_this.currentAction, "saveAndNew")) {
                    _this.ResetData(res);
                    _this.loadDraft({});
                } else if (Object.is(_this.currentAction, "saveAndExit")) {
                    if (res) {
                        _this.closeView(res.data);
                    }
                }
            });
        }
    }

    /**
     * 表单加载完成
     *
     * @public
     * @param {*} [data={}]
     * @param {string} [action]
     * @memberof EditFormControlBase
     */
    public async onFormLoad(data: any = {}, action: string): Promise<void> {
        if (this.appDeCodeName.toLowerCase()) {
            if (Object.is(action, "save") || Object.is(action, "autoSave") || Object.is(action, "submit"))
                // 更新context的实体主键
                if (data[this.appDeCodeName.toLowerCase()]) {
                    Object.assign(this.context, { [this.appDeCodeName.toLowerCase()]: data[this.appDeCodeName.toLowerCase()] })
                }
        }
        this.setFormEnableCond(data);
        this.computeButtonState(data);
        await this.fillForm(data, action)
        this.oldData = {};
        Object.assign(this.oldData, Util.deepCopy(this.data));
        this.$store.commit('viewAction/setViewDataChange', { viewTag: this.viewtag, viewDataChange: false });
        this.formLogic({ name: '' });
    }

    /**
     * 值填充
     *
     * @param {*} [_datas={}]
     * @param {string} [action]
     * @memberof EditFormControlBase
     */
    public async fillForm(_datas: any = {}, action: string): Promise<void> {
        this.ignorefieldvaluechange = true;
        for (let i = 0; i < Object.keys(_datas).length; i++) {
            const name = Object.keys(_datas)[i];
            if (this.data.hasOwnProperty(name)) {
                // 是否转化为代码项文本
                if (this.detailsModel[name] && this.detailsModel[name]['convertToCodeItemText'] && this.detailsModel[name]['codelist']) {
                    const codeListTranslator: CodeListTranslator = new CodeListTranslator();
                    const text: string = await codeListTranslator.getCodeListText(_datas[name], this.detailsModel[name]['codelist'], this, this.context, this.viewparams);
                    this.data[name] = text;
                } else {
                    this.data[name] = _datas[name];
                }
            }
        }
        if (Object.is(action, 'loadDraft')) {
            this.createDefault();
        }
        if (Object.is(action, 'load')) {
            this.updateDefault();
        }
        this.$nextTick(function () {
            this.ignorefieldvaluechange = false;
        })
    }

    /**
      * 置空对象
      *
      * @param {*} _datas 置空对象
      * @memberof EditFormControlBase
      */
    public ResetData(_datas: any) {
        if (Object.keys(_datas).length > 0) {
            Object.keys(_datas).forEach((name: string) => {
                if (this.data.hasOwnProperty(name)) {
                    this.data[name] = null;
                }
            });
        }
    }

    /**
     * 重置
     *
     * @memberof EditFormControlBase
     */
    public onReset() {
        this.ResetData(this.data);
        this.$nextTick(() => {
            this.formState.next({ type: 'load', data: this.data });
        });
    }

    /**
     * 表单项检查逻辑
     *
     * @public
     * @param {string} name 属性名
     * @return {*}  {Promise<any>}
     * @memberof EditFormControlBase
     */
    public checkItem(name: string): Promise<any> {
        return new Promise((resolve, reject) => {
            if (!this.rules[name]) {
                resolve(true);
            }
            const validator = new schema({ [name]: this.rules[name] });
            validator.validate({ [name]: this.data[name] }).then(() => {
                resolve(true);
            })
                .catch(() => {
                    resolve(false);
                });;
        })
    }

    /**
     * 获取锚点项数据
     *
     * @public
     * @param {*} data
     * @memberof EditFormControlBase
     */
    public setAnchorItems(item: any, anchorArray: any[]) {
        if (Object.is(item.detailType, 'GROUPPANEL')) {
            const itemDetails: any[] = item.getPSDEFormDetails() || [];
            if (itemDetails.length > 0) {
                itemDetails.forEach((item1: IPSDEFormDetail, index: number) => {
                    this.setAnchorItems(item1, anchorArray);
                })
            }
        } else {
            if ((item as any).enableAnchor) {
                anchorArray.push({
                    name: item.name,
                    editor: (item as IPSDEFormItem).getPSEditor() || {}
                });
            }
        }
    }

    /**
     * 表单按钮行为触发
     *
     * @param {*} arg
     * @returns {void}
     * @memberof EditFormControlBase
     */
    public async onFormItemActionClick({ formdetail, event }: any) {
        if (formdetail && formdetail.actionType && Object.is(formdetail.actionType, 'FIUPDATE')) {
            const itemUpdate = formdetail.getPSDEFormItemUpdate();
            const showBusyIndicator = itemUpdate?.showBusyIndicator;
            const getPSAppDEMethod = itemUpdate?.getPSAppDEMethod();
            const getPSDEFIUpdateDetails = itemUpdate?.getPSDEFIUpdateDetails();
            let details: string[] = [];
            getPSDEFIUpdateDetails?.forEach((item: IPSDEFIUpdateDetail) => {
                details.push(item.name);
            })
            if (formdetail.getParamPickupPSAppView()) {
                const pickupview = formdetail.getParamPickupPSAppView();
                await pickupview.fill();
                if (!pickupview) {
                    this.updateFormItems(getPSAppDEMethod?.codeName as string, this.data, details, showBusyIndicator);
                } else {
                    const tempContext: any = Util.deepCopy(this.context);
                    const data: any = Util.deepCopy(this.viewparams);
                    if (formdetail?.getPSNavigateContexts() && (formdetail.getPSNavigateContexts().length > 0)) {
                        let _context: any = Util.computedNavData(this.data, tempContext, data, formdetail.getPSNavigateContexts());
                        Object.assign(tempContext, _context);
                    }
                    if (formdetail?.getPSNavigateParams() && (formdetail.getPSNavigateParams().length > 0)) {
                        let _param: any = Util.computedNavData(this.data, tempContext, data, formdetail.getPSNavigateParams());
                        Object.assign(data, _param);
                    }
                    if (pickupview.openMode.indexOf('DRAWER') !== -1) {
                        const view: any = {
                            viewname: 'app-view-shell',
                            height: pickupview.height,
                            width: pickupview.width,
                            title: pickupview.title,
                            placement: pickupview.openMode,
                        };
                        if (pickupview && pickupview.modelPath) {
                            Object.assign(tempContext, { viewpath: pickupview.modelPath });
                        }
                        const appdrawer = this.$appdrawer.openDrawer(view, Util.getViewProps(tempContext, data));
                        appdrawer.subscribe((result: any) => {
                            if (result && Object.is(result.ret, 'OK')) {
                                const arg: any = this.getData();
                                Object.assign(arg, { srfactionparam: result.datas });
                                this.updateFormItems(getPSAppDEMethod?.codeName as string, arg, details, showBusyIndicator);
                            }
                        });
                    } else {
                        const view: any = {
                            viewname: 'app-view-shell',
                            height: pickupview.height,
                            width: pickupview.width,
                            title: pickupview.title,
                        };
                        if (pickupview && pickupview.modelPath) {
                            Object.assign(tempContext, { viewpath: pickupview.modelPath });
                        }
                        const appmodal = this.$appmodal.openModal(view, tempContext, data);
                        appmodal.subscribe((result: any) => {
                            if (result && Object.is(result.ret, 'OK')) {
                                const arg: any = this.getData();
                                Object.assign(arg, { srfactionparam: result.datas });
                                this.updateFormItems(getPSAppDEMethod?.codeName as string, arg, details, showBusyIndicator);
                            }
                        });
                    }
                }
            } else {
                this.updateFormItems(getPSAppDEMethod?.codeName as string, this.data, details, showBusyIndicator);
            }
        } else {
            AppViewLogicService.getInstance().executeViewLogic(`${this.controlInstance.name}_${formdetail.name}_click`, event, this, null, this.controlInstance.getPSAppViewLogics());
        }
    }

    /**
     * 计算表单按钮权限状态
     *
     * @param {*} [data] 传入数据
     * @memberof EditFormControlBase
     */
    public computeButtonState(data: any) {
        let targetData: any = this.transformData(data);
        ViewTool.calcActionItemAuthState(targetData, this.actionModel, this.appUIService);
        if (this.detailsModel && Object.keys(this.detailsModel).length > 0) {
            Object.keys(this.detailsModel).forEach((name: any) => {
                const model = this.detailsModel[name];
                if (model?.detailType == "BUTTON" && model.uiaction?.tag) {
                    // 更新detailsModel里的按钮的权限状态值
                    if (Util.isEmpty(this.actionModel[model.uiaction.tag].dataActionResult)) {
                        this.detailsModel[name].isPower = true;
                        this.detailsModel[name].visible = true;
                        this.detailsModel[name].disabled = false;
                    } else {
                        this.detailsModel[name].visible = this.actionModel[model.uiaction.tag].visabled;
                        this.detailsModel[name].disabled = this.actionModel[model.uiaction.tag].disabled;
                        this.detailsModel[name].isPower = this.actionModel[model.uiaction.tag].dataActionResult === 1 ? true : false;
                    }
                } else if (model?.detailType == 'GROUPPANEL' && model.uiActionGroup?.details?.length > 0) {
                    // 更新分组面板界面行为组的权限状态值
                    model.uiActionGroup.details.forEach((actionDetail: any) => {
                        actionDetail.visible = this.actionModel[actionDetail.tag].visabled;
                        actionDetail.disabled = this.actionModel[actionDetail.tag].disabled;
                    })
                }
            })
        }
    }

    /**
     * 初始化界面行为模型
     *
     * @type {*}
     * @memberof EditFormControlBase
     */
    public initCtrlActionModel() {
        let allFormDetails: IPSDEFormDetail[] = ModelTool.getAllFormDetails(this.controlInstance);
        if (allFormDetails?.length > 0) {
            allFormDetails.forEach((detail: IPSDEFormDetail) => {
                if (detail?.detailType == 'BUTTON' && (detail as IPSDEFormButton).getPSUIAction()) {
                    // 添加表单按钮的actionModel
                    const appUIAction: IPSAppDEUIAction = ((detail as IPSDEFormButton).getPSUIAction() as IPSAppDEUIAction);
                    const appUIAction_M = Util.deepCopy(appUIAction.M);
                    this.actionModel[appUIAction.uIActionTag] = Object.assign(appUIAction_M, { disabled: false, visabled: true, getNoPrivDisplayMode: appUIAction.noPrivDisplayMode ? appUIAction.noPrivDisplayMode : 6 });
                } else if (detail?.detailType == 'GROUPPANEL' && (detail as IPSDEFormGroupPanel).getPSUIActionGroup()?.getPSUIActionGroupDetails()?.length) {
                    // 添加表单分组界面行为组的actionModel
                    (detail as IPSDEFormGroupPanel).getPSUIActionGroup()?.getPSUIActionGroupDetails()?.forEach((actionDetail: IPSUIActionGroupDetail) => {
                        if (actionDetail?.getPSUIAction()) {
                            const appUIAction: IPSAppDEUIAction = (actionDetail.getPSUIAction() as IPSAppDEUIAction);
                            const appUIAction_M = Util.deepCopy(appUIAction.M);
                            this.actionModel[appUIAction.uIActionTag] = Object.assign(appUIAction_M, { disabled: false, visabled: true, getNoPrivDisplayMode: appUIAction.noPrivDisplayMode ? appUIAction.noPrivDisplayMode : 6 });
                        }
                    })
                }
            })
        }
    }

    /**
     * 设置表单项错误提示信息
     * 
     * @param {*} prop 表单项字段名
     * @param {*} status 校验状态
     * @param {*} error 错误信息
     * @memberof EditFormControlBase
     */
    public formItemValidate(prop: string, status: boolean, error: string) {
        error = error ? error : '';
        const index = this.errorMessages.findIndex((errorMessage: any) => Object.is(errorMessage.prop, prop));
        if (status) {
            if (index != -1) {
                this.errorMessages.splice(index, 1);
            }
        } else {
            this.handleBottomLabelErrorMsgPosition(prop);
            if (index != -1) {
                this.errorMessages[index].error = error;
            } else {
                this.errorMessages.push({ prop: prop, error: error });
            }
        }
    }

    /**
     * 处理下方标签错误信息位置
     * 
     * @param {*} prop 表单项字段名
     * @memberof EditFormControlBase
     */
    public handleBottomLabelErrorMsgPosition(prop: any) {
        const element: any = document.querySelector(`.${this.appDeCodeName.toLowerCase()}-${this.controlInstance.codeName?.toLowerCase()}-item-${prop}`);
        if (element && element.className.indexOf('label-bottom') != -1) {
            this.$nextTick(() => {
                const labelWidth = element.querySelector('.app-form-item-label')?.offsetWidth;
                const errorDom = element.querySelector('.ivu-form-item-error-tip');
                if (labelWidth && errorDom) {
                    errorDom.style.left = labelWidth + 'px';
                }
            })
        }
    }

    /**
     * 显示更多模式切换操作
     *
     * @param {string} name 名称
     * @memberof EditFormControlBase
     */
    public manageContainerClick(name: string) {
        let model = this.detailsModel[name];
        if (model.isManageContainer) {
            model.setManageContainerStatus(!model.manageContainerStatus);
            model.controlledItems.forEach((item: any) => {
                if (this.detailsModel[item].isControlledContent) {
                    this.detailsModel[item].setVisible(model.manageContainerStatus ? this.detailsModel[item].oldVisible : false);
                }
            });
            this.$forceUpdate();
        }
    }

    /**
     * 打印
     *
     * @memberof EditFormControlBase
     */
    public print() {
        let _this: any = this;
        _this.$print({ id: `${this.appDeCodeName.toLowerCase()}_${this.controlInstance.codeName}`, popTitle: `${this.controlInstance.getPSAppDataEntity()?.logicName}` });
    }

    /**
     * 新建默认值
     *
     * @memberof EditFormControlBase
     */
    public createDefault() {
        const allFormDetails: IPSDEEditFormItem[] = ModelTool.getAllFormItems(this.controlInstance);
        if (allFormDetails.length > 0) {
            for (const detail of allFormDetails) {
                const property = detail?.codeName?.toLowerCase();
                if ((detail?.createDV || detail?.createDVT) && this.data.hasOwnProperty(property)) {
                    switch (detail.createDVT) {
                        case 'CONTEXT':
                            this.data[property] = this.viewparams[detail?.createDV];
                            break;
                        case 'SESSION':
                            this.data[property] = this.context[detail?.createDV];
                            break;
                        case 'APPDATA':
                            this.data[property] = this.context[detail?.createDV];
                            break;
                        case 'OPERATORNAME':
                            this.data[property] = this.context['srfusername'];
                            break;
                        case 'OPERATOR':
                            this.data[property] = this.context['srfuserid'];
                            break;
                        case 'CURTIME':
                            this.data[property] = Util.dateFormat(new Date(), detail.getPSAppDEField()?.valueFormat);
                            break;
                        case 'PARAM':
                            this.data[property] = this.service.getRemoteCopyData()?.[property] || null;
                            break;
                        default:
                            this.data[property] = ModelTool.isNumberField(detail?.getPSAppDEField()) ? Number(detail?.createDV) : detail?.createDV;
                            break;
                    }
                }
            }
        }
    }

    /**
     * 更新默认值
     *
     * @memberof EditFormControlBase
     */
    public updateDefault() {
        const allFormDetails: IPSDEEditFormItem[] = ModelTool.getAllFormItems(this.controlInstance);
        if (allFormDetails.length > 0) {
            for (const detail of allFormDetails) {
                const property = detail?.codeName?.toLowerCase();
                if ((detail?.updateDV || detail?.updateDVT) && this.data.hasOwnProperty(property) && (this.data[property] == null || this.data[property] == undefined)) {
                    switch (detail?.updateDVT) {
                        case 'CONTEXT':
                            this.data[property] = this.viewparams[detail?.updateDV];
                            break;
                        case 'SESSION':
                            this.data[property] = this.context[detail?.updateDV];
                            break;
                        case 'APPDATA':
                            this.data[property] = this.context[detail?.updateDV];
                            break;
                        case 'OPERATORNAME':
                            this.data[property] = this.context['srfusername'];
                            break;
                        case 'OPERATOR':
                            this.data[property] = this.context['srfuserid'];
                            break;
                        case 'CURTIME':
                            this.data[property] = Util.dateFormat(new Date(), detail.getPSAppDEField()?.valueFormat);
                            break;
                        case 'PARAM':
                            this.data[property] = this.service.getRemoteCopyData()?.[property] || null;
                            break;
                        case 'RESET':
                            this.data[property] = null;
                        default:
                            this.data[property] = ModelTool.isNumberField(detail?.getPSAppDEField()) ? Number(detail?.updateDV) : detail?.updateDV;
                            break;
                    }
                }
            }
        }
    }

    /**
     * 初始化值规则
     *
     * @memberof EditFormControlBase
     */
    public initRules() {
        // 先初始化系统值规则和属性值规则
        let staticRules: any = {};
        const allFormItemVRs = this.controlInstance.getPSDEFormItemVRs();
        allFormItemVRs?.forEach((item: IPSDEFormItemVR) => {
            const { checkMode, valueRuleType } = item;
            const formItemName = item.getPSDEFormItemName() || '';
            const sysRule = item.getPSSysValueRule();
            const deRule = item.getPSDEFValueRule();
            if (!staticRules[formItemName]) {
                staticRules[formItemName] = [];
            }
            // 排除后台检查的值规则
            if (checkMode == 2) {
                return
            }
            // 系统值规则
            if (valueRuleType == 'SYSVALUERULE' && sysRule) {
                // 正则值规则
                if (sysRule.ruleType == 'REG') {
                    staticRules[formItemName].push({
                        pattern: new RegExp(sysRule.regExCode),
                        message: sysRule.ruleInfo,
                        trigger: "change blur"
                    })
                    // 脚本值规则
                } else if (sysRule.ruleType == 'SCRIPT') {
                    staticRules[formItemName].push({
                        validator: (rule: any, value: any, callback: any) => {
                            // 空值时不校验
                            if (Util.isEmpty(this.data[formItemName])) {
                                return true
                            }
                            let source = this.data;
                            try {
                                eval(sysRule.scriptCode);
                            } catch (error) {
                                this.$throw(error, 'initRules');
                            }
                            return true;
                        },
                        trigger: "change blur"
                    })
                }
                // 属性值规则
            } else if (valueRuleType == 'DEFVALUERULE' && deRule) {
                // 有值项的情况，校验值项的值
                let formItem = this.controlInstance.findPSDEFormItem(formItemName);
                let valueName = formItem?.valueItemName || formItemName;
                staticRules[formItemName].push({
                    validator: (rule: any, value: any, callback: any, source: any) => {
                        // 空值时不校验
                        if (Util.isEmpty(this.data[valueName])) {
                            return true
                        }
                        const { isPast, infoMessage } = Verify.verifyDeRules(valueName, this.data, deRule.getPSDEFVRGroupCondition());
                        if (!isPast) {
                            callback(new Error(infoMessage || deRule.ruleInfo));
                        }
                        return true;
                    },
                    trigger: "change blur"
                })
            }
        })

        // 初始化非空值规则和数据类型值规则
        this.rules = {};
        const allFormItems = ModelTool.getAllFormItems(this.controlInstance) as IPSDEEditFormItem[];
        if (allFormItems && allFormItems.length > 0) {
            for (const detail of allFormItems) {
                if (detail.detailType == 'FORMITEM' && detail.getPSEditor()?.editorType != 'HIDDEN' && !detail.compositeItem) {
                    let type = ModelTool.isNumberField(detail?.getPSAppDEField()) ? 'number' : 'string';
                    let otherRules = staticRules[detail.name] || [];
                    let editorRules = Verify.buildVerConditions(detail.getPSEditor())
                    this.rules[detail.name] = [
                        // 非空值规则
                        { validator: (rule: any, value: any, callback: any) => { return !(this.detailsModel[detail.name].required && (value === null || value === undefined || value === "")) }, message: `${this.$t('app.formpage.valueverif') as string}${detail.caption}` },
                        // 表单值规则
                        ...otherRules,
                        // 编辑器基础值规则
                        ...editorRules
                    ]
                } else if (detail.detailType == 'FORMITEM' && detail.getPSEditor()?.editorType != 'HIDDEN' && detail.compositeItem) {
                    let compositeItemRules = this.getCompositeItemRules(detail);
                    this.rules[detail.name] = [
                        // 复合表单项基础值规则
                        ...compositeItemRules,
                    ]
                }
            }
        }
    }

    /**
     * 复合表单项值规则
     * 
     * @param detail 复合表单项
     */
    public getCompositeItemRules(detail: IPSDEEditFormItem) {
        let rules: Array<any> = [];
        if (detail.compositeItem && detail.getPSEditor()?.editorType == 'DATEPICKER') {
            const formItems = (detail as IPSDEFormItemEx).getPSDEFormItems();
            if (formItems && formItems.length > 1) {
                rules.push(
                    {
                        validator: (rule: any, value: any, callback: any) => {
                            return this.data[formItems[0].name] && this.data[formItems[1].name] && moment(this.data[formItems[0].name]).isAfter(this.data[formItems[1].name]) ? false : true;
                        },
                        message: this.$t('app.formpage.compositeitem.datepicker'),
                    }
                )
            }
        }
        return rules;
    }

    /**
     * 初始化表单成员模型
     *
     * @memberof EditFormControlBase
     */
    public initDetailsModel() {
        this.detailsModel = {};
        const { noTabHeader, name } = this.controlInstance;
        const allFormDetails: IPSDEFormDetail[] = ModelTool.getAllFormDetails(this.controlInstance);
        if (allFormDetails.length > 0) {
            for (const detail of allFormDetails) {
                let detailOpts: any = {
                    name: detail.name,
                    caption: detail.caption,
                    isShowCaption: detail.showCaption,
                    detailType: detail.detailType,
                    visible: !ModelTool.findGroupLogicByLogicCat('PANELVISIBLE', detail.getPSDEFDGroupLogics()),
                    form: this,
                    isControlledContent: detail.showMoreMode == 1,
                };
                let detailModel: any = null;
                switch (detail.detailType) {
                    case 'BUTTON':
                        Object.assign(detailOpts, {
                            disabled: false,
                        });
                        const uiAction = (detail as IPSDEFormButton).getPSUIAction?.();
                        if (uiAction) {
                            detailOpts.uiaction = {
                                type: uiAction.uIActionType,
                                tag: uiAction.uIActionTag,
                                visabled: true,
                                disabled: false,
                            }
                            if (uiAction.actionTarget) {
                                detailOpts.uiaction.actiontarget = uiAction.actionTarget;
                            }
                            if ((uiAction as IPSAppDEUIAction).noPrivDisplayMode) {
                                detailOpts.uiaction.noprivdisplaymode = (uiAction as IPSAppDEUIAction).noPrivDisplayMode;
                            }
                            if (uiAction.dataAccessAction) {
                                detailOpts.uiaction.dataaccaction = uiAction.dataAccessAction;
                                if (this.Environment.enablePermissionValid) {
                                    detailOpts.visible = false;
                                }
                            }
                        }
                        detailModel = new FormButtonModel(detailOpts);
                        break;
                    case 'FORMITEM':
                        Object.assign(detailOpts, {
                            disabled: false,
                            required: !(detail as IPSDEFormItem).allowEmpty,
                            enableCond: (detail as IPSDEFormItem).enableCond,
                            ignoreInput: (detail as IPSDEFormItem).ignoreInput,
                            captionItemName: (detail as IPSDEFormItem).captionItemName,
                            convertToCodeItemText: (detail as IPSDEFormItem).convertToCodeItemText ? true : false,
                        });
                        if ((detail as IPSDEFormItem).getPSEditor() && ((detail as IPSDEFormItem).getPSEditor() as IPSCodeListEditor).getPSAppCodeList instanceof Function) {
                            Object.assign(detailOpts, {
                                codelist: ((detail as IPSDEFormItem).getPSEditor() as IPSCodeListEditor).getPSAppCodeList()
                            });
                        }
                        detailModel = new FormItemModel(detailOpts);
                        break;
                    case 'GROUPPANEL':
                        detailOpts.isManageContainer = detail.showMoreMode == 2;
                        const PSUIActionGroup = (detail as IPSDEFormGroupPanel).getPSUIActionGroup();
                        // 界面行为组
                        let uiActionGroup: any = {
                            caption: PSUIActionGroup?.name,
                            langbase: '',
                            extractMode: (detail as IPSDEFormGroupPanel).actionGroupExtractMode || 'ITEM',
                            details: [],
                        }
                        const PSUIActionGroupDetails = PSUIActionGroup?.getPSUIActionGroupDetails?.() || [];
                        if (PSUIActionGroupDetails.length > 0) {
                            PSUIActionGroupDetails.forEach((actionDetail: IPSUIActionGroupDetail) => {
                                let uiaction = actionDetail?.getPSUIAction?.() as IPSAppDEUIAction;
                                let temp: any = {
                                    name: `${detail.name}_${actionDetail.name}`,
                                    tag: uiaction?.uIActionTag,
                                    caption: this.$tl(uiaction?.getCapPSLanguageRes()?.lanResTag, uiaction?.caption || ''),
                                    disabled: false,
                                    visabled: true,
                                    visible: true,
                                    noprivdisplaymode: uiaction?.noPrivDisplayMode,
                                    actiontarget: uiaction?.actionTarget || '',
                                    dataaccaction: uiaction?.dataAccessAction || '',
                                    isShowCaption: actionDetail.showCaption,
                                    isShowIcon: actionDetail.showIcon,
                                }
                                if (uiaction?.getPSAppDataEntity?.()) {
                                    temp.uiactiontag = `${uiaction?.getPSAppDataEntity?.()?.codeName?.toLowerCase()}_${uiaction?.uIActionTag?.toLowerCase()}`
                                }
                                // 图标
                                if (uiaction?.getPSSysImage?.()) {
                                    let image = uiaction.getPSSysImage();
                                    if (image?.cssClass) {
                                        temp.icon = image.cssClass;
                                    } else {
                                        temp.img = image?.imagePath;
                                    }
                                }
                                uiActionGroup.details.push(temp);
                            })
                        }
                        detailOpts.uiActionGroup = uiActionGroup;

                        // 受控容器的成员
                        let showMoreModeItems: any[] = [];
                        //  支持锚点的成员
                        let anchorPoints: any[] = [];
                        // 分组面板支持锚点
                        if ((detail as IPSDEFormGroupPanel)?.enableAnchor) {
                            this.groupAnchorDatas.push({
                                caption: detail.caption,
                                codeName: detail.codeName
                            });
                        };
                        (detail as IPSDEFormGroupPanel).getPSDEFormDetails()?.forEach((item: IPSDEFormDetail, index: number) => {
                            if (!item) return;
                            if (item.showMoreMode == 1) {
                                showMoreModeItems.push(item.name);
                            }
                            this.setAnchorItems(item, anchorPoints);
                            const showMore = item.getShowMoreMgrPSDEFormDetail?.();
                            if (showMore && showMore.id && Object.is(showMore.id, detail.name)) {
                                detailOpts.isManageContainer = true;
                            }
                        })
                        detailOpts.controlledItems = showMoreModeItems;
                        detailOpts.anchorPoints = anchorPoints;
                        detailModel = new FormGroupPanelModel(detailOpts);
                        break;
                    case 'TABPANEL':
                        // 添加tab分页
                        let tabPages: any[] = [];
                        (detail as IPSDEFormTabPanel).getPSDEFormTabPages()?.forEach((item: IPSDEFormTabPage, index: number) => {
                            tabPages.push({
                                name: item.name,
                                index: index,
                                visible: !ModelTool.findGroupLogicByLogicCat('PANELVISIBLE', detail.getPSDEFDGroupLogics())
                            })
                        })
                        Object.assign(detailOpts, {
                            tabPages: tabPages,
                        });
                        detailModel = new FormTabPanelModel(detailOpts);
                        break;
                    case 'TABPAGE':
                        detailModel = new FormTabPageModel(detailOpts);
                        break;
                    case 'FORMPAGE':
                        detailModel = new FormPageModel(detailOpts);
                        break;
                    case 'FORMPART':
                        detailModel = new FormPartModel(detailOpts);
                        break;
                    case 'DRUIPART':
                        this.drCount++;
                        detailModel = new FormDruipartModel(detailOpts);
                        break;
                    case 'IFRAME':
                        detailModel = new FormIFrameModel(detailOpts);
                        break;
                    case 'RAWITEM':
                        detailModel = new FormRawItemModel(detailOpts);
                        break;
                    case 'USERCONTROL':
                        detailModel = new FormUserControlModel(detailOpts);
                        break;
                }
                this.$set(this.detailsModel, detail.name, detailModel)
            }
        }
        // 有分页头表格时
        if (!noTabHeader) {
            let formPages: any[] = [];
            this.controlInstance.getPSDEFormPages()?.forEach((item: IPSDEFormPage, index: number) => {
                formPages.push({
                    name: item.name,
                    index: index,
                    visible: !ModelTool.findGroupLogicByLogicCat('PANELVISIBLE', item.getPSDEFDGroupLogics())
                })
            })
            this.$set(this.detailsModel, name, new FormTabPanelModel({
                caption: name,
                detailType: 'TABPANEL',
                name: name,
                visible: true,
                isShowCaption: true,
                form: this,
                tabPages: formPages,
            }))
        }
    }

    /**
     * 重置表单项值
     *
     * @public
     * @param {{ name: string }} { name } 名称
     * @memberof AppDefaultForm
     */
    public resetFormData({ name }: { name: string }): void {
        const formItems: IPSDEEditFormItem[] = ModelTool.getAllFormItems(this.controlInstance);
        if (formItems && formItems.length > 0) {
            for (const item of formItems) {
                if (item.resetItemName && item.resetItemName == name) {
                    this.onFormItemValueChange({ name: item.name, value: null });
                    if (item.valueItemName) {
                        this.onFormItemValueChange({ name: item.valueItemName, value: null });
                    }
                }
            }
        }
    }

    /**
     * 表单逻辑
     * 
     * @public
     * @param {{ name: string }} { name } 名称
     * @memberof FormControlBase
     */
    public async formLogic({ name }: { name: string }) {
        const allFormDetails: IPSDEFormDetail[] = ModelTool.getAllFormDetails(this.controlInstance);
        // 表单动态逻辑
        allFormDetails?.forEach((detail: IPSDEFormDetail) => {
            detail.getPSDEFDGroupLogics()?.forEach((logic: IPSDEFDCatGroupLogic) => {
                // todo lxm 缺少getRelatedDetailNames
                let relatedNames = logic.getRelatedDetailNames() || [];
                if (Object.is(name, '') || relatedNames.indexOf(name) != -1) {
                    let ret = this.verifyGroupLogic(this.data, logic);
                    switch (logic.logicCat) {
                        // 动态空输入，不满足则必填
                        case 'ITEMBLANK':
                            this.detailsModel[detail.name].required = !ret;
                            break;
                        // 动态启用，满足则启用
                        case 'ITEMENABLE':
                            this.detailsModel[detail.name].setDisabled(!ret);
                            break;
                        // 动态显示，满足则显示
                        case 'PANELVISIBLE':
                            this.detailsModel[detail.name].setVisible(ret);
                            break;
                    }
                }
            })
            if (Object.is(detail.detailType, 'FORMITEM')) {
                const captionItemName = (detail as IPSDEFormItem).captionItemName;
                if (captionItemName) {
                    // 动态标题值项
                    this.detailsModel[detail.name].caption = captionItemName ? this.data[captionItemName] : this.detailsModel[detail.name].caption;
                    // 值规则错误提示更新
                    if (this.rules[detail.name]) {
                        this.rules[detail.name].forEach((rule: any) => {
                            rule.message = `${this.$t('app.formpage.valueverif') as string}${this.data[captionItemName]}`;
                        })
                    }
                    this.$forceUpdate();
                }
            }
        })

        // 表单项更新
        let formDetail: IPSDEFormItem = ModelTool.getFormDetailByName(this.controlInstance, name);
        const formItemUpdate: IPSDEFormItemUpdate | null = formDetail?.getPSDEFormItemUpdate?.();
        if (formItemUpdate) {
            if (await this.checkItem(formDetail.name)) {
                if (formItemUpdate.customCode) {
                    if (formItemUpdate.scriptCode) {
                        const context = Util.deepCopy(this.context);
                        const viewparams = Util.deepCopy(this.viewparams);
                        let data = this.data;
                        eval(formItemUpdate.scriptCode);
                    }
                } else {
                    const showBusyIndicator = formItemUpdate.showBusyIndicator;
                    const getPSAppDEMethod = formItemUpdate.getPSAppDEMethod();
                    const getPSDEFIUpdateDetails = formItemUpdate.getPSDEFIUpdateDetails();
                    let details: string[] = [];
                    getPSDEFIUpdateDetails?.forEach((item: IPSDEFIUpdateDetail) => {
                        details.push(item.name)
                    })
                    this.updateFormItems(getPSAppDEMethod?.codeName as string, this.data, details, showBusyIndicator);
                }
            }
        }
    }

    /**
     * 校验动态逻辑结果
     *
     * @param {*} data 数据对象
     * @param {*} logic 逻辑对象
     * @returns
     * @memberof EditFormControlBase
     */
    public verifyGroupLogic(data: any, logic: IPSDEFDCatGroupLogic): boolean {
        if (logic.logicType == 'GROUP' && logic?.getPSDEFDLogics()?.length) {
            let result: boolean = true;
            if (logic.groupOP == 'AND') {
                let falseItem: IPSDEFDLogic | undefined = logic.getPSDEFDLogics()?.find((childLogic: any) => {
                    return !this.verifyGroupLogic(data, childLogic);
                })
                result = falseItem == undefined;
            } else if (logic.groupOP == 'OR') {
                let trueItem: IPSDEFDLogic | undefined = logic.getPSDEFDLogics()?.find((childLogic: any) => {
                    return this.verifyGroupLogic(data, childLogic);
                })
                result = trueItem != undefined;
            }
            // 是否取反
            return logic.notMode ? !result : result;
        } else if (logic.logicType == 'SINGLE') {
            let singleLogic: IPSDEFDSingleLogic = logic as any;
            return Verify.testCond(data[singleLogic.dEFDName.toLowerCase()], singleLogic.condOP, singleLogic.value)
        }
        return false;
    }


    /**
     * 处理操作列点击
     * 
     * @param {*} event 事件对象
     * @param {*} formDetail 表单成员模型对象
     * @param {*} actionDetal 界面行为模型对象
     * @memberof EditFormControlBase
     */
    public handleActionClick(event: any, formDetail: any, actionDetal: any) {
        AppViewLogicService.getInstance().executeViewLogic(this.getViewLogicTag(this.controlInstance.name, formDetail.codeName, actionDetal.name), event, this, undefined, this.controlInstance.getPSAppViewLogics() || []);
    }
}