package cn.ibizlab.eam.webapi.rest;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.math.BigInteger;
import java.util.HashMap;
import lombok.extern.slf4j.Slf4j;
import com.alibaba.fastjson.JSONObject;
import javax.servlet.ServletRequest;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cglib.beans.BeanCopier;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import org.springframework.http.HttpStatus;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.util.StringUtils;
import org.springframework.context.annotation.Lazy;
import org.springframework.security.access.prepost.PreAuthorize;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import cn.ibizlab.eam.webapi.dto.*;
import cn.ibizlab.eam.webapi.mapping.*;
import cn.ibizlab.eam.core.eam_core.domain.EMBrand;
import cn.ibizlab.eam.core.eam_core.service.IEMBrandService;
import cn.ibizlab.eam.core.eam_core.filter.EMBrandSearchContext;

@Slf4j
@Api(tags = {"EMBrand" })
@RestController("WebApi-embrand")
@RequestMapping("")
public class EMBrandResource {

    @Autowired
    private IEMBrandService embrandService;

    @Autowired
    @Lazy
    public EMBrandMapping embrandMapping;

    public EMBrandDTO permissionDTO=new EMBrandDTO();

    @PreAuthorize("hasPermission('','Create',{'Sql',this.embrandMapping,#embranddto})")
    @ApiOperation(value = "Create", tags = {"EMBrand" },  notes = "Create")
	@RequestMapping(method = RequestMethod.POST, value = "/embrands")
    @Transactional
    public ResponseEntity<EMBrandDTO> create(@RequestBody EMBrandDTO embranddto) {
        EMBrand domain = embrandMapping.toDomain(embranddto);
		embrandService.create(domain);
        EMBrandDTO dto = embrandMapping.toDto(domain);
		return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasPermission('Create',{'Sql',this.embrandMapping,#embranddtos})")
    @ApiOperation(value = "createBatch", tags = {"EMBrand" },  notes = "createBatch")
	@RequestMapping(method = RequestMethod.POST, value = "/embrands/batch")
    public ResponseEntity<Boolean> createBatch(@RequestBody List<EMBrandDTO> embranddtos) {
        embrandService.createBatch(embrandMapping.toDomain(embranddtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasPermission(#embrand_id,'Get',{'Sql',this.embrandMapping,this.permissionDTO})")
    @ApiOperation(value = "Get", tags = {"EMBrand" },  notes = "Get")
	@RequestMapping(method = RequestMethod.GET, value = "/embrands/{embrand_id}")
    public ResponseEntity<EMBrandDTO> get(@PathVariable("embrand_id") String embrand_id) {
        EMBrand domain = embrandService.get(embrand_id);
        EMBrandDTO dto = embrandMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasPermission('','Save',{'Sql',this.embrandMapping,#embranddto})")
    @ApiOperation(value = "Save", tags = {"EMBrand" },  notes = "Save")
	@RequestMapping(method = RequestMethod.POST, value = "/embrands/save")
    public ResponseEntity<Boolean> save(@RequestBody EMBrandDTO embranddto) {
        return ResponseEntity.status(HttpStatus.OK).body(embrandService.save(embrandMapping.toDomain(embranddto)));
    }

    @PreAuthorize("hasPermission('Save',{'Sql',this.embrandMapping,#embranddtos})")
    @ApiOperation(value = "SaveBatch", tags = {"EMBrand" },  notes = "SaveBatch")
	@RequestMapping(method = RequestMethod.POST, value = "/embrands/savebatch")
    public ResponseEntity<Boolean> saveBatch(@RequestBody List<EMBrandDTO> embranddtos) {
        embrandService.saveBatch(embrandMapping.toDomain(embranddtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','eam_practice01-EMBrand-GetDraft-all')")
    @ApiOperation(value = "GetDraft", tags = {"EMBrand" },  notes = "GetDraft")
	@RequestMapping(method = RequestMethod.GET, value = "/embrands/getdraft")
    public ResponseEntity<EMBrandDTO> getDraft() {
        return ResponseEntity.status(HttpStatus.OK).body(embrandMapping.toDto(embrandService.getDraft(new EMBrand())));
    }

    @PreAuthorize("hasPermission(#embrand_id,'Update',{'Sql',this.embrandMapping,#embranddto})")
    @ApiOperation(value = "Update", tags = {"EMBrand" },  notes = "Update")
	@RequestMapping(method = RequestMethod.PUT, value = "/embrands/{embrand_id}")
    @Transactional
    public ResponseEntity<EMBrandDTO> update(@PathVariable("embrand_id") String embrand_id, @RequestBody EMBrandDTO embranddto) {
		EMBrand domain  = embrandMapping.toDomain(embranddto);
        domain .setEmbrandid(embrand_id);
		embrandService.update(domain );
		EMBrandDTO dto = embrandMapping.toDto(domain );
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasPermission('Update',{'Sql',this.embrandMapping,#embranddtos})")
    @ApiOperation(value = "UpdateBatch", tags = {"EMBrand" },  notes = "UpdateBatch")
	@RequestMapping(method = RequestMethod.PUT, value = "/embrands/batch")
    public ResponseEntity<Boolean> updateBatch(@RequestBody List<EMBrandDTO> embranddtos) {
        embrandService.updateBatch(embrandMapping.toDomain(embranddtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasPermission(#embrand_id,'Remove',{'Sql',this.embrandMapping,this.permissionDTO})")
    @ApiOperation(value = "Remove", tags = {"EMBrand" },  notes = "Remove")
	@RequestMapping(method = RequestMethod.DELETE, value = "/embrands/{embrand_id}")
    @Transactional
    public ResponseEntity<Boolean> remove(@PathVariable("embrand_id") String embrand_id) {
         return ResponseEntity.status(HttpStatus.OK).body(embrandService.remove(embrand_id));
    }

    @PreAuthorize("hasPermission('Remove',{'Sql',this.embrandMapping,this.permissionDTO,#ids})")
    @ApiOperation(value = "RemoveBatch", tags = {"EMBrand" },  notes = "RemoveBatch")
	@RequestMapping(method = RequestMethod.DELETE, value = "/embrands/batch")
    public ResponseEntity<Boolean> removeBatch(@RequestBody List<String> ids) {
        embrandService.removeBatch(ids);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','eam_practice01-EMBrand-CheckKey-all')")
    @ApiOperation(value = "CheckKey", tags = {"EMBrand" },  notes = "CheckKey")
	@RequestMapping(method = RequestMethod.POST, value = "/embrands/checkkey")
    public ResponseEntity<Boolean> checkKey(@RequestBody EMBrandDTO embranddto) {
        return  ResponseEntity.status(HttpStatus.OK).body(embrandService.checkKey(embrandMapping.toDomain(embranddto)));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','eam_practice01-EMBrand-Default-all')")
	@ApiOperation(value = "fetchDEFAULT", tags = {"EMBrand" } ,notes = "fetchDEFAULT")
    @RequestMapping(method= RequestMethod.GET , value="/embrands/fetchdefault")
	public ResponseEntity<List<EMBrandDTO>> fetchDefault(EMBrandSearchContext context) {
        Page<EMBrand> domains = embrandService.searchDefault(context) ;
        List<EMBrandDTO> list = embrandMapping.toDto(domains.getContent());
        return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(list);
	}

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','eam_practice01-EMBrand-Default-all')")
	@ApiOperation(value = "searchDEFAULT", tags = {"EMBrand" } ,notes = "searchDEFAULT")
    @RequestMapping(method= RequestMethod.POST , value="/embrands/searchdefault")
	public ResponseEntity<Page<EMBrandDTO>> searchDefault(@RequestBody EMBrandSearchContext context) {
        Page<EMBrand> domains = embrandService.searchDefault(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(embrandMapping.toDto(domains.getContent()), context.getPageable(), domains.getTotalElements()));
	}
}
