package cn.ibizlab.eam.webapi.rest;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.math.BigInteger;
import java.util.HashMap;
import lombok.extern.slf4j.Slf4j;
import com.alibaba.fastjson.JSONObject;
import javax.servlet.ServletRequest;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cglib.beans.BeanCopier;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import org.springframework.http.HttpStatus;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.util.StringUtils;
import org.springframework.context.annotation.Lazy;
import org.springframework.security.access.prepost.PreAuthorize;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import cn.ibizlab.eam.webapi.dto.*;
import cn.ibizlab.eam.webapi.mapping.*;
import cn.ibizlab.eam.core.eam_core.domain.EMEICam;
import cn.ibizlab.eam.core.eam_core.service.IEMEICamService;
import cn.ibizlab.eam.core.eam_core.filter.EMEICamSearchContext;

@Slf4j
@Api(tags = {"EMEICam" })
@RestController("WebApi-emeicam")
@RequestMapping("")
public class EMEICamResource {

    @Autowired
    private IEMEICamService emeicamService;

    @Autowired
    @Lazy
    public EMEICamMapping emeicamMapping;

    public EMEICamDTO permissionDTO=new EMEICamDTO();

    @PreAuthorize("hasPermission(#emeicam_id,'Get',{'Sql',this.emeicamMapping,this.permissionDTO})")
    @ApiOperation(value = "Get", tags = {"EMEICam" },  notes = "Get")
	@RequestMapping(method = RequestMethod.GET, value = "/emeicams/{emeicam_id}")
    public ResponseEntity<EMEICamDTO> get(@PathVariable("emeicam_id") String emeicam_id) {
        EMEICam domain = emeicamService.get(emeicam_id);
        EMEICamDTO dto = emeicamMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','eam_practice01-EMEICam-CheckKey-all')")
    @ApiOperation(value = "CheckKey", tags = {"EMEICam" },  notes = "CheckKey")
	@RequestMapping(method = RequestMethod.POST, value = "/emeicams/checkkey")
    public ResponseEntity<Boolean> checkKey(@RequestBody EMEICamDTO emeicamdto) {
        return  ResponseEntity.status(HttpStatus.OK).body(emeicamService.checkKey(emeicamMapping.toDomain(emeicamdto)));
    }

    @PreAuthorize("hasPermission('','Save',{'Sql',this.emeicamMapping,#emeicamdto})")
    @ApiOperation(value = "Save", tags = {"EMEICam" },  notes = "Save")
	@RequestMapping(method = RequestMethod.POST, value = "/emeicams/save")
    public ResponseEntity<Boolean> save(@RequestBody EMEICamDTO emeicamdto) {
        return ResponseEntity.status(HttpStatus.OK).body(emeicamService.save(emeicamMapping.toDomain(emeicamdto)));
    }

    @PreAuthorize("hasPermission('Save',{'Sql',this.emeicamMapping,#emeicamdtos})")
    @ApiOperation(value = "SaveBatch", tags = {"EMEICam" },  notes = "SaveBatch")
	@RequestMapping(method = RequestMethod.POST, value = "/emeicams/savebatch")
    public ResponseEntity<Boolean> saveBatch(@RequestBody List<EMEICamDTO> emeicamdtos) {
        emeicamService.saveBatch(emeicamMapping.toDomain(emeicamdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','eam_practice01-EMEICam-GetDraft-all')")
    @ApiOperation(value = "GetDraft", tags = {"EMEICam" },  notes = "GetDraft")
	@RequestMapping(method = RequestMethod.GET, value = "/emeicams/getdraft")
    public ResponseEntity<EMEICamDTO> getDraft() {
        return ResponseEntity.status(HttpStatus.OK).body(emeicamMapping.toDto(emeicamService.getDraft(new EMEICam())));
    }

    @PreAuthorize("hasPermission(#emeicam_id,'Remove',{'Sql',this.emeicamMapping,this.permissionDTO})")
    @ApiOperation(value = "Remove", tags = {"EMEICam" },  notes = "Remove")
	@RequestMapping(method = RequestMethod.DELETE, value = "/emeicams/{emeicam_id}")
    @Transactional
    public ResponseEntity<Boolean> remove(@PathVariable("emeicam_id") String emeicam_id) {
         return ResponseEntity.status(HttpStatus.OK).body(emeicamService.remove(emeicam_id));
    }

    @PreAuthorize("hasPermission('Remove',{'Sql',this.emeicamMapping,this.permissionDTO,#ids})")
    @ApiOperation(value = "RemoveBatch", tags = {"EMEICam" },  notes = "RemoveBatch")
	@RequestMapping(method = RequestMethod.DELETE, value = "/emeicams/batch")
    public ResponseEntity<Boolean> removeBatch(@RequestBody List<String> ids) {
        emeicamService.removeBatch(ids);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasPermission(#emeicam_id,'Update',{'Sql',this.emeicamMapping,#emeicamdto})")
    @ApiOperation(value = "Update", tags = {"EMEICam" },  notes = "Update")
	@RequestMapping(method = RequestMethod.PUT, value = "/emeicams/{emeicam_id}")
    @Transactional
    public ResponseEntity<EMEICamDTO> update(@PathVariable("emeicam_id") String emeicam_id, @RequestBody EMEICamDTO emeicamdto) {
		EMEICam domain  = emeicamMapping.toDomain(emeicamdto);
        domain .setEmeicamid(emeicam_id);
		emeicamService.update(domain );
		EMEICamDTO dto = emeicamMapping.toDto(domain );
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasPermission('Update',{'Sql',this.emeicamMapping,#emeicamdtos})")
    @ApiOperation(value = "UpdateBatch", tags = {"EMEICam" },  notes = "UpdateBatch")
	@RequestMapping(method = RequestMethod.PUT, value = "/emeicams/batch")
    public ResponseEntity<Boolean> updateBatch(@RequestBody List<EMEICamDTO> emeicamdtos) {
        emeicamService.updateBatch(emeicamMapping.toDomain(emeicamdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasPermission('','Create',{'Sql',this.emeicamMapping,#emeicamdto})")
    @ApiOperation(value = "Create", tags = {"EMEICam" },  notes = "Create")
	@RequestMapping(method = RequestMethod.POST, value = "/emeicams")
    @Transactional
    public ResponseEntity<EMEICamDTO> create(@RequestBody EMEICamDTO emeicamdto) {
        EMEICam domain = emeicamMapping.toDomain(emeicamdto);
		emeicamService.create(domain);
        EMEICamDTO dto = emeicamMapping.toDto(domain);
		return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasPermission('Create',{'Sql',this.emeicamMapping,#emeicamdtos})")
    @ApiOperation(value = "createBatch", tags = {"EMEICam" },  notes = "createBatch")
	@RequestMapping(method = RequestMethod.POST, value = "/emeicams/batch")
    public ResponseEntity<Boolean> createBatch(@RequestBody List<EMEICamDTO> emeicamdtos) {
        emeicamService.createBatch(emeicamMapping.toDomain(emeicamdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','eam_practice01-EMEICam-Default-all')")
	@ApiOperation(value = "fetchDEFAULT", tags = {"EMEICam" } ,notes = "fetchDEFAULT")
    @RequestMapping(method= RequestMethod.GET , value="/emeicams/fetchdefault")
	public ResponseEntity<List<EMEICamDTO>> fetchDefault(EMEICamSearchContext context) {
        Page<EMEICam> domains = emeicamService.searchDefault(context) ;
        List<EMEICamDTO> list = emeicamMapping.toDto(domains.getContent());
        return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(list);
	}

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','eam_practice01-EMEICam-Default-all')")
	@ApiOperation(value = "searchDEFAULT", tags = {"EMEICam" } ,notes = "searchDEFAULT")
    @RequestMapping(method= RequestMethod.POST , value="/emeicams/searchdefault")
	public ResponseEntity<Page<EMEICamDTO>> searchDefault(@RequestBody EMEICamSearchContext context) {
        Page<EMEICam> domains = emeicamService.searchDefault(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(emeicamMapping.toDto(domains.getContent()), context.getPageable(), domains.getTotalElements()));
	}
}
