package cn.ibizlab.eam.webapi.rest;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.math.BigInteger;
import java.util.HashMap;
import lombok.extern.slf4j.Slf4j;
import com.alibaba.fastjson.JSONObject;
import javax.servlet.ServletRequest;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cglib.beans.BeanCopier;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import org.springframework.http.HttpStatus;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.util.StringUtils;
import org.springframework.context.annotation.Lazy;
import org.springframework.security.access.prepost.PreAuthorize;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import cn.ibizlab.eam.webapi.dto.*;
import cn.ibizlab.eam.webapi.mapping.*;
import cn.ibizlab.eam.core.eam_core.domain.EMStore;
import cn.ibizlab.eam.core.eam_core.service.IEMStoreService;
import cn.ibizlab.eam.core.eam_core.filter.EMStoreSearchContext;

@Slf4j
@Api(tags = {"EMStore" })
@RestController("WebApi-emstore")
@RequestMapping("")
public class EMStoreResource {

    @Autowired
    private IEMStoreService emstoreService;

    @Autowired
    @Lazy
    public EMStoreMapping emstoreMapping;

    public EMStoreDTO permissionDTO=new EMStoreDTO();

    @PreAuthorize("hasPermission(#emstore_id,'Update',{'Sql',this.emstoreMapping,#emstoredto})")
    @ApiOperation(value = "Update", tags = {"EMStore" },  notes = "Update")
	@RequestMapping(method = RequestMethod.PUT, value = "/emstores/{emstore_id}")
    @Transactional
    public ResponseEntity<EMStoreDTO> update(@PathVariable("emstore_id") String emstore_id, @RequestBody EMStoreDTO emstoredto) {
		EMStore domain  = emstoreMapping.toDomain(emstoredto);
        domain .setEmstoreid(emstore_id);
		emstoreService.update(domain );
		EMStoreDTO dto = emstoreMapping.toDto(domain );
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasPermission('Update',{'Sql',this.emstoreMapping,#emstoredtos})")
    @ApiOperation(value = "UpdateBatch", tags = {"EMStore" },  notes = "UpdateBatch")
	@RequestMapping(method = RequestMethod.PUT, value = "/emstores/batch")
    public ResponseEntity<Boolean> updateBatch(@RequestBody List<EMStoreDTO> emstoredtos) {
        emstoreService.updateBatch(emstoreMapping.toDomain(emstoredtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','eam_practice01-EMStore-GetDraft-all')")
    @ApiOperation(value = "GetDraft", tags = {"EMStore" },  notes = "GetDraft")
	@RequestMapping(method = RequestMethod.GET, value = "/emstores/getdraft")
    public ResponseEntity<EMStoreDTO> getDraft() {
        return ResponseEntity.status(HttpStatus.OK).body(emstoreMapping.toDto(emstoreService.getDraft(new EMStore())));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','eam_practice01-EMStore-CheckKey-all')")
    @ApiOperation(value = "CheckKey", tags = {"EMStore" },  notes = "CheckKey")
	@RequestMapping(method = RequestMethod.POST, value = "/emstores/checkkey")
    public ResponseEntity<Boolean> checkKey(@RequestBody EMStoreDTO emstoredto) {
        return  ResponseEntity.status(HttpStatus.OK).body(emstoreService.checkKey(emstoreMapping.toDomain(emstoredto)));
    }

    @PreAuthorize("hasPermission(#emstore_id,'Get',{'Sql',this.emstoreMapping,this.permissionDTO})")
    @ApiOperation(value = "Get", tags = {"EMStore" },  notes = "Get")
	@RequestMapping(method = RequestMethod.GET, value = "/emstores/{emstore_id}")
    public ResponseEntity<EMStoreDTO> get(@PathVariable("emstore_id") String emstore_id) {
        EMStore domain = emstoreService.get(emstore_id);
        EMStoreDTO dto = emstoreMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasPermission('','Create',{'Sql',this.emstoreMapping,#emstoredto})")
    @ApiOperation(value = "Create", tags = {"EMStore" },  notes = "Create")
	@RequestMapping(method = RequestMethod.POST, value = "/emstores")
    @Transactional
    public ResponseEntity<EMStoreDTO> create(@RequestBody EMStoreDTO emstoredto) {
        EMStore domain = emstoreMapping.toDomain(emstoredto);
		emstoreService.create(domain);
        EMStoreDTO dto = emstoreMapping.toDto(domain);
		return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasPermission('Create',{'Sql',this.emstoreMapping,#emstoredtos})")
    @ApiOperation(value = "createBatch", tags = {"EMStore" },  notes = "createBatch")
	@RequestMapping(method = RequestMethod.POST, value = "/emstores/batch")
    public ResponseEntity<Boolean> createBatch(@RequestBody List<EMStoreDTO> emstoredtos) {
        emstoreService.createBatch(emstoreMapping.toDomain(emstoredtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasPermission('','Save',{'Sql',this.emstoreMapping,#emstoredto})")
    @ApiOperation(value = "Save", tags = {"EMStore" },  notes = "Save")
	@RequestMapping(method = RequestMethod.POST, value = "/emstores/save")
    public ResponseEntity<Boolean> save(@RequestBody EMStoreDTO emstoredto) {
        return ResponseEntity.status(HttpStatus.OK).body(emstoreService.save(emstoreMapping.toDomain(emstoredto)));
    }

    @PreAuthorize("hasPermission('Save',{'Sql',this.emstoreMapping,#emstoredtos})")
    @ApiOperation(value = "SaveBatch", tags = {"EMStore" },  notes = "SaveBatch")
	@RequestMapping(method = RequestMethod.POST, value = "/emstores/savebatch")
    public ResponseEntity<Boolean> saveBatch(@RequestBody List<EMStoreDTO> emstoredtos) {
        emstoreService.saveBatch(emstoreMapping.toDomain(emstoredtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasPermission(#emstore_id,'Remove',{'Sql',this.emstoreMapping,this.permissionDTO})")
    @ApiOperation(value = "Remove", tags = {"EMStore" },  notes = "Remove")
	@RequestMapping(method = RequestMethod.DELETE, value = "/emstores/{emstore_id}")
    @Transactional
    public ResponseEntity<Boolean> remove(@PathVariable("emstore_id") String emstore_id) {
         return ResponseEntity.status(HttpStatus.OK).body(emstoreService.remove(emstore_id));
    }

    @PreAuthorize("hasPermission('Remove',{'Sql',this.emstoreMapping,this.permissionDTO,#ids})")
    @ApiOperation(value = "RemoveBatch", tags = {"EMStore" },  notes = "RemoveBatch")
	@RequestMapping(method = RequestMethod.DELETE, value = "/emstores/batch")
    public ResponseEntity<Boolean> removeBatch(@RequestBody List<String> ids) {
        emstoreService.removeBatch(ids);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','eam_practice01-EMStore-Default-all')")
	@ApiOperation(value = "fetchDEFAULT", tags = {"EMStore" } ,notes = "fetchDEFAULT")
    @RequestMapping(method= RequestMethod.GET , value="/emstores/fetchdefault")
	public ResponseEntity<List<EMStoreDTO>> fetchDefault(EMStoreSearchContext context) {
        Page<EMStore> domains = emstoreService.searchDefault(context) ;
        List<EMStoreDTO> list = emstoreMapping.toDto(domains.getContent());
        return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(list);
	}

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','eam_practice01-EMStore-Default-all')")
	@ApiOperation(value = "searchDEFAULT", tags = {"EMStore" } ,notes = "searchDEFAULT")
    @RequestMapping(method= RequestMethod.POST , value="/emstores/searchdefault")
	public ResponseEntity<Page<EMStoreDTO>> searchDefault(@RequestBody EMStoreSearchContext context) {
        Page<EMStore> domains = emstoreService.searchDefault(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(emstoreMapping.toDto(domains.getContent()), context.getPageable(), domains.getTotalElements()));
	}
}
