package cn.ibizlab.eam.webapi.rest;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.math.BigInteger;
import java.util.HashMap;

import lombok.extern.slf4j.Slf4j;
import com.alibaba.fastjson.JSONObject;

import javax.servlet.ServletRequest;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cglib.beans.BeanCopier;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import org.springframework.http.HttpStatus;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.util.StringUtils;
import org.springframework.context.annotation.Lazy;
import org.springframework.security.access.prepost.PreAuthorize;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;

import cn.ibizlab.eam.webapi.dto.*;
import cn.ibizlab.eam.webapi.mapping.*;
import cn.ibizlab.eam.core.eam_core.domain.EMStock;
import cn.ibizlab.eam.core.eam_core.service.IEMStockService;
import cn.ibizlab.eam.core.eam_core.filter.EMStockSearchContext;




@Slf4j
@Api(tags = {"EMStock" })
@RestController("WebApi-emstock")
@RequestMapping("")
public class EMStockResource {

    @Autowired
    private IEMStockService emstockService;

    @Autowired
    @Lazy
    private EMStockMapping emstockMapping;




    @ApiOperation(value = "CheckKey", tags = {"EMStock" },  notes = "CheckKey")
	@RequestMapping(method = RequestMethod.POST, value = "/emstocks/checkkey")
    public ResponseEntity<Boolean> checkKey(@RequestBody EMStockDTO emstockdto) {
        return  ResponseEntity.status(HttpStatus.OK).body(emstockService.checkKey(emstockMapping.toDomain(emstockdto)));
    }




    @PreAuthorize("hasPermission(#emstock_id,'Update',{this.getEntity(),'Sql'})")
    @ApiOperation(value = "Update", tags = {"EMStock" },  notes = "Update")
	@RequestMapping(method = RequestMethod.PUT, value = "/emstocks/{emstock_id}")
    @Transactional
    public ResponseEntity<EMStockDTO> update(@PathVariable("emstock_id") String emstock_id, @RequestBody EMStockDTO emstockdto) {
		EMStock domain = emstockMapping.toDomain(emstockdto);
        domain.setEmstockid(emstock_id);
		emstockService.update(domain);
		EMStockDTO dto = emstockMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasPermission(#emstock_id,'Update',{this.getEntity(),'Sql'})")
    @ApiOperation(value = "UpdateBatch", tags = {"EMStock" },  notes = "UpdateBatch")
	@RequestMapping(method = RequestMethod.PUT, value = "/emstocks/batch")
    public ResponseEntity<Boolean> updateBatch(@RequestBody List<EMStockDTO> emstockdtos) {
        emstockService.updateBatch(emstockMapping.toDomain(emstockdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }




    @ApiOperation(value = "Save", tags = {"EMStock" },  notes = "Save")
	@RequestMapping(method = RequestMethod.POST, value = "/emstocks/save")
    public ResponseEntity<Boolean> save(@RequestBody EMStockDTO emstockdto) {
        return ResponseEntity.status(HttpStatus.OK).body(emstockService.save(emstockMapping.toDomain(emstockdto)));
    }

    @ApiOperation(value = "SaveBatch", tags = {"EMStock" },  notes = "SaveBatch")
	@RequestMapping(method = RequestMethod.POST, value = "/emstocks/savebatch")
    public ResponseEntity<Boolean> saveBatch(@RequestBody List<EMStockDTO> emstockdtos) {
        emstockService.saveBatch(emstockMapping.toDomain(emstockdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }




    @PreAuthorize("hasPermission('','Create',{this.getEntity(),'Sql'})")
    @ApiOperation(value = "Create", tags = {"EMStock" },  notes = "Create")
	@RequestMapping(method = RequestMethod.POST, value = "/emstocks")
    @Transactional
    public ResponseEntity<EMStockDTO> create(@RequestBody EMStockDTO emstockdto) {
        EMStock domain = emstockMapping.toDomain(emstockdto);
		emstockService.create(domain);
        EMStockDTO dto = emstockMapping.toDto(domain);
		return ResponseEntity.status(HttpStatus.OK).body(dto);
    }
    @PreAuthorize("hasPermission('','Create',{this.getEntity(),'Sql'})")
    @ApiOperation(value = "createBatch", tags = {"EMStock" },  notes = "createBatch")
	@RequestMapping(method = RequestMethod.POST, value = "/emstocks/batch")
    public ResponseEntity<Boolean> createBatch(@RequestBody List<EMStockDTO> emstockdtos) {
        emstockService.createBatch(emstockMapping.toDomain(emstockdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }




    @PreAuthorize("hasPermission(#emstock_id,'Remove',{this.getEntity(),'Sql'})")
    @ApiOperation(value = "Remove", tags = {"EMStock" },  notes = "Remove")
	@RequestMapping(method = RequestMethod.DELETE, value = "/emstocks/{emstock_id}")
    @Transactional
    public ResponseEntity<Boolean> remove(@PathVariable("emstock_id") String emstock_id) {
         return ResponseEntity.status(HttpStatus.OK).body(emstockService.remove(emstock_id));
    }

    @ApiOperation(value = "RemoveBatch", tags = {"EMStock" },  notes = "RemoveBatch")
	@RequestMapping(method = RequestMethod.DELETE, value = "/emstocks/batch")
    public ResponseEntity<Boolean> removeBatch(@RequestBody List<String> ids) {
        emstockService.removeBatch(ids);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }




    @ApiOperation(value = "GetDraft", tags = {"EMStock" },  notes = "GetDraft")
	@RequestMapping(method = RequestMethod.GET, value = "/emstocks/getdraft")
    public ResponseEntity<EMStockDTO> getDraft() {
        return ResponseEntity.status(HttpStatus.OK).body(emstockMapping.toDto(emstockService.getDraft(new EMStock())));
    }




    @PreAuthorize("hasPermission(#emstock_id,'Get',{this.getEntity(),'Sql'})")
    @ApiOperation(value = "Get", tags = {"EMStock" },  notes = "Get")
	@RequestMapping(method = RequestMethod.GET, value = "/emstocks/{emstock_id}")
    public ResponseEntity<EMStockDTO> get(@PathVariable("emstock_id") String emstock_id) {
        EMStock domain = emstockService.get(emstock_id);
        EMStockDTO dto = emstockMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','eam_practice01-EMStock-Default-all')")
	@ApiOperation(value = "fetchDEFAULT", tags = {"EMStock" } ,notes = "fetchDEFAULT")
    @RequestMapping(method= RequestMethod.GET , value="/emstocks/fetchdefault")
	public ResponseEntity<List<EMStockDTO>> fetchDefault(EMStockSearchContext context) {
        Page<EMStock> domains = emstockService.searchDefault(context) ;
        List<EMStockDTO> list = emstockMapping.toDto(domains.getContent());
        return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(list);
	}

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','eam_practice01-EMStock-Default-all')")
	@ApiOperation(value = "searchDEFAULT", tags = {"EMStock" } ,notes = "searchDEFAULT")
    @RequestMapping(method= RequestMethod.POST , value="/emstocks/searchdefault")
	public ResponseEntity<Page<EMStockDTO>> searchDefault(@RequestBody EMStockSearchContext context) {
        Page<EMStock> domains = emstockService.searchDefault(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(emstockMapping.toDto(domains.getContent()), context.getPageable(), domains.getTotalElements()));
	}

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','eam_practice01-EMStock-TypeStock-all')")
	@ApiOperation(value = "fetch类型库存统计", tags = {"EMStock" } ,notes = "fetch类型库存统计")
    @RequestMapping(method= RequestMethod.GET , value="/emstocks/fetchtypestock")
	public ResponseEntity<List<HashMap>> fetchTypeStock(EMStockSearchContext context) {
        Page<HashMap> domains = emstockService.searchTypeStock(context) ;
        return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(domains.getContent());
	}

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','eam_practice01-EMStock-TypeStock-all')")
	@ApiOperation(value = "search类型库存统计", tags = {"EMStock" } ,notes = "search类型库存统计")
    @RequestMapping(method= RequestMethod.POST , value="/emstocks/searchtypestock")
	public ResponseEntity<Page<HashMap>> searchTypeStock(@RequestBody EMStockSearchContext context) {
        Page<HashMap> domains = emstockService.searchTypeStock(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(domains.getContent(), context.getPageable(), domains.getTotalElements()));
	}



    @ApiOperation(value = "CheckKeyByEMItem", tags = {"EMStock" },  notes = "CheckKeyByEMItem")
	@RequestMapping(method = RequestMethod.POST, value = "/emitems/{emitem_id}/emstocks/checkkey")
    public ResponseEntity<Boolean> checkKeyByEMItem(@PathVariable("emitem_id") String emitem_id, @RequestBody EMStockDTO emstockdto) {
        return  ResponseEntity.status(HttpStatus.OK).body(emstockService.checkKey(emstockMapping.toDomain(emstockdto)));
    }

    @ApiOperation(value = "UpdateByEMItem", tags = {"EMStock" },  notes = "UpdateByEMItem")
	@RequestMapping(method = RequestMethod.PUT, value = "/emitems/{emitem_id}/emstocks/{emstock_id}")
    @Transactional
    public ResponseEntity<EMStockDTO> updateByEMItem(@PathVariable("emitem_id") String emitem_id, @PathVariable("emstock_id") String emstock_id, @RequestBody EMStockDTO emstockdto) {
        EMStock domain = emstockMapping.toDomain(emstockdto);
        domain.setItemid(emitem_id);
        domain.setEmstockid(emstock_id);
		emstockService.update(domain);
        EMStockDTO dto = emstockMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "UpdateBatchByEMItem", tags = {"EMStock" },  notes = "UpdateBatchByEMItem")
	@RequestMapping(method = RequestMethod.PUT, value = "/emitems/{emitem_id}/emstocks/batch")
    public ResponseEntity<Boolean> updateBatchByEMItem(@PathVariable("emitem_id") String emitem_id, @RequestBody List<EMStockDTO> emstockdtos) {
        List<EMStock> domainlist=emstockMapping.toDomain(emstockdtos);
        for(EMStock domain:domainlist){
            domain.setItemid(emitem_id);
        }
        emstockService.updateBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "SaveByEMItem", tags = {"EMStock" },  notes = "SaveByEMItem")
	@RequestMapping(method = RequestMethod.POST, value = "/emitems/{emitem_id}/emstocks/save")
    public ResponseEntity<Boolean> saveByEMItem(@PathVariable("emitem_id") String emitem_id, @RequestBody EMStockDTO emstockdto) {
        EMStock domain = emstockMapping.toDomain(emstockdto);
        domain.setItemid(emitem_id);
        return ResponseEntity.status(HttpStatus.OK).body(emstockService.save(domain));
    }

    @ApiOperation(value = "SaveBatchByEMItem", tags = {"EMStock" },  notes = "SaveBatchByEMItem")
	@RequestMapping(method = RequestMethod.POST, value = "/emitems/{emitem_id}/emstocks/savebatch")
    public ResponseEntity<Boolean> saveBatchByEMItem(@PathVariable("emitem_id") String emitem_id, @RequestBody List<EMStockDTO> emstockdtos) {
        List<EMStock> domainlist=emstockMapping.toDomain(emstockdtos);
        for(EMStock domain:domainlist){
             domain.setItemid(emitem_id);
        }
        emstockService.saveBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "CreateByEMItem", tags = {"EMStock" },  notes = "CreateByEMItem")
	@RequestMapping(method = RequestMethod.POST, value = "/emitems/{emitem_id}/emstocks")
    @Transactional
    public ResponseEntity<EMStockDTO> createByEMItem(@PathVariable("emitem_id") String emitem_id, @RequestBody EMStockDTO emstockdto) {
        EMStock domain = emstockMapping.toDomain(emstockdto);
        domain.setItemid(emitem_id);
		emstockService.create(domain);
        EMStockDTO dto = emstockMapping.toDto(domain);
		return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "createBatchByEMItem", tags = {"EMStock" },  notes = "createBatchByEMItem")
	@RequestMapping(method = RequestMethod.POST, value = "/emitems/{emitem_id}/emstocks/batch")
    public ResponseEntity<Boolean> createBatchByEMItem(@PathVariable("emitem_id") String emitem_id, @RequestBody List<EMStockDTO> emstockdtos) {
        List<EMStock> domainlist=emstockMapping.toDomain(emstockdtos);
        for(EMStock domain:domainlist){
            domain.setItemid(emitem_id);
        }
        emstockService.createBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "RemoveByEMItem", tags = {"EMStock" },  notes = "RemoveByEMItem")
	@RequestMapping(method = RequestMethod.DELETE, value = "/emitems/{emitem_id}/emstocks/{emstock_id}")
    @Transactional
    public ResponseEntity<Boolean> removeByEMItem(@PathVariable("emitem_id") String emitem_id, @PathVariable("emstock_id") String emstock_id) {
		return ResponseEntity.status(HttpStatus.OK).body(emstockService.remove(emstock_id));
    }

    @ApiOperation(value = "RemoveBatchByEMItem", tags = {"EMStock" },  notes = "RemoveBatchByEMItem")
	@RequestMapping(method = RequestMethod.DELETE, value = "/emitems/{emitem_id}/emstocks/batch")
    public ResponseEntity<Boolean> removeBatchByEMItem(@RequestBody List<String> ids) {
        emstockService.removeBatch(ids);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "GetDraftByEMItem", tags = {"EMStock" },  notes = "GetDraftByEMItem")
    @RequestMapping(method = RequestMethod.GET, value = "/emitems/{emitem_id}/emstocks/getdraft")
    public ResponseEntity<EMStockDTO> getDraftByEMItem(@PathVariable("emitem_id") String emitem_id) {
        EMStock domain = new EMStock();
        domain.setItemid(emitem_id);
        return ResponseEntity.status(HttpStatus.OK).body(emstockMapping.toDto(emstockService.getDraft(domain)));
    }

    @ApiOperation(value = "GetByEMItem", tags = {"EMStock" },  notes = "GetByEMItem")
	@RequestMapping(method = RequestMethod.GET, value = "/emitems/{emitem_id}/emstocks/{emstock_id}")
    public ResponseEntity<EMStockDTO> getByEMItem(@PathVariable("emitem_id") String emitem_id, @PathVariable("emstock_id") String emstock_id) {
        EMStock domain = emstockService.get(emstock_id);
        EMStockDTO dto = emstockMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

	@ApiOperation(value = "fetchDEFAULTByEMItem", tags = {"EMStock" } ,notes = "fetchDEFAULTByEMItem")
    @RequestMapping(method= RequestMethod.GET , value="/emitems/{emitem_id}/emstocks/fetchdefault")
	public ResponseEntity<List<EMStockDTO>> fetchEMStockDefaultByEMItem(@PathVariable("emitem_id") String emitem_id,EMStockSearchContext context) {
        context.setN_itemid_eq(emitem_id);
        Page<EMStock> domains = emstockService.searchDefault(context) ;
        List<EMStockDTO> list = emstockMapping.toDto(domains.getContent());
	    return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(list);
	}

	@ApiOperation(value = "searchDEFAULTByEMItem", tags = {"EMStock" } ,notes = "searchDEFAULTByEMItem")
    @RequestMapping(method= RequestMethod.POST , value="/emitems/{emitem_id}/emstocks/searchdefault")
	public ResponseEntity<Page<EMStockDTO>> searchEMStockDefaultByEMItem(@PathVariable("emitem_id") String emitem_id, @RequestBody EMStockSearchContext context) {
        context.setN_itemid_eq(emitem_id);
        Page<EMStock> domains = emstockService.searchDefault(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(emstockMapping.toDto(domains.getContent()), context.getPageable(), domains.getTotalElements()));
	}

	@ApiOperation(value = "fetch类型库存统计ByEMItem", tags = {"EMStock" } ,notes = "fetch类型库存统计ByEMItem")
    @RequestMapping(method= RequestMethod.GET , value="/emitems/{emitem_id}/emstocks/fetchtypestock")
	public ResponseEntity<List<HashMap>> fetchEMStockTypeStockByEMItem(@PathVariable("emitem_id") String emitem_id,EMStockSearchContext context) {
        context.setN_itemid_eq(emitem_id);
        Page<HashMap> domains = emstockService.searchTypeStock(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(domains.getContent());
	}

	@ApiOperation(value = "search类型库存统计ByEMItem", tags = {"EMStock" } ,notes = "search类型库存统计ByEMItem")
    @RequestMapping(method= RequestMethod.POST , value="/emitems/{emitem_id}/emstocks/searchtypestock")
	public ResponseEntity<Page<HashMap>> searchEMStockTypeStockByEMItem(@PathVariable("emitem_id") String emitem_id, @RequestBody EMStockSearchContext context) {
        context.setN_itemid_eq(emitem_id);
        Page<HashMap> domains = emstockService.searchTypeStock(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(domains.getContent(), context.getPageable(), domains.getTotalElements()));
	}



    @ApiOperation(value = "CheckKeyByEMStorePart", tags = {"EMStock" },  notes = "CheckKeyByEMStorePart")
	@RequestMapping(method = RequestMethod.POST, value = "/emstoreparts/{emstorepart_id}/emstocks/checkkey")
    public ResponseEntity<Boolean> checkKeyByEMStorePart(@PathVariable("emstorepart_id") String emstorepart_id, @RequestBody EMStockDTO emstockdto) {
        return  ResponseEntity.status(HttpStatus.OK).body(emstockService.checkKey(emstockMapping.toDomain(emstockdto)));
    }

    @ApiOperation(value = "UpdateByEMStorePart", tags = {"EMStock" },  notes = "UpdateByEMStorePart")
	@RequestMapping(method = RequestMethod.PUT, value = "/emstoreparts/{emstorepart_id}/emstocks/{emstock_id}")
    @Transactional
    public ResponseEntity<EMStockDTO> updateByEMStorePart(@PathVariable("emstorepart_id") String emstorepart_id, @PathVariable("emstock_id") String emstock_id, @RequestBody EMStockDTO emstockdto) {
        EMStock domain = emstockMapping.toDomain(emstockdto);
        domain.setStorepartid(emstorepart_id);
        domain.setEmstockid(emstock_id);
		emstockService.update(domain);
        EMStockDTO dto = emstockMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "UpdateBatchByEMStorePart", tags = {"EMStock" },  notes = "UpdateBatchByEMStorePart")
	@RequestMapping(method = RequestMethod.PUT, value = "/emstoreparts/{emstorepart_id}/emstocks/batch")
    public ResponseEntity<Boolean> updateBatchByEMStorePart(@PathVariable("emstorepart_id") String emstorepart_id, @RequestBody List<EMStockDTO> emstockdtos) {
        List<EMStock> domainlist=emstockMapping.toDomain(emstockdtos);
        for(EMStock domain:domainlist){
            domain.setStorepartid(emstorepart_id);
        }
        emstockService.updateBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "SaveByEMStorePart", tags = {"EMStock" },  notes = "SaveByEMStorePart")
	@RequestMapping(method = RequestMethod.POST, value = "/emstoreparts/{emstorepart_id}/emstocks/save")
    public ResponseEntity<Boolean> saveByEMStorePart(@PathVariable("emstorepart_id") String emstorepart_id, @RequestBody EMStockDTO emstockdto) {
        EMStock domain = emstockMapping.toDomain(emstockdto);
        domain.setStorepartid(emstorepart_id);
        return ResponseEntity.status(HttpStatus.OK).body(emstockService.save(domain));
    }

    @ApiOperation(value = "SaveBatchByEMStorePart", tags = {"EMStock" },  notes = "SaveBatchByEMStorePart")
	@RequestMapping(method = RequestMethod.POST, value = "/emstoreparts/{emstorepart_id}/emstocks/savebatch")
    public ResponseEntity<Boolean> saveBatchByEMStorePart(@PathVariable("emstorepart_id") String emstorepart_id, @RequestBody List<EMStockDTO> emstockdtos) {
        List<EMStock> domainlist=emstockMapping.toDomain(emstockdtos);
        for(EMStock domain:domainlist){
             domain.setStorepartid(emstorepart_id);
        }
        emstockService.saveBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "CreateByEMStorePart", tags = {"EMStock" },  notes = "CreateByEMStorePart")
	@RequestMapping(method = RequestMethod.POST, value = "/emstoreparts/{emstorepart_id}/emstocks")
    @Transactional
    public ResponseEntity<EMStockDTO> createByEMStorePart(@PathVariable("emstorepart_id") String emstorepart_id, @RequestBody EMStockDTO emstockdto) {
        EMStock domain = emstockMapping.toDomain(emstockdto);
        domain.setStorepartid(emstorepart_id);
		emstockService.create(domain);
        EMStockDTO dto = emstockMapping.toDto(domain);
		return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "createBatchByEMStorePart", tags = {"EMStock" },  notes = "createBatchByEMStorePart")
	@RequestMapping(method = RequestMethod.POST, value = "/emstoreparts/{emstorepart_id}/emstocks/batch")
    public ResponseEntity<Boolean> createBatchByEMStorePart(@PathVariable("emstorepart_id") String emstorepart_id, @RequestBody List<EMStockDTO> emstockdtos) {
        List<EMStock> domainlist=emstockMapping.toDomain(emstockdtos);
        for(EMStock domain:domainlist){
            domain.setStorepartid(emstorepart_id);
        }
        emstockService.createBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "RemoveByEMStorePart", tags = {"EMStock" },  notes = "RemoveByEMStorePart")
	@RequestMapping(method = RequestMethod.DELETE, value = "/emstoreparts/{emstorepart_id}/emstocks/{emstock_id}")
    @Transactional
    public ResponseEntity<Boolean> removeByEMStorePart(@PathVariable("emstorepart_id") String emstorepart_id, @PathVariable("emstock_id") String emstock_id) {
		return ResponseEntity.status(HttpStatus.OK).body(emstockService.remove(emstock_id));
    }

    @ApiOperation(value = "RemoveBatchByEMStorePart", tags = {"EMStock" },  notes = "RemoveBatchByEMStorePart")
	@RequestMapping(method = RequestMethod.DELETE, value = "/emstoreparts/{emstorepart_id}/emstocks/batch")
    public ResponseEntity<Boolean> removeBatchByEMStorePart(@RequestBody List<String> ids) {
        emstockService.removeBatch(ids);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "GetDraftByEMStorePart", tags = {"EMStock" },  notes = "GetDraftByEMStorePart")
    @RequestMapping(method = RequestMethod.GET, value = "/emstoreparts/{emstorepart_id}/emstocks/getdraft")
    public ResponseEntity<EMStockDTO> getDraftByEMStorePart(@PathVariable("emstorepart_id") String emstorepart_id) {
        EMStock domain = new EMStock();
        domain.setStorepartid(emstorepart_id);
        return ResponseEntity.status(HttpStatus.OK).body(emstockMapping.toDto(emstockService.getDraft(domain)));
    }

    @ApiOperation(value = "GetByEMStorePart", tags = {"EMStock" },  notes = "GetByEMStorePart")
	@RequestMapping(method = RequestMethod.GET, value = "/emstoreparts/{emstorepart_id}/emstocks/{emstock_id}")
    public ResponseEntity<EMStockDTO> getByEMStorePart(@PathVariable("emstorepart_id") String emstorepart_id, @PathVariable("emstock_id") String emstock_id) {
        EMStock domain = emstockService.get(emstock_id);
        EMStockDTO dto = emstockMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

	@ApiOperation(value = "fetchDEFAULTByEMStorePart", tags = {"EMStock" } ,notes = "fetchDEFAULTByEMStorePart")
    @RequestMapping(method= RequestMethod.GET , value="/emstoreparts/{emstorepart_id}/emstocks/fetchdefault")
	public ResponseEntity<List<EMStockDTO>> fetchEMStockDefaultByEMStorePart(@PathVariable("emstorepart_id") String emstorepart_id,EMStockSearchContext context) {
        context.setN_storepartid_eq(emstorepart_id);
        Page<EMStock> domains = emstockService.searchDefault(context) ;
        List<EMStockDTO> list = emstockMapping.toDto(domains.getContent());
	    return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(list);
	}

	@ApiOperation(value = "searchDEFAULTByEMStorePart", tags = {"EMStock" } ,notes = "searchDEFAULTByEMStorePart")
    @RequestMapping(method= RequestMethod.POST , value="/emstoreparts/{emstorepart_id}/emstocks/searchdefault")
	public ResponseEntity<Page<EMStockDTO>> searchEMStockDefaultByEMStorePart(@PathVariable("emstorepart_id") String emstorepart_id, @RequestBody EMStockSearchContext context) {
        context.setN_storepartid_eq(emstorepart_id);
        Page<EMStock> domains = emstockService.searchDefault(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(emstockMapping.toDto(domains.getContent()), context.getPageable(), domains.getTotalElements()));
	}

	@ApiOperation(value = "fetch类型库存统计ByEMStorePart", tags = {"EMStock" } ,notes = "fetch类型库存统计ByEMStorePart")
    @RequestMapping(method= RequestMethod.GET , value="/emstoreparts/{emstorepart_id}/emstocks/fetchtypestock")
	public ResponseEntity<List<HashMap>> fetchEMStockTypeStockByEMStorePart(@PathVariable("emstorepart_id") String emstorepart_id,EMStockSearchContext context) {
        context.setN_storepartid_eq(emstorepart_id);
        Page<HashMap> domains = emstockService.searchTypeStock(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(domains.getContent());
	}

	@ApiOperation(value = "search类型库存统计ByEMStorePart", tags = {"EMStock" } ,notes = "search类型库存统计ByEMStorePart")
    @RequestMapping(method= RequestMethod.POST , value="/emstoreparts/{emstorepart_id}/emstocks/searchtypestock")
	public ResponseEntity<Page<HashMap>> searchEMStockTypeStockByEMStorePart(@PathVariable("emstorepart_id") String emstorepart_id, @RequestBody EMStockSearchContext context) {
        context.setN_storepartid_eq(emstorepart_id);
        Page<HashMap> domains = emstockService.searchTypeStock(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(domains.getContent(), context.getPageable(), domains.getTotalElements()));
	}



    @ApiOperation(value = "CheckKeyByEMStore", tags = {"EMStock" },  notes = "CheckKeyByEMStore")
	@RequestMapping(method = RequestMethod.POST, value = "/emstores/{emstore_id}/emstocks/checkkey")
    public ResponseEntity<Boolean> checkKeyByEMStore(@PathVariable("emstore_id") String emstore_id, @RequestBody EMStockDTO emstockdto) {
        return  ResponseEntity.status(HttpStatus.OK).body(emstockService.checkKey(emstockMapping.toDomain(emstockdto)));
    }

    @ApiOperation(value = "UpdateByEMStore", tags = {"EMStock" },  notes = "UpdateByEMStore")
	@RequestMapping(method = RequestMethod.PUT, value = "/emstores/{emstore_id}/emstocks/{emstock_id}")
    @Transactional
    public ResponseEntity<EMStockDTO> updateByEMStore(@PathVariable("emstore_id") String emstore_id, @PathVariable("emstock_id") String emstock_id, @RequestBody EMStockDTO emstockdto) {
        EMStock domain = emstockMapping.toDomain(emstockdto);
        domain.setStoreid(emstore_id);
        domain.setEmstockid(emstock_id);
		emstockService.update(domain);
        EMStockDTO dto = emstockMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "UpdateBatchByEMStore", tags = {"EMStock" },  notes = "UpdateBatchByEMStore")
	@RequestMapping(method = RequestMethod.PUT, value = "/emstores/{emstore_id}/emstocks/batch")
    public ResponseEntity<Boolean> updateBatchByEMStore(@PathVariable("emstore_id") String emstore_id, @RequestBody List<EMStockDTO> emstockdtos) {
        List<EMStock> domainlist=emstockMapping.toDomain(emstockdtos);
        for(EMStock domain:domainlist){
            domain.setStoreid(emstore_id);
        }
        emstockService.updateBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "SaveByEMStore", tags = {"EMStock" },  notes = "SaveByEMStore")
	@RequestMapping(method = RequestMethod.POST, value = "/emstores/{emstore_id}/emstocks/save")
    public ResponseEntity<Boolean> saveByEMStore(@PathVariable("emstore_id") String emstore_id, @RequestBody EMStockDTO emstockdto) {
        EMStock domain = emstockMapping.toDomain(emstockdto);
        domain.setStoreid(emstore_id);
        return ResponseEntity.status(HttpStatus.OK).body(emstockService.save(domain));
    }

    @ApiOperation(value = "SaveBatchByEMStore", tags = {"EMStock" },  notes = "SaveBatchByEMStore")
	@RequestMapping(method = RequestMethod.POST, value = "/emstores/{emstore_id}/emstocks/savebatch")
    public ResponseEntity<Boolean> saveBatchByEMStore(@PathVariable("emstore_id") String emstore_id, @RequestBody List<EMStockDTO> emstockdtos) {
        List<EMStock> domainlist=emstockMapping.toDomain(emstockdtos);
        for(EMStock domain:domainlist){
             domain.setStoreid(emstore_id);
        }
        emstockService.saveBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "CreateByEMStore", tags = {"EMStock" },  notes = "CreateByEMStore")
	@RequestMapping(method = RequestMethod.POST, value = "/emstores/{emstore_id}/emstocks")
    @Transactional
    public ResponseEntity<EMStockDTO> createByEMStore(@PathVariable("emstore_id") String emstore_id, @RequestBody EMStockDTO emstockdto) {
        EMStock domain = emstockMapping.toDomain(emstockdto);
        domain.setStoreid(emstore_id);
		emstockService.create(domain);
        EMStockDTO dto = emstockMapping.toDto(domain);
		return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "createBatchByEMStore", tags = {"EMStock" },  notes = "createBatchByEMStore")
	@RequestMapping(method = RequestMethod.POST, value = "/emstores/{emstore_id}/emstocks/batch")
    public ResponseEntity<Boolean> createBatchByEMStore(@PathVariable("emstore_id") String emstore_id, @RequestBody List<EMStockDTO> emstockdtos) {
        List<EMStock> domainlist=emstockMapping.toDomain(emstockdtos);
        for(EMStock domain:domainlist){
            domain.setStoreid(emstore_id);
        }
        emstockService.createBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "RemoveByEMStore", tags = {"EMStock" },  notes = "RemoveByEMStore")
	@RequestMapping(method = RequestMethod.DELETE, value = "/emstores/{emstore_id}/emstocks/{emstock_id}")
    @Transactional
    public ResponseEntity<Boolean> removeByEMStore(@PathVariable("emstore_id") String emstore_id, @PathVariable("emstock_id") String emstock_id) {
		return ResponseEntity.status(HttpStatus.OK).body(emstockService.remove(emstock_id));
    }

    @ApiOperation(value = "RemoveBatchByEMStore", tags = {"EMStock" },  notes = "RemoveBatchByEMStore")
	@RequestMapping(method = RequestMethod.DELETE, value = "/emstores/{emstore_id}/emstocks/batch")
    public ResponseEntity<Boolean> removeBatchByEMStore(@RequestBody List<String> ids) {
        emstockService.removeBatch(ids);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "GetDraftByEMStore", tags = {"EMStock" },  notes = "GetDraftByEMStore")
    @RequestMapping(method = RequestMethod.GET, value = "/emstores/{emstore_id}/emstocks/getdraft")
    public ResponseEntity<EMStockDTO> getDraftByEMStore(@PathVariable("emstore_id") String emstore_id) {
        EMStock domain = new EMStock();
        domain.setStoreid(emstore_id);
        return ResponseEntity.status(HttpStatus.OK).body(emstockMapping.toDto(emstockService.getDraft(domain)));
    }

    @ApiOperation(value = "GetByEMStore", tags = {"EMStock" },  notes = "GetByEMStore")
	@RequestMapping(method = RequestMethod.GET, value = "/emstores/{emstore_id}/emstocks/{emstock_id}")
    public ResponseEntity<EMStockDTO> getByEMStore(@PathVariable("emstore_id") String emstore_id, @PathVariable("emstock_id") String emstock_id) {
        EMStock domain = emstockService.get(emstock_id);
        EMStockDTO dto = emstockMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

	@ApiOperation(value = "fetchDEFAULTByEMStore", tags = {"EMStock" } ,notes = "fetchDEFAULTByEMStore")
    @RequestMapping(method= RequestMethod.GET , value="/emstores/{emstore_id}/emstocks/fetchdefault")
	public ResponseEntity<List<EMStockDTO>> fetchEMStockDefaultByEMStore(@PathVariable("emstore_id") String emstore_id,EMStockSearchContext context) {
        context.setN_storeid_eq(emstore_id);
        Page<EMStock> domains = emstockService.searchDefault(context) ;
        List<EMStockDTO> list = emstockMapping.toDto(domains.getContent());
	    return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(list);
	}

	@ApiOperation(value = "searchDEFAULTByEMStore", tags = {"EMStock" } ,notes = "searchDEFAULTByEMStore")
    @RequestMapping(method= RequestMethod.POST , value="/emstores/{emstore_id}/emstocks/searchdefault")
	public ResponseEntity<Page<EMStockDTO>> searchEMStockDefaultByEMStore(@PathVariable("emstore_id") String emstore_id, @RequestBody EMStockSearchContext context) {
        context.setN_storeid_eq(emstore_id);
        Page<EMStock> domains = emstockService.searchDefault(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(emstockMapping.toDto(domains.getContent()), context.getPageable(), domains.getTotalElements()));
	}

	@ApiOperation(value = "fetch类型库存统计ByEMStore", tags = {"EMStock" } ,notes = "fetch类型库存统计ByEMStore")
    @RequestMapping(method= RequestMethod.GET , value="/emstores/{emstore_id}/emstocks/fetchtypestock")
	public ResponseEntity<List<HashMap>> fetchEMStockTypeStockByEMStore(@PathVariable("emstore_id") String emstore_id,EMStockSearchContext context) {
        context.setN_storeid_eq(emstore_id);
        Page<HashMap> domains = emstockService.searchTypeStock(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(domains.getContent());
	}

	@ApiOperation(value = "search类型库存统计ByEMStore", tags = {"EMStock" } ,notes = "search类型库存统计ByEMStore")
    @RequestMapping(method= RequestMethod.POST , value="/emstores/{emstore_id}/emstocks/searchtypestock")
	public ResponseEntity<Page<HashMap>> searchEMStockTypeStockByEMStore(@PathVariable("emstore_id") String emstore_id, @RequestBody EMStockSearchContext context) {
        context.setN_storeid_eq(emstore_id);
        Page<HashMap> domains = emstockService.searchTypeStock(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(domains.getContent(), context.getPageable(), domains.getTotalElements()));
	}




    @ApiOperation(value = "CheckKeyByEMServiceEMItem", tags = {"EMStock" },  notes = "CheckKeyByEMServiceEMItem")
	@RequestMapping(method = RequestMethod.POST, value = "/emservices/{emservice_id}/emitems/{emitem_id}/emstocks/checkkey")
    public ResponseEntity<Boolean> checkKeyByEMServiceEMItem(@PathVariable("emservice_id") String emservice_id, @PathVariable("emitem_id") String emitem_id, @RequestBody EMStockDTO emstockdto) {
        return  ResponseEntity.status(HttpStatus.OK).body(emstockService.checkKey(emstockMapping.toDomain(emstockdto)));
    }

    @ApiOperation(value = "UpdateByEMServiceEMItem", tags = {"EMStock" },  notes = "UpdateByEMServiceEMItem")
	@RequestMapping(method = RequestMethod.PUT, value = "/emservices/{emservice_id}/emitems/{emitem_id}/emstocks/{emstock_id}")
    @Transactional
    public ResponseEntity<EMStockDTO> updateByEMServiceEMItem(@PathVariable("emservice_id") String emservice_id, @PathVariable("emitem_id") String emitem_id, @PathVariable("emstock_id") String emstock_id, @RequestBody EMStockDTO emstockdto) {
        EMStock domain = emstockMapping.toDomain(emstockdto);
        domain.setItemid(emitem_id);
        domain.setEmstockid(emstock_id);
		emstockService.update(domain);
        EMStockDTO dto = emstockMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "UpdateBatchByEMServiceEMItem", tags = {"EMStock" },  notes = "UpdateBatchByEMServiceEMItem")
	@RequestMapping(method = RequestMethod.PUT, value = "/emservices/{emservice_id}/emitems/{emitem_id}/emstocks/batch")
    public ResponseEntity<Boolean> updateBatchByEMServiceEMItem(@PathVariable("emservice_id") String emservice_id, @PathVariable("emitem_id") String emitem_id, @RequestBody List<EMStockDTO> emstockdtos) {
        List<EMStock> domainlist=emstockMapping.toDomain(emstockdtos);
        for(EMStock domain:domainlist){
            domain.setItemid(emitem_id);
        }
        emstockService.updateBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "SaveByEMServiceEMItem", tags = {"EMStock" },  notes = "SaveByEMServiceEMItem")
	@RequestMapping(method = RequestMethod.POST, value = "/emservices/{emservice_id}/emitems/{emitem_id}/emstocks/save")
    public ResponseEntity<Boolean> saveByEMServiceEMItem(@PathVariable("emservice_id") String emservice_id, @PathVariable("emitem_id") String emitem_id, @RequestBody EMStockDTO emstockdto) {
        EMStock domain = emstockMapping.toDomain(emstockdto);
        domain.setItemid(emitem_id);
        return ResponseEntity.status(HttpStatus.OK).body(emstockService.save(domain));
    }

    @ApiOperation(value = "SaveBatchByEMServiceEMItem", tags = {"EMStock" },  notes = "SaveBatchByEMServiceEMItem")
	@RequestMapping(method = RequestMethod.POST, value = "/emservices/{emservice_id}/emitems/{emitem_id}/emstocks/savebatch")
    public ResponseEntity<Boolean> saveBatchByEMServiceEMItem(@PathVariable("emservice_id") String emservice_id, @PathVariable("emitem_id") String emitem_id, @RequestBody List<EMStockDTO> emstockdtos) {
        List<EMStock> domainlist=emstockMapping.toDomain(emstockdtos);
        for(EMStock domain:domainlist){
             domain.setItemid(emitem_id);
        }
        emstockService.saveBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "CreateByEMServiceEMItem", tags = {"EMStock" },  notes = "CreateByEMServiceEMItem")
	@RequestMapping(method = RequestMethod.POST, value = "/emservices/{emservice_id}/emitems/{emitem_id}/emstocks")
    @Transactional
    public ResponseEntity<EMStockDTO> createByEMServiceEMItem(@PathVariable("emservice_id") String emservice_id, @PathVariable("emitem_id") String emitem_id, @RequestBody EMStockDTO emstockdto) {
        EMStock domain = emstockMapping.toDomain(emstockdto);
        domain.setItemid(emitem_id);
		emstockService.create(domain);
        EMStockDTO dto = emstockMapping.toDto(domain);
		return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "createBatchByEMServiceEMItem", tags = {"EMStock" },  notes = "createBatchByEMServiceEMItem")
	@RequestMapping(method = RequestMethod.POST, value = "/emservices/{emservice_id}/emitems/{emitem_id}/emstocks/batch")
    public ResponseEntity<Boolean> createBatchByEMServiceEMItem(@PathVariable("emservice_id") String emservice_id, @PathVariable("emitem_id") String emitem_id, @RequestBody List<EMStockDTO> emstockdtos) {
        List<EMStock> domainlist=emstockMapping.toDomain(emstockdtos);
        for(EMStock domain:domainlist){
            domain.setItemid(emitem_id);
        }
        emstockService.createBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "RemoveByEMServiceEMItem", tags = {"EMStock" },  notes = "RemoveByEMServiceEMItem")
	@RequestMapping(method = RequestMethod.DELETE, value = "/emservices/{emservice_id}/emitems/{emitem_id}/emstocks/{emstock_id}")
    @Transactional
    public ResponseEntity<Boolean> removeByEMServiceEMItem(@PathVariable("emservice_id") String emservice_id, @PathVariable("emitem_id") String emitem_id, @PathVariable("emstock_id") String emstock_id) {
		return ResponseEntity.status(HttpStatus.OK).body(emstockService.remove(emstock_id));
    }

    @ApiOperation(value = "RemoveBatchByEMServiceEMItem", tags = {"EMStock" },  notes = "RemoveBatchByEMServiceEMItem")
	@RequestMapping(method = RequestMethod.DELETE, value = "/emservices/{emservice_id}/emitems/{emitem_id}/emstocks/batch")
    public ResponseEntity<Boolean> removeBatchByEMServiceEMItem(@RequestBody List<String> ids) {
        emstockService.removeBatch(ids);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "GetDraftByEMServiceEMItem", tags = {"EMStock" },  notes = "GetDraftByEMServiceEMItem")
    @RequestMapping(method = RequestMethod.GET, value = "/emservices/{emservice_id}/emitems/{emitem_id}/emstocks/getdraft")
    public ResponseEntity<EMStockDTO> getDraftByEMServiceEMItem(@PathVariable("emservice_id") String emservice_id, @PathVariable("emitem_id") String emitem_id) {
        EMStock domain = new EMStock();
        domain.setItemid(emitem_id);
        return ResponseEntity.status(HttpStatus.OK).body(emstockMapping.toDto(emstockService.getDraft(domain)));
    }

    @ApiOperation(value = "GetByEMServiceEMItem", tags = {"EMStock" },  notes = "GetByEMServiceEMItem")
	@RequestMapping(method = RequestMethod.GET, value = "/emservices/{emservice_id}/emitems/{emitem_id}/emstocks/{emstock_id}")
    public ResponseEntity<EMStockDTO> getByEMServiceEMItem(@PathVariable("emservice_id") String emservice_id, @PathVariable("emitem_id") String emitem_id, @PathVariable("emstock_id") String emstock_id) {
        EMStock domain = emstockService.get(emstock_id);
        EMStockDTO dto = emstockMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

	@ApiOperation(value = "fetchDEFAULTByEMServiceEMItem", tags = {"EMStock" } ,notes = "fetchDEFAULTByEMServiceEMItem")
    @RequestMapping(method= RequestMethod.GET , value="/emservices/{emservice_id}/emitems/{emitem_id}/emstocks/fetchdefault")
	public ResponseEntity<List<EMStockDTO>> fetchEMStockDefaultByEMServiceEMItem(@PathVariable("emservice_id") String emservice_id, @PathVariable("emitem_id") String emitem_id,EMStockSearchContext context) {
        context.setN_itemid_eq(emitem_id);
        Page<EMStock> domains = emstockService.searchDefault(context) ;
        List<EMStockDTO> list = emstockMapping.toDto(domains.getContent());
	    return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(list);
	}

	@ApiOperation(value = "searchDEFAULTByEMServiceEMItem", tags = {"EMStock" } ,notes = "searchDEFAULTByEMServiceEMItem")
    @RequestMapping(method= RequestMethod.POST , value="/emservices/{emservice_id}/emitems/{emitem_id}/emstocks/searchdefault")
	public ResponseEntity<Page<EMStockDTO>> searchEMStockDefaultByEMServiceEMItem(@PathVariable("emservice_id") String emservice_id, @PathVariable("emitem_id") String emitem_id, @RequestBody EMStockSearchContext context) {
        context.setN_itemid_eq(emitem_id);
        Page<EMStock> domains = emstockService.searchDefault(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(emstockMapping.toDto(domains.getContent()), context.getPageable(), domains.getTotalElements()));
	}

	@ApiOperation(value = "fetch类型库存统计ByEMServiceEMItem", tags = {"EMStock" } ,notes = "fetch类型库存统计ByEMServiceEMItem")
    @RequestMapping(method= RequestMethod.GET , value="/emservices/{emservice_id}/emitems/{emitem_id}/emstocks/fetchtypestock")
	public ResponseEntity<List<HashMap>> fetchEMStockTypeStockByEMServiceEMItem(@PathVariable("emservice_id") String emservice_id, @PathVariable("emitem_id") String emitem_id,EMStockSearchContext context) {
        context.setN_itemid_eq(emitem_id);
        Page<HashMap> domains = emstockService.searchTypeStock(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(domains.getContent());
	}

	@ApiOperation(value = "search类型库存统计ByEMServiceEMItem", tags = {"EMStock" } ,notes = "search类型库存统计ByEMServiceEMItem")
    @RequestMapping(method= RequestMethod.POST , value="/emservices/{emservice_id}/emitems/{emitem_id}/emstocks/searchtypestock")
	public ResponseEntity<Page<HashMap>> searchEMStockTypeStockByEMServiceEMItem(@PathVariable("emservice_id") String emservice_id, @PathVariable("emitem_id") String emitem_id, @RequestBody EMStockSearchContext context) {
        context.setN_itemid_eq(emitem_id);
        Page<HashMap> domains = emstockService.searchTypeStock(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(domains.getContent(), context.getPageable(), domains.getTotalElements()));
	}




    @ApiOperation(value = "CheckKeyByEMStorePartEMItem", tags = {"EMStock" },  notes = "CheckKeyByEMStorePartEMItem")
	@RequestMapping(method = RequestMethod.POST, value = "/emstoreparts/{emstorepart_id}/emitems/{emitem_id}/emstocks/checkkey")
    public ResponseEntity<Boolean> checkKeyByEMStorePartEMItem(@PathVariable("emstorepart_id") String emstorepart_id, @PathVariable("emitem_id") String emitem_id, @RequestBody EMStockDTO emstockdto) {
        return  ResponseEntity.status(HttpStatus.OK).body(emstockService.checkKey(emstockMapping.toDomain(emstockdto)));
    }

    @ApiOperation(value = "UpdateByEMStorePartEMItem", tags = {"EMStock" },  notes = "UpdateByEMStorePartEMItem")
	@RequestMapping(method = RequestMethod.PUT, value = "/emstoreparts/{emstorepart_id}/emitems/{emitem_id}/emstocks/{emstock_id}")
    @Transactional
    public ResponseEntity<EMStockDTO> updateByEMStorePartEMItem(@PathVariable("emstorepart_id") String emstorepart_id, @PathVariable("emitem_id") String emitem_id, @PathVariable("emstock_id") String emstock_id, @RequestBody EMStockDTO emstockdto) {
        EMStock domain = emstockMapping.toDomain(emstockdto);
        domain.setItemid(emitem_id);
        domain.setEmstockid(emstock_id);
		emstockService.update(domain);
        EMStockDTO dto = emstockMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "UpdateBatchByEMStorePartEMItem", tags = {"EMStock" },  notes = "UpdateBatchByEMStorePartEMItem")
	@RequestMapping(method = RequestMethod.PUT, value = "/emstoreparts/{emstorepart_id}/emitems/{emitem_id}/emstocks/batch")
    public ResponseEntity<Boolean> updateBatchByEMStorePartEMItem(@PathVariable("emstorepart_id") String emstorepart_id, @PathVariable("emitem_id") String emitem_id, @RequestBody List<EMStockDTO> emstockdtos) {
        List<EMStock> domainlist=emstockMapping.toDomain(emstockdtos);
        for(EMStock domain:domainlist){
            domain.setItemid(emitem_id);
        }
        emstockService.updateBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "SaveByEMStorePartEMItem", tags = {"EMStock" },  notes = "SaveByEMStorePartEMItem")
	@RequestMapping(method = RequestMethod.POST, value = "/emstoreparts/{emstorepart_id}/emitems/{emitem_id}/emstocks/save")
    public ResponseEntity<Boolean> saveByEMStorePartEMItem(@PathVariable("emstorepart_id") String emstorepart_id, @PathVariable("emitem_id") String emitem_id, @RequestBody EMStockDTO emstockdto) {
        EMStock domain = emstockMapping.toDomain(emstockdto);
        domain.setItemid(emitem_id);
        return ResponseEntity.status(HttpStatus.OK).body(emstockService.save(domain));
    }

    @ApiOperation(value = "SaveBatchByEMStorePartEMItem", tags = {"EMStock" },  notes = "SaveBatchByEMStorePartEMItem")
	@RequestMapping(method = RequestMethod.POST, value = "/emstoreparts/{emstorepart_id}/emitems/{emitem_id}/emstocks/savebatch")
    public ResponseEntity<Boolean> saveBatchByEMStorePartEMItem(@PathVariable("emstorepart_id") String emstorepart_id, @PathVariable("emitem_id") String emitem_id, @RequestBody List<EMStockDTO> emstockdtos) {
        List<EMStock> domainlist=emstockMapping.toDomain(emstockdtos);
        for(EMStock domain:domainlist){
             domain.setItemid(emitem_id);
        }
        emstockService.saveBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "CreateByEMStorePartEMItem", tags = {"EMStock" },  notes = "CreateByEMStorePartEMItem")
	@RequestMapping(method = RequestMethod.POST, value = "/emstoreparts/{emstorepart_id}/emitems/{emitem_id}/emstocks")
    @Transactional
    public ResponseEntity<EMStockDTO> createByEMStorePartEMItem(@PathVariable("emstorepart_id") String emstorepart_id, @PathVariable("emitem_id") String emitem_id, @RequestBody EMStockDTO emstockdto) {
        EMStock domain = emstockMapping.toDomain(emstockdto);
        domain.setItemid(emitem_id);
		emstockService.create(domain);
        EMStockDTO dto = emstockMapping.toDto(domain);
		return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "createBatchByEMStorePartEMItem", tags = {"EMStock" },  notes = "createBatchByEMStorePartEMItem")
	@RequestMapping(method = RequestMethod.POST, value = "/emstoreparts/{emstorepart_id}/emitems/{emitem_id}/emstocks/batch")
    public ResponseEntity<Boolean> createBatchByEMStorePartEMItem(@PathVariable("emstorepart_id") String emstorepart_id, @PathVariable("emitem_id") String emitem_id, @RequestBody List<EMStockDTO> emstockdtos) {
        List<EMStock> domainlist=emstockMapping.toDomain(emstockdtos);
        for(EMStock domain:domainlist){
            domain.setItemid(emitem_id);
        }
        emstockService.createBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "RemoveByEMStorePartEMItem", tags = {"EMStock" },  notes = "RemoveByEMStorePartEMItem")
	@RequestMapping(method = RequestMethod.DELETE, value = "/emstoreparts/{emstorepart_id}/emitems/{emitem_id}/emstocks/{emstock_id}")
    @Transactional
    public ResponseEntity<Boolean> removeByEMStorePartEMItem(@PathVariable("emstorepart_id") String emstorepart_id, @PathVariable("emitem_id") String emitem_id, @PathVariable("emstock_id") String emstock_id) {
		return ResponseEntity.status(HttpStatus.OK).body(emstockService.remove(emstock_id));
    }

    @ApiOperation(value = "RemoveBatchByEMStorePartEMItem", tags = {"EMStock" },  notes = "RemoveBatchByEMStorePartEMItem")
	@RequestMapping(method = RequestMethod.DELETE, value = "/emstoreparts/{emstorepart_id}/emitems/{emitem_id}/emstocks/batch")
    public ResponseEntity<Boolean> removeBatchByEMStorePartEMItem(@RequestBody List<String> ids) {
        emstockService.removeBatch(ids);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "GetDraftByEMStorePartEMItem", tags = {"EMStock" },  notes = "GetDraftByEMStorePartEMItem")
    @RequestMapping(method = RequestMethod.GET, value = "/emstoreparts/{emstorepart_id}/emitems/{emitem_id}/emstocks/getdraft")
    public ResponseEntity<EMStockDTO> getDraftByEMStorePartEMItem(@PathVariable("emstorepart_id") String emstorepart_id, @PathVariable("emitem_id") String emitem_id) {
        EMStock domain = new EMStock();
        domain.setItemid(emitem_id);
        return ResponseEntity.status(HttpStatus.OK).body(emstockMapping.toDto(emstockService.getDraft(domain)));
    }

    @ApiOperation(value = "GetByEMStorePartEMItem", tags = {"EMStock" },  notes = "GetByEMStorePartEMItem")
	@RequestMapping(method = RequestMethod.GET, value = "/emstoreparts/{emstorepart_id}/emitems/{emitem_id}/emstocks/{emstock_id}")
    public ResponseEntity<EMStockDTO> getByEMStorePartEMItem(@PathVariable("emstorepart_id") String emstorepart_id, @PathVariable("emitem_id") String emitem_id, @PathVariable("emstock_id") String emstock_id) {
        EMStock domain = emstockService.get(emstock_id);
        EMStockDTO dto = emstockMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

	@ApiOperation(value = "fetchDEFAULTByEMStorePartEMItem", tags = {"EMStock" } ,notes = "fetchDEFAULTByEMStorePartEMItem")
    @RequestMapping(method= RequestMethod.GET , value="/emstoreparts/{emstorepart_id}/emitems/{emitem_id}/emstocks/fetchdefault")
	public ResponseEntity<List<EMStockDTO>> fetchEMStockDefaultByEMStorePartEMItem(@PathVariable("emstorepart_id") String emstorepart_id, @PathVariable("emitem_id") String emitem_id,EMStockSearchContext context) {
        context.setN_itemid_eq(emitem_id);
        Page<EMStock> domains = emstockService.searchDefault(context) ;
        List<EMStockDTO> list = emstockMapping.toDto(domains.getContent());
	    return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(list);
	}

	@ApiOperation(value = "searchDEFAULTByEMStorePartEMItem", tags = {"EMStock" } ,notes = "searchDEFAULTByEMStorePartEMItem")
    @RequestMapping(method= RequestMethod.POST , value="/emstoreparts/{emstorepart_id}/emitems/{emitem_id}/emstocks/searchdefault")
	public ResponseEntity<Page<EMStockDTO>> searchEMStockDefaultByEMStorePartEMItem(@PathVariable("emstorepart_id") String emstorepart_id, @PathVariable("emitem_id") String emitem_id, @RequestBody EMStockSearchContext context) {
        context.setN_itemid_eq(emitem_id);
        Page<EMStock> domains = emstockService.searchDefault(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(emstockMapping.toDto(domains.getContent()), context.getPageable(), domains.getTotalElements()));
	}

	@ApiOperation(value = "fetch类型库存统计ByEMStorePartEMItem", tags = {"EMStock" } ,notes = "fetch类型库存统计ByEMStorePartEMItem")
    @RequestMapping(method= RequestMethod.GET , value="/emstoreparts/{emstorepart_id}/emitems/{emitem_id}/emstocks/fetchtypestock")
	public ResponseEntity<List<HashMap>> fetchEMStockTypeStockByEMStorePartEMItem(@PathVariable("emstorepart_id") String emstorepart_id, @PathVariable("emitem_id") String emitem_id,EMStockSearchContext context) {
        context.setN_itemid_eq(emitem_id);
        Page<HashMap> domains = emstockService.searchTypeStock(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(domains.getContent());
	}

	@ApiOperation(value = "search类型库存统计ByEMStorePartEMItem", tags = {"EMStock" } ,notes = "search类型库存统计ByEMStorePartEMItem")
    @RequestMapping(method= RequestMethod.POST , value="/emstoreparts/{emstorepart_id}/emitems/{emitem_id}/emstocks/searchtypestock")
	public ResponseEntity<Page<HashMap>> searchEMStockTypeStockByEMStorePartEMItem(@PathVariable("emstorepart_id") String emstorepart_id, @PathVariable("emitem_id") String emitem_id, @RequestBody EMStockSearchContext context) {
        context.setN_itemid_eq(emitem_id);
        Page<HashMap> domains = emstockService.searchTypeStock(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(domains.getContent(), context.getPageable(), domains.getTotalElements()));
	}




    @ApiOperation(value = "CheckKeyByEMStoreEMItem", tags = {"EMStock" },  notes = "CheckKeyByEMStoreEMItem")
	@RequestMapping(method = RequestMethod.POST, value = "/emstores/{emstore_id}/emitems/{emitem_id}/emstocks/checkkey")
    public ResponseEntity<Boolean> checkKeyByEMStoreEMItem(@PathVariable("emstore_id") String emstore_id, @PathVariable("emitem_id") String emitem_id, @RequestBody EMStockDTO emstockdto) {
        return  ResponseEntity.status(HttpStatus.OK).body(emstockService.checkKey(emstockMapping.toDomain(emstockdto)));
    }

    @ApiOperation(value = "UpdateByEMStoreEMItem", tags = {"EMStock" },  notes = "UpdateByEMStoreEMItem")
	@RequestMapping(method = RequestMethod.PUT, value = "/emstores/{emstore_id}/emitems/{emitem_id}/emstocks/{emstock_id}")
    @Transactional
    public ResponseEntity<EMStockDTO> updateByEMStoreEMItem(@PathVariable("emstore_id") String emstore_id, @PathVariable("emitem_id") String emitem_id, @PathVariable("emstock_id") String emstock_id, @RequestBody EMStockDTO emstockdto) {
        EMStock domain = emstockMapping.toDomain(emstockdto);
        domain.setItemid(emitem_id);
        domain.setEmstockid(emstock_id);
		emstockService.update(domain);
        EMStockDTO dto = emstockMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "UpdateBatchByEMStoreEMItem", tags = {"EMStock" },  notes = "UpdateBatchByEMStoreEMItem")
	@RequestMapping(method = RequestMethod.PUT, value = "/emstores/{emstore_id}/emitems/{emitem_id}/emstocks/batch")
    public ResponseEntity<Boolean> updateBatchByEMStoreEMItem(@PathVariable("emstore_id") String emstore_id, @PathVariable("emitem_id") String emitem_id, @RequestBody List<EMStockDTO> emstockdtos) {
        List<EMStock> domainlist=emstockMapping.toDomain(emstockdtos);
        for(EMStock domain:domainlist){
            domain.setItemid(emitem_id);
        }
        emstockService.updateBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "SaveByEMStoreEMItem", tags = {"EMStock" },  notes = "SaveByEMStoreEMItem")
	@RequestMapping(method = RequestMethod.POST, value = "/emstores/{emstore_id}/emitems/{emitem_id}/emstocks/save")
    public ResponseEntity<Boolean> saveByEMStoreEMItem(@PathVariable("emstore_id") String emstore_id, @PathVariable("emitem_id") String emitem_id, @RequestBody EMStockDTO emstockdto) {
        EMStock domain = emstockMapping.toDomain(emstockdto);
        domain.setItemid(emitem_id);
        return ResponseEntity.status(HttpStatus.OK).body(emstockService.save(domain));
    }

    @ApiOperation(value = "SaveBatchByEMStoreEMItem", tags = {"EMStock" },  notes = "SaveBatchByEMStoreEMItem")
	@RequestMapping(method = RequestMethod.POST, value = "/emstores/{emstore_id}/emitems/{emitem_id}/emstocks/savebatch")
    public ResponseEntity<Boolean> saveBatchByEMStoreEMItem(@PathVariable("emstore_id") String emstore_id, @PathVariable("emitem_id") String emitem_id, @RequestBody List<EMStockDTO> emstockdtos) {
        List<EMStock> domainlist=emstockMapping.toDomain(emstockdtos);
        for(EMStock domain:domainlist){
             domain.setItemid(emitem_id);
        }
        emstockService.saveBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "CreateByEMStoreEMItem", tags = {"EMStock" },  notes = "CreateByEMStoreEMItem")
	@RequestMapping(method = RequestMethod.POST, value = "/emstores/{emstore_id}/emitems/{emitem_id}/emstocks")
    @Transactional
    public ResponseEntity<EMStockDTO> createByEMStoreEMItem(@PathVariable("emstore_id") String emstore_id, @PathVariable("emitem_id") String emitem_id, @RequestBody EMStockDTO emstockdto) {
        EMStock domain = emstockMapping.toDomain(emstockdto);
        domain.setItemid(emitem_id);
		emstockService.create(domain);
        EMStockDTO dto = emstockMapping.toDto(domain);
		return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "createBatchByEMStoreEMItem", tags = {"EMStock" },  notes = "createBatchByEMStoreEMItem")
	@RequestMapping(method = RequestMethod.POST, value = "/emstores/{emstore_id}/emitems/{emitem_id}/emstocks/batch")
    public ResponseEntity<Boolean> createBatchByEMStoreEMItem(@PathVariable("emstore_id") String emstore_id, @PathVariable("emitem_id") String emitem_id, @RequestBody List<EMStockDTO> emstockdtos) {
        List<EMStock> domainlist=emstockMapping.toDomain(emstockdtos);
        for(EMStock domain:domainlist){
            domain.setItemid(emitem_id);
        }
        emstockService.createBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "RemoveByEMStoreEMItem", tags = {"EMStock" },  notes = "RemoveByEMStoreEMItem")
	@RequestMapping(method = RequestMethod.DELETE, value = "/emstores/{emstore_id}/emitems/{emitem_id}/emstocks/{emstock_id}")
    @Transactional
    public ResponseEntity<Boolean> removeByEMStoreEMItem(@PathVariable("emstore_id") String emstore_id, @PathVariable("emitem_id") String emitem_id, @PathVariable("emstock_id") String emstock_id) {
		return ResponseEntity.status(HttpStatus.OK).body(emstockService.remove(emstock_id));
    }

    @ApiOperation(value = "RemoveBatchByEMStoreEMItem", tags = {"EMStock" },  notes = "RemoveBatchByEMStoreEMItem")
	@RequestMapping(method = RequestMethod.DELETE, value = "/emstores/{emstore_id}/emitems/{emitem_id}/emstocks/batch")
    public ResponseEntity<Boolean> removeBatchByEMStoreEMItem(@RequestBody List<String> ids) {
        emstockService.removeBatch(ids);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "GetDraftByEMStoreEMItem", tags = {"EMStock" },  notes = "GetDraftByEMStoreEMItem")
    @RequestMapping(method = RequestMethod.GET, value = "/emstores/{emstore_id}/emitems/{emitem_id}/emstocks/getdraft")
    public ResponseEntity<EMStockDTO> getDraftByEMStoreEMItem(@PathVariable("emstore_id") String emstore_id, @PathVariable("emitem_id") String emitem_id) {
        EMStock domain = new EMStock();
        domain.setItemid(emitem_id);
        return ResponseEntity.status(HttpStatus.OK).body(emstockMapping.toDto(emstockService.getDraft(domain)));
    }

    @ApiOperation(value = "GetByEMStoreEMItem", tags = {"EMStock" },  notes = "GetByEMStoreEMItem")
	@RequestMapping(method = RequestMethod.GET, value = "/emstores/{emstore_id}/emitems/{emitem_id}/emstocks/{emstock_id}")
    public ResponseEntity<EMStockDTO> getByEMStoreEMItem(@PathVariable("emstore_id") String emstore_id, @PathVariable("emitem_id") String emitem_id, @PathVariable("emstock_id") String emstock_id) {
        EMStock domain = emstockService.get(emstock_id);
        EMStockDTO dto = emstockMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

	@ApiOperation(value = "fetchDEFAULTByEMStoreEMItem", tags = {"EMStock" } ,notes = "fetchDEFAULTByEMStoreEMItem")
    @RequestMapping(method= RequestMethod.GET , value="/emstores/{emstore_id}/emitems/{emitem_id}/emstocks/fetchdefault")
	public ResponseEntity<List<EMStockDTO>> fetchEMStockDefaultByEMStoreEMItem(@PathVariable("emstore_id") String emstore_id, @PathVariable("emitem_id") String emitem_id,EMStockSearchContext context) {
        context.setN_itemid_eq(emitem_id);
        Page<EMStock> domains = emstockService.searchDefault(context) ;
        List<EMStockDTO> list = emstockMapping.toDto(domains.getContent());
	    return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(list);
	}

	@ApiOperation(value = "searchDEFAULTByEMStoreEMItem", tags = {"EMStock" } ,notes = "searchDEFAULTByEMStoreEMItem")
    @RequestMapping(method= RequestMethod.POST , value="/emstores/{emstore_id}/emitems/{emitem_id}/emstocks/searchdefault")
	public ResponseEntity<Page<EMStockDTO>> searchEMStockDefaultByEMStoreEMItem(@PathVariable("emstore_id") String emstore_id, @PathVariable("emitem_id") String emitem_id, @RequestBody EMStockSearchContext context) {
        context.setN_itemid_eq(emitem_id);
        Page<EMStock> domains = emstockService.searchDefault(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(emstockMapping.toDto(domains.getContent()), context.getPageable(), domains.getTotalElements()));
	}

	@ApiOperation(value = "fetch类型库存统计ByEMStoreEMItem", tags = {"EMStock" } ,notes = "fetch类型库存统计ByEMStoreEMItem")
    @RequestMapping(method= RequestMethod.GET , value="/emstores/{emstore_id}/emitems/{emitem_id}/emstocks/fetchtypestock")
	public ResponseEntity<List<HashMap>> fetchEMStockTypeStockByEMStoreEMItem(@PathVariable("emstore_id") String emstore_id, @PathVariable("emitem_id") String emitem_id,EMStockSearchContext context) {
        context.setN_itemid_eq(emitem_id);
        Page<HashMap> domains = emstockService.searchTypeStock(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(domains.getContent());
	}

	@ApiOperation(value = "search类型库存统计ByEMStoreEMItem", tags = {"EMStock" } ,notes = "search类型库存统计ByEMStoreEMItem")
    @RequestMapping(method= RequestMethod.POST , value="/emstores/{emstore_id}/emitems/{emitem_id}/emstocks/searchtypestock")
	public ResponseEntity<Page<HashMap>> searchEMStockTypeStockByEMStoreEMItem(@PathVariable("emstore_id") String emstore_id, @PathVariable("emitem_id") String emitem_id, @RequestBody EMStockSearchContext context) {
        context.setN_itemid_eq(emitem_id);
        Page<HashMap> domains = emstockService.searchTypeStock(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(domains.getContent(), context.getPageable(), domains.getTotalElements()));
	}




    @ApiOperation(value = "CheckKeyByPFEmpEMItem", tags = {"EMStock" },  notes = "CheckKeyByPFEmpEMItem")
	@RequestMapping(method = RequestMethod.POST, value = "/pfemps/{pfemp_id}/emitems/{emitem_id}/emstocks/checkkey")
    public ResponseEntity<Boolean> checkKeyByPFEmpEMItem(@PathVariable("pfemp_id") String pfemp_id, @PathVariable("emitem_id") String emitem_id, @RequestBody EMStockDTO emstockdto) {
        return  ResponseEntity.status(HttpStatus.OK).body(emstockService.checkKey(emstockMapping.toDomain(emstockdto)));
    }

    @ApiOperation(value = "UpdateByPFEmpEMItem", tags = {"EMStock" },  notes = "UpdateByPFEmpEMItem")
	@RequestMapping(method = RequestMethod.PUT, value = "/pfemps/{pfemp_id}/emitems/{emitem_id}/emstocks/{emstock_id}")
    @Transactional
    public ResponseEntity<EMStockDTO> updateByPFEmpEMItem(@PathVariable("pfemp_id") String pfemp_id, @PathVariable("emitem_id") String emitem_id, @PathVariable("emstock_id") String emstock_id, @RequestBody EMStockDTO emstockdto) {
        EMStock domain = emstockMapping.toDomain(emstockdto);
        domain.setItemid(emitem_id);
        domain.setEmstockid(emstock_id);
		emstockService.update(domain);
        EMStockDTO dto = emstockMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "UpdateBatchByPFEmpEMItem", tags = {"EMStock" },  notes = "UpdateBatchByPFEmpEMItem")
	@RequestMapping(method = RequestMethod.PUT, value = "/pfemps/{pfemp_id}/emitems/{emitem_id}/emstocks/batch")
    public ResponseEntity<Boolean> updateBatchByPFEmpEMItem(@PathVariable("pfemp_id") String pfemp_id, @PathVariable("emitem_id") String emitem_id, @RequestBody List<EMStockDTO> emstockdtos) {
        List<EMStock> domainlist=emstockMapping.toDomain(emstockdtos);
        for(EMStock domain:domainlist){
            domain.setItemid(emitem_id);
        }
        emstockService.updateBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "SaveByPFEmpEMItem", tags = {"EMStock" },  notes = "SaveByPFEmpEMItem")
	@RequestMapping(method = RequestMethod.POST, value = "/pfemps/{pfemp_id}/emitems/{emitem_id}/emstocks/save")
    public ResponseEntity<Boolean> saveByPFEmpEMItem(@PathVariable("pfemp_id") String pfemp_id, @PathVariable("emitem_id") String emitem_id, @RequestBody EMStockDTO emstockdto) {
        EMStock domain = emstockMapping.toDomain(emstockdto);
        domain.setItemid(emitem_id);
        return ResponseEntity.status(HttpStatus.OK).body(emstockService.save(domain));
    }

    @ApiOperation(value = "SaveBatchByPFEmpEMItem", tags = {"EMStock" },  notes = "SaveBatchByPFEmpEMItem")
	@RequestMapping(method = RequestMethod.POST, value = "/pfemps/{pfemp_id}/emitems/{emitem_id}/emstocks/savebatch")
    public ResponseEntity<Boolean> saveBatchByPFEmpEMItem(@PathVariable("pfemp_id") String pfemp_id, @PathVariable("emitem_id") String emitem_id, @RequestBody List<EMStockDTO> emstockdtos) {
        List<EMStock> domainlist=emstockMapping.toDomain(emstockdtos);
        for(EMStock domain:domainlist){
             domain.setItemid(emitem_id);
        }
        emstockService.saveBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "CreateByPFEmpEMItem", tags = {"EMStock" },  notes = "CreateByPFEmpEMItem")
	@RequestMapping(method = RequestMethod.POST, value = "/pfemps/{pfemp_id}/emitems/{emitem_id}/emstocks")
    @Transactional
    public ResponseEntity<EMStockDTO> createByPFEmpEMItem(@PathVariable("pfemp_id") String pfemp_id, @PathVariable("emitem_id") String emitem_id, @RequestBody EMStockDTO emstockdto) {
        EMStock domain = emstockMapping.toDomain(emstockdto);
        domain.setItemid(emitem_id);
		emstockService.create(domain);
        EMStockDTO dto = emstockMapping.toDto(domain);
		return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "createBatchByPFEmpEMItem", tags = {"EMStock" },  notes = "createBatchByPFEmpEMItem")
	@RequestMapping(method = RequestMethod.POST, value = "/pfemps/{pfemp_id}/emitems/{emitem_id}/emstocks/batch")
    public ResponseEntity<Boolean> createBatchByPFEmpEMItem(@PathVariable("pfemp_id") String pfemp_id, @PathVariable("emitem_id") String emitem_id, @RequestBody List<EMStockDTO> emstockdtos) {
        List<EMStock> domainlist=emstockMapping.toDomain(emstockdtos);
        for(EMStock domain:domainlist){
            domain.setItemid(emitem_id);
        }
        emstockService.createBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "RemoveByPFEmpEMItem", tags = {"EMStock" },  notes = "RemoveByPFEmpEMItem")
	@RequestMapping(method = RequestMethod.DELETE, value = "/pfemps/{pfemp_id}/emitems/{emitem_id}/emstocks/{emstock_id}")
    @Transactional
    public ResponseEntity<Boolean> removeByPFEmpEMItem(@PathVariable("pfemp_id") String pfemp_id, @PathVariable("emitem_id") String emitem_id, @PathVariable("emstock_id") String emstock_id) {
		return ResponseEntity.status(HttpStatus.OK).body(emstockService.remove(emstock_id));
    }

    @ApiOperation(value = "RemoveBatchByPFEmpEMItem", tags = {"EMStock" },  notes = "RemoveBatchByPFEmpEMItem")
	@RequestMapping(method = RequestMethod.DELETE, value = "/pfemps/{pfemp_id}/emitems/{emitem_id}/emstocks/batch")
    public ResponseEntity<Boolean> removeBatchByPFEmpEMItem(@RequestBody List<String> ids) {
        emstockService.removeBatch(ids);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "GetDraftByPFEmpEMItem", tags = {"EMStock" },  notes = "GetDraftByPFEmpEMItem")
    @RequestMapping(method = RequestMethod.GET, value = "/pfemps/{pfemp_id}/emitems/{emitem_id}/emstocks/getdraft")
    public ResponseEntity<EMStockDTO> getDraftByPFEmpEMItem(@PathVariable("pfemp_id") String pfemp_id, @PathVariable("emitem_id") String emitem_id) {
        EMStock domain = new EMStock();
        domain.setItemid(emitem_id);
        return ResponseEntity.status(HttpStatus.OK).body(emstockMapping.toDto(emstockService.getDraft(domain)));
    }

    @ApiOperation(value = "GetByPFEmpEMItem", tags = {"EMStock" },  notes = "GetByPFEmpEMItem")
	@RequestMapping(method = RequestMethod.GET, value = "/pfemps/{pfemp_id}/emitems/{emitem_id}/emstocks/{emstock_id}")
    public ResponseEntity<EMStockDTO> getByPFEmpEMItem(@PathVariable("pfemp_id") String pfemp_id, @PathVariable("emitem_id") String emitem_id, @PathVariable("emstock_id") String emstock_id) {
        EMStock domain = emstockService.get(emstock_id);
        EMStockDTO dto = emstockMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

	@ApiOperation(value = "fetchDEFAULTByPFEmpEMItem", tags = {"EMStock" } ,notes = "fetchDEFAULTByPFEmpEMItem")
    @RequestMapping(method= RequestMethod.GET , value="/pfemps/{pfemp_id}/emitems/{emitem_id}/emstocks/fetchdefault")
	public ResponseEntity<List<EMStockDTO>> fetchEMStockDefaultByPFEmpEMItem(@PathVariable("pfemp_id") String pfemp_id, @PathVariable("emitem_id") String emitem_id,EMStockSearchContext context) {
        context.setN_itemid_eq(emitem_id);
        Page<EMStock> domains = emstockService.searchDefault(context) ;
        List<EMStockDTO> list = emstockMapping.toDto(domains.getContent());
	    return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(list);
	}

	@ApiOperation(value = "searchDEFAULTByPFEmpEMItem", tags = {"EMStock" } ,notes = "searchDEFAULTByPFEmpEMItem")
    @RequestMapping(method= RequestMethod.POST , value="/pfemps/{pfemp_id}/emitems/{emitem_id}/emstocks/searchdefault")
	public ResponseEntity<Page<EMStockDTO>> searchEMStockDefaultByPFEmpEMItem(@PathVariable("pfemp_id") String pfemp_id, @PathVariable("emitem_id") String emitem_id, @RequestBody EMStockSearchContext context) {
        context.setN_itemid_eq(emitem_id);
        Page<EMStock> domains = emstockService.searchDefault(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(emstockMapping.toDto(domains.getContent()), context.getPageable(), domains.getTotalElements()));
	}

	@ApiOperation(value = "fetch类型库存统计ByPFEmpEMItem", tags = {"EMStock" } ,notes = "fetch类型库存统计ByPFEmpEMItem")
    @RequestMapping(method= RequestMethod.GET , value="/pfemps/{pfemp_id}/emitems/{emitem_id}/emstocks/fetchtypestock")
	public ResponseEntity<List<HashMap>> fetchEMStockTypeStockByPFEmpEMItem(@PathVariable("pfemp_id") String pfemp_id, @PathVariable("emitem_id") String emitem_id,EMStockSearchContext context) {
        context.setN_itemid_eq(emitem_id);
        Page<HashMap> domains = emstockService.searchTypeStock(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(domains.getContent());
	}

	@ApiOperation(value = "search类型库存统计ByPFEmpEMItem", tags = {"EMStock" } ,notes = "search类型库存统计ByPFEmpEMItem")
    @RequestMapping(method= RequestMethod.POST , value="/pfemps/{pfemp_id}/emitems/{emitem_id}/emstocks/searchtypestock")
	public ResponseEntity<Page<HashMap>> searchEMStockTypeStockByPFEmpEMItem(@PathVariable("pfemp_id") String pfemp_id, @PathVariable("emitem_id") String emitem_id, @RequestBody EMStockSearchContext context) {
        context.setN_itemid_eq(emitem_id);
        Page<HashMap> domains = emstockService.searchTypeStock(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(domains.getContent(), context.getPageable(), domains.getTotalElements()));
	}




    @ApiOperation(value = "CheckKeyByPFUnitEMItem", tags = {"EMStock" },  notes = "CheckKeyByPFUnitEMItem")
	@RequestMapping(method = RequestMethod.POST, value = "/pfunits/{pfunit_id}/emitems/{emitem_id}/emstocks/checkkey")
    public ResponseEntity<Boolean> checkKeyByPFUnitEMItem(@PathVariable("pfunit_id") String pfunit_id, @PathVariable("emitem_id") String emitem_id, @RequestBody EMStockDTO emstockdto) {
        return  ResponseEntity.status(HttpStatus.OK).body(emstockService.checkKey(emstockMapping.toDomain(emstockdto)));
    }

    @ApiOperation(value = "UpdateByPFUnitEMItem", tags = {"EMStock" },  notes = "UpdateByPFUnitEMItem")
	@RequestMapping(method = RequestMethod.PUT, value = "/pfunits/{pfunit_id}/emitems/{emitem_id}/emstocks/{emstock_id}")
    @Transactional
    public ResponseEntity<EMStockDTO> updateByPFUnitEMItem(@PathVariable("pfunit_id") String pfunit_id, @PathVariable("emitem_id") String emitem_id, @PathVariable("emstock_id") String emstock_id, @RequestBody EMStockDTO emstockdto) {
        EMStock domain = emstockMapping.toDomain(emstockdto);
        domain.setItemid(emitem_id);
        domain.setEmstockid(emstock_id);
		emstockService.update(domain);
        EMStockDTO dto = emstockMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "UpdateBatchByPFUnitEMItem", tags = {"EMStock" },  notes = "UpdateBatchByPFUnitEMItem")
	@RequestMapping(method = RequestMethod.PUT, value = "/pfunits/{pfunit_id}/emitems/{emitem_id}/emstocks/batch")
    public ResponseEntity<Boolean> updateBatchByPFUnitEMItem(@PathVariable("pfunit_id") String pfunit_id, @PathVariable("emitem_id") String emitem_id, @RequestBody List<EMStockDTO> emstockdtos) {
        List<EMStock> domainlist=emstockMapping.toDomain(emstockdtos);
        for(EMStock domain:domainlist){
            domain.setItemid(emitem_id);
        }
        emstockService.updateBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "SaveByPFUnitEMItem", tags = {"EMStock" },  notes = "SaveByPFUnitEMItem")
	@RequestMapping(method = RequestMethod.POST, value = "/pfunits/{pfunit_id}/emitems/{emitem_id}/emstocks/save")
    public ResponseEntity<Boolean> saveByPFUnitEMItem(@PathVariable("pfunit_id") String pfunit_id, @PathVariable("emitem_id") String emitem_id, @RequestBody EMStockDTO emstockdto) {
        EMStock domain = emstockMapping.toDomain(emstockdto);
        domain.setItemid(emitem_id);
        return ResponseEntity.status(HttpStatus.OK).body(emstockService.save(domain));
    }

    @ApiOperation(value = "SaveBatchByPFUnitEMItem", tags = {"EMStock" },  notes = "SaveBatchByPFUnitEMItem")
	@RequestMapping(method = RequestMethod.POST, value = "/pfunits/{pfunit_id}/emitems/{emitem_id}/emstocks/savebatch")
    public ResponseEntity<Boolean> saveBatchByPFUnitEMItem(@PathVariable("pfunit_id") String pfunit_id, @PathVariable("emitem_id") String emitem_id, @RequestBody List<EMStockDTO> emstockdtos) {
        List<EMStock> domainlist=emstockMapping.toDomain(emstockdtos);
        for(EMStock domain:domainlist){
             domain.setItemid(emitem_id);
        }
        emstockService.saveBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "CreateByPFUnitEMItem", tags = {"EMStock" },  notes = "CreateByPFUnitEMItem")
	@RequestMapping(method = RequestMethod.POST, value = "/pfunits/{pfunit_id}/emitems/{emitem_id}/emstocks")
    @Transactional
    public ResponseEntity<EMStockDTO> createByPFUnitEMItem(@PathVariable("pfunit_id") String pfunit_id, @PathVariable("emitem_id") String emitem_id, @RequestBody EMStockDTO emstockdto) {
        EMStock domain = emstockMapping.toDomain(emstockdto);
        domain.setItemid(emitem_id);
		emstockService.create(domain);
        EMStockDTO dto = emstockMapping.toDto(domain);
		return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "createBatchByPFUnitEMItem", tags = {"EMStock" },  notes = "createBatchByPFUnitEMItem")
	@RequestMapping(method = RequestMethod.POST, value = "/pfunits/{pfunit_id}/emitems/{emitem_id}/emstocks/batch")
    public ResponseEntity<Boolean> createBatchByPFUnitEMItem(@PathVariable("pfunit_id") String pfunit_id, @PathVariable("emitem_id") String emitem_id, @RequestBody List<EMStockDTO> emstockdtos) {
        List<EMStock> domainlist=emstockMapping.toDomain(emstockdtos);
        for(EMStock domain:domainlist){
            domain.setItemid(emitem_id);
        }
        emstockService.createBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "RemoveByPFUnitEMItem", tags = {"EMStock" },  notes = "RemoveByPFUnitEMItem")
	@RequestMapping(method = RequestMethod.DELETE, value = "/pfunits/{pfunit_id}/emitems/{emitem_id}/emstocks/{emstock_id}")
    @Transactional
    public ResponseEntity<Boolean> removeByPFUnitEMItem(@PathVariable("pfunit_id") String pfunit_id, @PathVariable("emitem_id") String emitem_id, @PathVariable("emstock_id") String emstock_id) {
		return ResponseEntity.status(HttpStatus.OK).body(emstockService.remove(emstock_id));
    }

    @ApiOperation(value = "RemoveBatchByPFUnitEMItem", tags = {"EMStock" },  notes = "RemoveBatchByPFUnitEMItem")
	@RequestMapping(method = RequestMethod.DELETE, value = "/pfunits/{pfunit_id}/emitems/{emitem_id}/emstocks/batch")
    public ResponseEntity<Boolean> removeBatchByPFUnitEMItem(@RequestBody List<String> ids) {
        emstockService.removeBatch(ids);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "GetDraftByPFUnitEMItem", tags = {"EMStock" },  notes = "GetDraftByPFUnitEMItem")
    @RequestMapping(method = RequestMethod.GET, value = "/pfunits/{pfunit_id}/emitems/{emitem_id}/emstocks/getdraft")
    public ResponseEntity<EMStockDTO> getDraftByPFUnitEMItem(@PathVariable("pfunit_id") String pfunit_id, @PathVariable("emitem_id") String emitem_id) {
        EMStock domain = new EMStock();
        domain.setItemid(emitem_id);
        return ResponseEntity.status(HttpStatus.OK).body(emstockMapping.toDto(emstockService.getDraft(domain)));
    }

    @ApiOperation(value = "GetByPFUnitEMItem", tags = {"EMStock" },  notes = "GetByPFUnitEMItem")
	@RequestMapping(method = RequestMethod.GET, value = "/pfunits/{pfunit_id}/emitems/{emitem_id}/emstocks/{emstock_id}")
    public ResponseEntity<EMStockDTO> getByPFUnitEMItem(@PathVariable("pfunit_id") String pfunit_id, @PathVariable("emitem_id") String emitem_id, @PathVariable("emstock_id") String emstock_id) {
        EMStock domain = emstockService.get(emstock_id);
        EMStockDTO dto = emstockMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

	@ApiOperation(value = "fetchDEFAULTByPFUnitEMItem", tags = {"EMStock" } ,notes = "fetchDEFAULTByPFUnitEMItem")
    @RequestMapping(method= RequestMethod.GET , value="/pfunits/{pfunit_id}/emitems/{emitem_id}/emstocks/fetchdefault")
	public ResponseEntity<List<EMStockDTO>> fetchEMStockDefaultByPFUnitEMItem(@PathVariable("pfunit_id") String pfunit_id, @PathVariable("emitem_id") String emitem_id,EMStockSearchContext context) {
        context.setN_itemid_eq(emitem_id);
        Page<EMStock> domains = emstockService.searchDefault(context) ;
        List<EMStockDTO> list = emstockMapping.toDto(domains.getContent());
	    return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(list);
	}

	@ApiOperation(value = "searchDEFAULTByPFUnitEMItem", tags = {"EMStock" } ,notes = "searchDEFAULTByPFUnitEMItem")
    @RequestMapping(method= RequestMethod.POST , value="/pfunits/{pfunit_id}/emitems/{emitem_id}/emstocks/searchdefault")
	public ResponseEntity<Page<EMStockDTO>> searchEMStockDefaultByPFUnitEMItem(@PathVariable("pfunit_id") String pfunit_id, @PathVariable("emitem_id") String emitem_id, @RequestBody EMStockSearchContext context) {
        context.setN_itemid_eq(emitem_id);
        Page<EMStock> domains = emstockService.searchDefault(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(emstockMapping.toDto(domains.getContent()), context.getPageable(), domains.getTotalElements()));
	}

	@ApiOperation(value = "fetch类型库存统计ByPFUnitEMItem", tags = {"EMStock" } ,notes = "fetch类型库存统计ByPFUnitEMItem")
    @RequestMapping(method= RequestMethod.GET , value="/pfunits/{pfunit_id}/emitems/{emitem_id}/emstocks/fetchtypestock")
	public ResponseEntity<List<HashMap>> fetchEMStockTypeStockByPFUnitEMItem(@PathVariable("pfunit_id") String pfunit_id, @PathVariable("emitem_id") String emitem_id,EMStockSearchContext context) {
        context.setN_itemid_eq(emitem_id);
        Page<HashMap> domains = emstockService.searchTypeStock(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(domains.getContent());
	}

	@ApiOperation(value = "search类型库存统计ByPFUnitEMItem", tags = {"EMStock" } ,notes = "search类型库存统计ByPFUnitEMItem")
    @RequestMapping(method= RequestMethod.POST , value="/pfunits/{pfunit_id}/emitems/{emitem_id}/emstocks/searchtypestock")
	public ResponseEntity<Page<HashMap>> searchEMStockTypeStockByPFUnitEMItem(@PathVariable("pfunit_id") String pfunit_id, @PathVariable("emitem_id") String emitem_id, @RequestBody EMStockSearchContext context) {
        context.setN_itemid_eq(emitem_id);
        Page<HashMap> domains = emstockService.searchTypeStock(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(domains.getContent(), context.getPageable(), domains.getTotalElements()));
	}




    @ApiOperation(value = "CheckKeyByEMStoreEMStorePart", tags = {"EMStock" },  notes = "CheckKeyByEMStoreEMStorePart")
	@RequestMapping(method = RequestMethod.POST, value = "/emstores/{emstore_id}/emstoreparts/{emstorepart_id}/emstocks/checkkey")
    public ResponseEntity<Boolean> checkKeyByEMStoreEMStorePart(@PathVariable("emstore_id") String emstore_id, @PathVariable("emstorepart_id") String emstorepart_id, @RequestBody EMStockDTO emstockdto) {
        return  ResponseEntity.status(HttpStatus.OK).body(emstockService.checkKey(emstockMapping.toDomain(emstockdto)));
    }

    @ApiOperation(value = "UpdateByEMStoreEMStorePart", tags = {"EMStock" },  notes = "UpdateByEMStoreEMStorePart")
	@RequestMapping(method = RequestMethod.PUT, value = "/emstores/{emstore_id}/emstoreparts/{emstorepart_id}/emstocks/{emstock_id}")
    @Transactional
    public ResponseEntity<EMStockDTO> updateByEMStoreEMStorePart(@PathVariable("emstore_id") String emstore_id, @PathVariable("emstorepart_id") String emstorepart_id, @PathVariable("emstock_id") String emstock_id, @RequestBody EMStockDTO emstockdto) {
        EMStock domain = emstockMapping.toDomain(emstockdto);
        domain.setStorepartid(emstorepart_id);
        domain.setEmstockid(emstock_id);
		emstockService.update(domain);
        EMStockDTO dto = emstockMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "UpdateBatchByEMStoreEMStorePart", tags = {"EMStock" },  notes = "UpdateBatchByEMStoreEMStorePart")
	@RequestMapping(method = RequestMethod.PUT, value = "/emstores/{emstore_id}/emstoreparts/{emstorepart_id}/emstocks/batch")
    public ResponseEntity<Boolean> updateBatchByEMStoreEMStorePart(@PathVariable("emstore_id") String emstore_id, @PathVariable("emstorepart_id") String emstorepart_id, @RequestBody List<EMStockDTO> emstockdtos) {
        List<EMStock> domainlist=emstockMapping.toDomain(emstockdtos);
        for(EMStock domain:domainlist){
            domain.setStorepartid(emstorepart_id);
        }
        emstockService.updateBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "SaveByEMStoreEMStorePart", tags = {"EMStock" },  notes = "SaveByEMStoreEMStorePart")
	@RequestMapping(method = RequestMethod.POST, value = "/emstores/{emstore_id}/emstoreparts/{emstorepart_id}/emstocks/save")
    public ResponseEntity<Boolean> saveByEMStoreEMStorePart(@PathVariable("emstore_id") String emstore_id, @PathVariable("emstorepart_id") String emstorepart_id, @RequestBody EMStockDTO emstockdto) {
        EMStock domain = emstockMapping.toDomain(emstockdto);
        domain.setStorepartid(emstorepart_id);
        return ResponseEntity.status(HttpStatus.OK).body(emstockService.save(domain));
    }

    @ApiOperation(value = "SaveBatchByEMStoreEMStorePart", tags = {"EMStock" },  notes = "SaveBatchByEMStoreEMStorePart")
	@RequestMapping(method = RequestMethod.POST, value = "/emstores/{emstore_id}/emstoreparts/{emstorepart_id}/emstocks/savebatch")
    public ResponseEntity<Boolean> saveBatchByEMStoreEMStorePart(@PathVariable("emstore_id") String emstore_id, @PathVariable("emstorepart_id") String emstorepart_id, @RequestBody List<EMStockDTO> emstockdtos) {
        List<EMStock> domainlist=emstockMapping.toDomain(emstockdtos);
        for(EMStock domain:domainlist){
             domain.setStorepartid(emstorepart_id);
        }
        emstockService.saveBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "CreateByEMStoreEMStorePart", tags = {"EMStock" },  notes = "CreateByEMStoreEMStorePart")
	@RequestMapping(method = RequestMethod.POST, value = "/emstores/{emstore_id}/emstoreparts/{emstorepart_id}/emstocks")
    @Transactional
    public ResponseEntity<EMStockDTO> createByEMStoreEMStorePart(@PathVariable("emstore_id") String emstore_id, @PathVariable("emstorepart_id") String emstorepart_id, @RequestBody EMStockDTO emstockdto) {
        EMStock domain = emstockMapping.toDomain(emstockdto);
        domain.setStorepartid(emstorepart_id);
		emstockService.create(domain);
        EMStockDTO dto = emstockMapping.toDto(domain);
		return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "createBatchByEMStoreEMStorePart", tags = {"EMStock" },  notes = "createBatchByEMStoreEMStorePart")
	@RequestMapping(method = RequestMethod.POST, value = "/emstores/{emstore_id}/emstoreparts/{emstorepart_id}/emstocks/batch")
    public ResponseEntity<Boolean> createBatchByEMStoreEMStorePart(@PathVariable("emstore_id") String emstore_id, @PathVariable("emstorepart_id") String emstorepart_id, @RequestBody List<EMStockDTO> emstockdtos) {
        List<EMStock> domainlist=emstockMapping.toDomain(emstockdtos);
        for(EMStock domain:domainlist){
            domain.setStorepartid(emstorepart_id);
        }
        emstockService.createBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "RemoveByEMStoreEMStorePart", tags = {"EMStock" },  notes = "RemoveByEMStoreEMStorePart")
	@RequestMapping(method = RequestMethod.DELETE, value = "/emstores/{emstore_id}/emstoreparts/{emstorepart_id}/emstocks/{emstock_id}")
    @Transactional
    public ResponseEntity<Boolean> removeByEMStoreEMStorePart(@PathVariable("emstore_id") String emstore_id, @PathVariable("emstorepart_id") String emstorepart_id, @PathVariable("emstock_id") String emstock_id) {
		return ResponseEntity.status(HttpStatus.OK).body(emstockService.remove(emstock_id));
    }

    @ApiOperation(value = "RemoveBatchByEMStoreEMStorePart", tags = {"EMStock" },  notes = "RemoveBatchByEMStoreEMStorePart")
	@RequestMapping(method = RequestMethod.DELETE, value = "/emstores/{emstore_id}/emstoreparts/{emstorepart_id}/emstocks/batch")
    public ResponseEntity<Boolean> removeBatchByEMStoreEMStorePart(@RequestBody List<String> ids) {
        emstockService.removeBatch(ids);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "GetDraftByEMStoreEMStorePart", tags = {"EMStock" },  notes = "GetDraftByEMStoreEMStorePart")
    @RequestMapping(method = RequestMethod.GET, value = "/emstores/{emstore_id}/emstoreparts/{emstorepart_id}/emstocks/getdraft")
    public ResponseEntity<EMStockDTO> getDraftByEMStoreEMStorePart(@PathVariable("emstore_id") String emstore_id, @PathVariable("emstorepart_id") String emstorepart_id) {
        EMStock domain = new EMStock();
        domain.setStorepartid(emstorepart_id);
        return ResponseEntity.status(HttpStatus.OK).body(emstockMapping.toDto(emstockService.getDraft(domain)));
    }

    @ApiOperation(value = "GetByEMStoreEMStorePart", tags = {"EMStock" },  notes = "GetByEMStoreEMStorePart")
	@RequestMapping(method = RequestMethod.GET, value = "/emstores/{emstore_id}/emstoreparts/{emstorepart_id}/emstocks/{emstock_id}")
    public ResponseEntity<EMStockDTO> getByEMStoreEMStorePart(@PathVariable("emstore_id") String emstore_id, @PathVariable("emstorepart_id") String emstorepart_id, @PathVariable("emstock_id") String emstock_id) {
        EMStock domain = emstockService.get(emstock_id);
        EMStockDTO dto = emstockMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

	@ApiOperation(value = "fetchDEFAULTByEMStoreEMStorePart", tags = {"EMStock" } ,notes = "fetchDEFAULTByEMStoreEMStorePart")
    @RequestMapping(method= RequestMethod.GET , value="/emstores/{emstore_id}/emstoreparts/{emstorepart_id}/emstocks/fetchdefault")
	public ResponseEntity<List<EMStockDTO>> fetchEMStockDefaultByEMStoreEMStorePart(@PathVariable("emstore_id") String emstore_id, @PathVariable("emstorepart_id") String emstorepart_id,EMStockSearchContext context) {
        context.setN_storepartid_eq(emstorepart_id);
        Page<EMStock> domains = emstockService.searchDefault(context) ;
        List<EMStockDTO> list = emstockMapping.toDto(domains.getContent());
	    return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(list);
	}

	@ApiOperation(value = "searchDEFAULTByEMStoreEMStorePart", tags = {"EMStock" } ,notes = "searchDEFAULTByEMStoreEMStorePart")
    @RequestMapping(method= RequestMethod.POST , value="/emstores/{emstore_id}/emstoreparts/{emstorepart_id}/emstocks/searchdefault")
	public ResponseEntity<Page<EMStockDTO>> searchEMStockDefaultByEMStoreEMStorePart(@PathVariable("emstore_id") String emstore_id, @PathVariable("emstorepart_id") String emstorepart_id, @RequestBody EMStockSearchContext context) {
        context.setN_storepartid_eq(emstorepart_id);
        Page<EMStock> domains = emstockService.searchDefault(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(emstockMapping.toDto(domains.getContent()), context.getPageable(), domains.getTotalElements()));
	}

	@ApiOperation(value = "fetch类型库存统计ByEMStoreEMStorePart", tags = {"EMStock" } ,notes = "fetch类型库存统计ByEMStoreEMStorePart")
    @RequestMapping(method= RequestMethod.GET , value="/emstores/{emstore_id}/emstoreparts/{emstorepart_id}/emstocks/fetchtypestock")
	public ResponseEntity<List<HashMap>> fetchEMStockTypeStockByEMStoreEMStorePart(@PathVariable("emstore_id") String emstore_id, @PathVariable("emstorepart_id") String emstorepart_id,EMStockSearchContext context) {
        context.setN_storepartid_eq(emstorepart_id);
        Page<HashMap> domains = emstockService.searchTypeStock(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(domains.getContent());
	}

	@ApiOperation(value = "search类型库存统计ByEMStoreEMStorePart", tags = {"EMStock" } ,notes = "search类型库存统计ByEMStoreEMStorePart")
    @RequestMapping(method= RequestMethod.POST , value="/emstores/{emstore_id}/emstoreparts/{emstorepart_id}/emstocks/searchtypestock")
	public ResponseEntity<Page<HashMap>> searchEMStockTypeStockByEMStoreEMStorePart(@PathVariable("emstore_id") String emstore_id, @PathVariable("emstorepart_id") String emstorepart_id, @RequestBody EMStockSearchContext context) {
        context.setN_storepartid_eq(emstorepart_id);
        Page<HashMap> domains = emstockService.searchTypeStock(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(domains.getContent(), context.getPageable(), domains.getTotalElements()));
	}





    @ApiOperation(value = "CheckKeyByEMStoreEMStorePartEMItem", tags = {"EMStock" },  notes = "CheckKeyByEMStoreEMStorePartEMItem")
	@RequestMapping(method = RequestMethod.POST, value = "/emstores/{emstore_id}/emstoreparts/{emstorepart_id}/emitems/{emitem_id}/emstocks/checkkey")
    public ResponseEntity<Boolean> checkKeyByEMStoreEMStorePartEMItem(@PathVariable("emstore_id") String emstore_id, @PathVariable("emstorepart_id") String emstorepart_id, @PathVariable("emitem_id") String emitem_id, @RequestBody EMStockDTO emstockdto) {
        return  ResponseEntity.status(HttpStatus.OK).body(emstockService.checkKey(emstockMapping.toDomain(emstockdto)));
    }

    @ApiOperation(value = "UpdateByEMStoreEMStorePartEMItem", tags = {"EMStock" },  notes = "UpdateByEMStoreEMStorePartEMItem")
	@RequestMapping(method = RequestMethod.PUT, value = "/emstores/{emstore_id}/emstoreparts/{emstorepart_id}/emitems/{emitem_id}/emstocks/{emstock_id}")
    @Transactional
    public ResponseEntity<EMStockDTO> updateByEMStoreEMStorePartEMItem(@PathVariable("emstore_id") String emstore_id, @PathVariable("emstorepart_id") String emstorepart_id, @PathVariable("emitem_id") String emitem_id, @PathVariable("emstock_id") String emstock_id, @RequestBody EMStockDTO emstockdto) {
        EMStock domain = emstockMapping.toDomain(emstockdto);
        domain.setItemid(emitem_id);
        domain.setEmstockid(emstock_id);
		emstockService.update(domain);
        EMStockDTO dto = emstockMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "UpdateBatchByEMStoreEMStorePartEMItem", tags = {"EMStock" },  notes = "UpdateBatchByEMStoreEMStorePartEMItem")
	@RequestMapping(method = RequestMethod.PUT, value = "/emstores/{emstore_id}/emstoreparts/{emstorepart_id}/emitems/{emitem_id}/emstocks/batch")
    public ResponseEntity<Boolean> updateBatchByEMStoreEMStorePartEMItem(@PathVariable("emstore_id") String emstore_id, @PathVariable("emstorepart_id") String emstorepart_id, @PathVariable("emitem_id") String emitem_id, @RequestBody List<EMStockDTO> emstockdtos) {
        List<EMStock> domainlist=emstockMapping.toDomain(emstockdtos);
        for(EMStock domain:domainlist){
            domain.setItemid(emitem_id);
        }
        emstockService.updateBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "SaveByEMStoreEMStorePartEMItem", tags = {"EMStock" },  notes = "SaveByEMStoreEMStorePartEMItem")
	@RequestMapping(method = RequestMethod.POST, value = "/emstores/{emstore_id}/emstoreparts/{emstorepart_id}/emitems/{emitem_id}/emstocks/save")
    public ResponseEntity<Boolean> saveByEMStoreEMStorePartEMItem(@PathVariable("emstore_id") String emstore_id, @PathVariable("emstorepart_id") String emstorepart_id, @PathVariable("emitem_id") String emitem_id, @RequestBody EMStockDTO emstockdto) {
        EMStock domain = emstockMapping.toDomain(emstockdto);
        domain.setItemid(emitem_id);
        return ResponseEntity.status(HttpStatus.OK).body(emstockService.save(domain));
    }

    @ApiOperation(value = "SaveBatchByEMStoreEMStorePartEMItem", tags = {"EMStock" },  notes = "SaveBatchByEMStoreEMStorePartEMItem")
	@RequestMapping(method = RequestMethod.POST, value = "/emstores/{emstore_id}/emstoreparts/{emstorepart_id}/emitems/{emitem_id}/emstocks/savebatch")
    public ResponseEntity<Boolean> saveBatchByEMStoreEMStorePartEMItem(@PathVariable("emstore_id") String emstore_id, @PathVariable("emstorepart_id") String emstorepart_id, @PathVariable("emitem_id") String emitem_id, @RequestBody List<EMStockDTO> emstockdtos) {
        List<EMStock> domainlist=emstockMapping.toDomain(emstockdtos);
        for(EMStock domain:domainlist){
             domain.setItemid(emitem_id);
        }
        emstockService.saveBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "CreateByEMStoreEMStorePartEMItem", tags = {"EMStock" },  notes = "CreateByEMStoreEMStorePartEMItem")
	@RequestMapping(method = RequestMethod.POST, value = "/emstores/{emstore_id}/emstoreparts/{emstorepart_id}/emitems/{emitem_id}/emstocks")
    @Transactional
    public ResponseEntity<EMStockDTO> createByEMStoreEMStorePartEMItem(@PathVariable("emstore_id") String emstore_id, @PathVariable("emstorepart_id") String emstorepart_id, @PathVariable("emitem_id") String emitem_id, @RequestBody EMStockDTO emstockdto) {
        EMStock domain = emstockMapping.toDomain(emstockdto);
        domain.setItemid(emitem_id);
		emstockService.create(domain);
        EMStockDTO dto = emstockMapping.toDto(domain);
		return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "createBatchByEMStoreEMStorePartEMItem", tags = {"EMStock" },  notes = "createBatchByEMStoreEMStorePartEMItem")
	@RequestMapping(method = RequestMethod.POST, value = "/emstores/{emstore_id}/emstoreparts/{emstorepart_id}/emitems/{emitem_id}/emstocks/batch")
    public ResponseEntity<Boolean> createBatchByEMStoreEMStorePartEMItem(@PathVariable("emstore_id") String emstore_id, @PathVariable("emstorepart_id") String emstorepart_id, @PathVariable("emitem_id") String emitem_id, @RequestBody List<EMStockDTO> emstockdtos) {
        List<EMStock> domainlist=emstockMapping.toDomain(emstockdtos);
        for(EMStock domain:domainlist){
            domain.setItemid(emitem_id);
        }
        emstockService.createBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "RemoveByEMStoreEMStorePartEMItem", tags = {"EMStock" },  notes = "RemoveByEMStoreEMStorePartEMItem")
	@RequestMapping(method = RequestMethod.DELETE, value = "/emstores/{emstore_id}/emstoreparts/{emstorepart_id}/emitems/{emitem_id}/emstocks/{emstock_id}")
    @Transactional
    public ResponseEntity<Boolean> removeByEMStoreEMStorePartEMItem(@PathVariable("emstore_id") String emstore_id, @PathVariable("emstorepart_id") String emstorepart_id, @PathVariable("emitem_id") String emitem_id, @PathVariable("emstock_id") String emstock_id) {
		return ResponseEntity.status(HttpStatus.OK).body(emstockService.remove(emstock_id));
    }

    @ApiOperation(value = "RemoveBatchByEMStoreEMStorePartEMItem", tags = {"EMStock" },  notes = "RemoveBatchByEMStoreEMStorePartEMItem")
	@RequestMapping(method = RequestMethod.DELETE, value = "/emstores/{emstore_id}/emstoreparts/{emstorepart_id}/emitems/{emitem_id}/emstocks/batch")
    public ResponseEntity<Boolean> removeBatchByEMStoreEMStorePartEMItem(@RequestBody List<String> ids) {
        emstockService.removeBatch(ids);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "GetDraftByEMStoreEMStorePartEMItem", tags = {"EMStock" },  notes = "GetDraftByEMStoreEMStorePartEMItem")
    @RequestMapping(method = RequestMethod.GET, value = "/emstores/{emstore_id}/emstoreparts/{emstorepart_id}/emitems/{emitem_id}/emstocks/getdraft")
    public ResponseEntity<EMStockDTO> getDraftByEMStoreEMStorePartEMItem(@PathVariable("emstore_id") String emstore_id, @PathVariable("emstorepart_id") String emstorepart_id, @PathVariable("emitem_id") String emitem_id) {
        EMStock domain = new EMStock();
        domain.setItemid(emitem_id);
        return ResponseEntity.status(HttpStatus.OK).body(emstockMapping.toDto(emstockService.getDraft(domain)));
    }

    @ApiOperation(value = "GetByEMStoreEMStorePartEMItem", tags = {"EMStock" },  notes = "GetByEMStoreEMStorePartEMItem")
	@RequestMapping(method = RequestMethod.GET, value = "/emstores/{emstore_id}/emstoreparts/{emstorepart_id}/emitems/{emitem_id}/emstocks/{emstock_id}")
    public ResponseEntity<EMStockDTO> getByEMStoreEMStorePartEMItem(@PathVariable("emstore_id") String emstore_id, @PathVariable("emstorepart_id") String emstorepart_id, @PathVariable("emitem_id") String emitem_id, @PathVariable("emstock_id") String emstock_id) {
        EMStock domain = emstockService.get(emstock_id);
        EMStockDTO dto = emstockMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

	@ApiOperation(value = "fetchDEFAULTByEMStoreEMStorePartEMItem", tags = {"EMStock" } ,notes = "fetchDEFAULTByEMStoreEMStorePartEMItem")
    @RequestMapping(method= RequestMethod.GET , value="/emstores/{emstore_id}/emstoreparts/{emstorepart_id}/emitems/{emitem_id}/emstocks/fetchdefault")
	public ResponseEntity<List<EMStockDTO>> fetchEMStockDefaultByEMStoreEMStorePartEMItem(@PathVariable("emstore_id") String emstore_id, @PathVariable("emstorepart_id") String emstorepart_id, @PathVariable("emitem_id") String emitem_id,EMStockSearchContext context) {
        context.setN_itemid_eq(emitem_id);
        Page<EMStock> domains = emstockService.searchDefault(context) ;
        List<EMStockDTO> list = emstockMapping.toDto(domains.getContent());
	    return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(list);
	}

	@ApiOperation(value = "searchDEFAULTByEMStoreEMStorePartEMItem", tags = {"EMStock" } ,notes = "searchDEFAULTByEMStoreEMStorePartEMItem")
    @RequestMapping(method= RequestMethod.POST , value="/emstores/{emstore_id}/emstoreparts/{emstorepart_id}/emitems/{emitem_id}/emstocks/searchdefault")
	public ResponseEntity<Page<EMStockDTO>> searchEMStockDefaultByEMStoreEMStorePartEMItem(@PathVariable("emstore_id") String emstore_id, @PathVariable("emstorepart_id") String emstorepart_id, @PathVariable("emitem_id") String emitem_id, @RequestBody EMStockSearchContext context) {
        context.setN_itemid_eq(emitem_id);
        Page<EMStock> domains = emstockService.searchDefault(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(emstockMapping.toDto(domains.getContent()), context.getPageable(), domains.getTotalElements()));
	}

	@ApiOperation(value = "fetch类型库存统计ByEMStoreEMStorePartEMItem", tags = {"EMStock" } ,notes = "fetch类型库存统计ByEMStoreEMStorePartEMItem")
    @RequestMapping(method= RequestMethod.GET , value="/emstores/{emstore_id}/emstoreparts/{emstorepart_id}/emitems/{emitem_id}/emstocks/fetchtypestock")
	public ResponseEntity<List<HashMap>> fetchEMStockTypeStockByEMStoreEMStorePartEMItem(@PathVariable("emstore_id") String emstore_id, @PathVariable("emstorepart_id") String emstorepart_id, @PathVariable("emitem_id") String emitem_id,EMStockSearchContext context) {
        context.setN_itemid_eq(emitem_id);
        Page<HashMap> domains = emstockService.searchTypeStock(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(domains.getContent());
	}

	@ApiOperation(value = "search类型库存统计ByEMStoreEMStorePartEMItem", tags = {"EMStock" } ,notes = "search类型库存统计ByEMStoreEMStorePartEMItem")
    @RequestMapping(method= RequestMethod.POST , value="/emstores/{emstore_id}/emstoreparts/{emstorepart_id}/emitems/{emitem_id}/emstocks/searchtypestock")
	public ResponseEntity<Page<HashMap>> searchEMStockTypeStockByEMStoreEMStorePartEMItem(@PathVariable("emstore_id") String emstore_id, @PathVariable("emstorepart_id") String emstorepart_id, @PathVariable("emitem_id") String emitem_id, @RequestBody EMStockSearchContext context) {
        context.setN_itemid_eq(emitem_id);
        Page<HashMap> domains = emstockService.searchTypeStock(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(domains.getContent(), context.getPageable(), domains.getTotalElements()));
	}


    /**
     * 用户权限校验
     * @return
     */
	public EMStock getEntity(){
        return new EMStock();
    }

}
