package cn.ibizlab.util.security;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import cn.ibizlab.util.annotation.DEField;
import cn.ibizlab.util.domain.EntityBase;
import cn.ibizlab.util.enums.DEPredefinedFieldType;
import cn.ibizlab.util.filter.QueryWrapperContext;
import cn.ibizlab.util.helper.DEFieldCacheMap;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.security.access.PermissionEvaluator;
import org.springframework.security.core.Authentication;
import org.springframework.stereotype.Component;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;
import java.io.Serializable;
import java.lang.reflect.Field;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * spring security 权限管理类
 * 重写权限控制方法
 */
@Component
public class AuthPermissionEvaluator implements PermissionEvaluator {

    @Value("${ibiz.enablePermissionValid:false}")
    boolean enablePermissionValid;  //是否开启权限校验

    /**
     * 表格权限检查 ：用于检查当前用户是否拥有表格数据的读取、删除权限
     *
     * @param authentication
     * @param deAction     表格行为，如：[READ,DELETE]
     * @param gridParam     表格参数，如：当前表格所处实体(EntityName)、表格删除的数据主键(srfkeys)
     * @return true/false true则允许当前行为，false拒绝行为
     */
    @Override
    public boolean hasPermission(Authentication authentication, Object deAction, Object gridParam) {

        //未开启权限校验、超级管理员则不进行权限检查
        if(AuthenticationUser.getAuthenticationUser().getSuperuser()==1  || !enablePermissionValid)
            return true;

        String action = "";
        if (deAction instanceof String)
            action = (String) deAction;

        if (StringUtils.isEmpty(action))
            return false;

        //获取当前用户权限列表
        JSONObject userPermission= AuthenticationUser.getAuthenticationUser().getPermisionList();

        if(userPermission==null)
            return false;

        List gridParamList = (ArrayList) gridParam;
        if(action.equals("DELETE")){ //grid delete
            //准备参数
            Object srfKey =gridParamList.get(0);
            EntityBase entity = (EntityBase) gridParamList.get(1);
            String entityName = entity.getClass().getSimpleName();

            //获取实体行为权限信息
            JSONObject permissionList=userPermission.getJSONObject("deActionPermission");

            //检查是否有操作权限[create.update.delete.read]
            if(!validHasPermission(permissionList,entityName,action)){
                return false;
            }
            //检查是否有数据权限[单行删除]
            ServiceImpl service= SpringContextHolder.getBean(String.format("%s%s",entityName,"ServiceImpl"));//获取实体service对象
            Map<String,String> permissionField=getPermissionField(entity);//获取组织、部门预置属性
            String permissionSQL=getPermissionSQLById(permissionList,entityName,action,srfKey,permissionField);//获取权限SQL
            if(StringUtils.isEmpty(permissionSQL))
                return false;
            QueryWrapper permissionWrapper=getPermissionWrapper(permissionSQL);//构造权限条件
            return testDataAccess(service,permissionWrapper);//执行权限检查
        }
        else{   //grid fetch
            //准备参数
            Object searchContext=gridParamList.get(0);
            String dataSet=String.valueOf(gridParamList.get(1));
            EntityBase entity = (EntityBase) gridParamList.get(2);
            String entityName = entity.getClass().getSimpleName();

            //获取数据集权限信息
            JSONObject permissionList=userPermission.getJSONObject("deDataSetPermission");

            if(StringUtils.isEmpty(entityName)|| StringUtils.isEmpty(dataSet)|| StringUtils.isEmpty(action))
                return false;

            //检查是否有操作权限[create.update.delete.read]
            if(!validHasPermission(permissionList,entityName,dataSet,action)){
                return false;
            }

            Map<String,String> permissionField=getPermissionField(entity);//获取组织、部门预置属性
            String permissionSQL=getPermissionSQLByList(permissionList,entityName,action,dataSet,permissionField);//获取权限SQL
            if(StringUtils.isEmpty(permissionSQL))
                return false;
            fillPermissionSQL(searchContext,permissionSQL);//将权限SQL添加到searchContext中，过滤出权限内数据
        }
            return true;
    }

    /**
     * 表单权限检查 ：用于检查当前用户是否拥有表单的新建、编辑、删除权限
     *
     * @param authentication
     * @param srfKey         当前操作数据的主键
     * @param action         当前操作行为：如：[READ、UPDATE、DELETE]
     * @param cur_entity     当前操作的实体对象
     * @return true/false true则允许当前行为，false拒绝行为
     */
    @Override
    public boolean hasPermission(Authentication authentication, Serializable srfKey, String action, Object cur_entity) {

        //未开启权限校验、超级管理员则不进行权限检查
        if(AuthenticationUser.getAuthenticationUser().getSuperuser()==1  || !enablePermissionValid)
            return true;

        EntityBase entity = null;
        if (cur_entity instanceof EntityBase)
            entity = (EntityBase) cur_entity;

        if (StringUtils.isEmpty(entity))
            return false;

        JSONObject userPermission= AuthenticationUser.getAuthenticationUser().getPermisionList();
        JSONObject permissionList=userPermission.getJSONObject("deActionPermission");
        String entityName = entity.getClass().getSimpleName();

        if(action.equals("CREATE")){
            return validHasPermission(permissionList,entityName,action);
        }
        else{
            //拥有全部数据访问权限时，则跳过权限检查
            if(isAllData(permissionList,entityName,action)){
                return true;
            }
            //检查是否有操作权限[create.update.delete.read]
            if(!validHasPermission(permissionList,entityName,action)){
                return false;
            }
            //检查是否有数据权限
            ServiceImpl service= SpringContextHolder.getBean(String.format("%s%s",entityName,"ServiceImpl"));
            Map<String,String> permissionField=getPermissionField(entity);//获取组织、部门预置属性
            String permissionSQL=getPermissionSQLById(permissionList,entityName,action,srfKey,permissionField);//获取权限SQL
            if(StringUtils.isEmpty(permissionSQL))
                return false;

            QueryWrapper permissionWrapper=getPermissionWrapper(permissionSQL);//构造权限条件
            return testDataAccess(service,permissionWrapper);//执行权限检查
        }
    }

    /**
     * 是否为全部数据
     * @param permissionList
     * @param entityName
     * @param action
     * @return
     */
    private boolean isAllData(JSONObject permissionList, String entityName, String action) {

        if(permissionList==null)
            return false;
        if(!permissionList.containsKey(entityName))
            return false;
        JSONObject entity=permissionList.getJSONObject(entityName);
        if(entity.containsKey(action) && entity.getJSONArray(action).contains("ALL"))
            return true;

            return false;
    }

    /**
     * 拼接表格查询条件
     * @param gridDataAbility
     * @param entityName
     * @param action
     * @param dataSetName
     * @param permissionField
     * @return
     */
    private String  getPermissionSQLByList(JSONObject gridDataAbility, String entityName, String action, String dataSetName, Map<String,String> permissionField){

        JSONObject entity=gridDataAbility.getJSONObject(entityName);//获取实体
        JSONObject dataSet=entity.getJSONObject(dataSetName);//获取实体数据集
        JSONArray opprivList=dataSet.getJSONArray(action);//行为：read；insert...
        if(opprivList.size()==0)
            return null;
        return getPermissionSQL(opprivList,permissionField); //拼接权限条件-查询
    }

    /**
     * 填充权限SQL
     * @param targetDomainObject
     * @param permissionCond
     */
    private void fillPermissionSQL(Object targetDomainObject, String permissionCond){

        if(targetDomainObject instanceof QueryWrapperContext){
            QueryWrapperContext queryWrapperContext = (QueryWrapperContext) targetDomainObject;
            QueryWrapper queryWrapper = queryWrapperContext.getSelectCond();
            queryWrapper.apply(permissionCond);
        }
    }

    /**
     * 校验是否有访问实体行为能力
     * @param permissionList 权限列表
     * @param entityName    实体名称
     * @param action        操作行为
     * @return
     */
    private boolean validHasPermission(JSONObject permissionList, String entityName, String action){

        boolean hasPermission=false;
        if(permissionList==null)
            return false;
        if(!permissionList.containsKey(entityName))
            return false;
        JSONObject entity=permissionList.getJSONObject(entityName);
        if(entity.containsKey(action)){
            hasPermission=true;
        }
            return hasPermission;
    }


    /**
     * 校验是否有访问数据集能力
     * @param permissionList
     * @param entityName
     * @param dataSetName
     * @param action
     * @return
     */
    private boolean validHasPermission(JSONObject permissionList, String entityName, String dataSetName, String action ){

        boolean hasPermission=false;
        if(permissionList==null)
            return false;
        if(!permissionList.containsKey(entityName))
            return false;
        JSONObject entity=permissionList.getJSONObject(entityName);
        if(!entity.containsKey(dataSetName))
            return false;
        JSONObject dataSet=entity.getJSONObject(dataSetName);//获取实体数据集
        if(dataSet.containsKey(action)){
            hasPermission=true;
        }
        return hasPermission;
    }


    /**
     * 获取单条权限数据SQL
     * @param formDataAbility
     * @param entityName
     * @param action
     * @param srfKey
     * @param permissionField
     * @return
     */
    private String getPermissionSQLById(JSONObject formDataAbility, String entityName, String action, Object srfKey, Map<String,String> permissionField){

        JSONObject entity=formDataAbility.getJSONObject(entityName);//获取实体
        JSONArray opprivList=entity.getJSONArray(action);//行为：read；insert...
        if(opprivList.size()==0)
            return null;
        String permissionSQL=getPermissionSQL(opprivList,permissionField);

        String keyField=permissionField.get("keyfield");
        if(StringUtils.isEmpty(keyField)){
            throw new RuntimeException("权限校验失败，请检查当前实体中是否已经配置主键属性!");
        }
        return String.format(" (%s) AND (%s='%s')",permissionSQL,keyField,srfKey); //拼接权限条件-编辑
    }


    /**
     * 表单权限检查
     * @param service
     * @param permissionCond
     * @return
     */
    private boolean testDataAccess(ServiceImpl service, QueryWrapper permissionCond){

        boolean isPermission=false;
         List list=service.list(permissionCond);
         if(list.size()>0)
             isPermission=true;
        return isPermission;
    }


    /**
     * 获取权限SQL
     * @param oppriList
     * @param permissionField
     * @return
     */
    private String  getPermissionSQL(JSONArray oppriList, Map<String,String> permissionField){

        String nPermissionSQL = "1<>1";
        String orgField=permissionField.get("orgfield");
        String orgDeptField=permissionField.get("orgsecfield");
        String createManField=permissionField.get("createmanfield");
        StringBuffer permissionSQL=new StringBuffer();
        AuthenticationUser authenticationUser = AuthenticationUser.getAuthenticationUser();
        JSONObject userInfo = authenticationUser.getOrgInfo();
        JSONObject orgObject = userInfo.getJSONObject("org");
        JSONArray orgParent = orgObject.getJSONArray("porg");
        JSONArray orgChild = orgObject.getJSONArray("sorg");
        JSONObject orgDeptObject = userInfo.getJSONObject("orgdept");
        JSONArray orgDeptParent = orgDeptObject.getJSONArray("porgdept");
        JSONArray orgDeptChild = orgDeptObject.getJSONArray("sorgdept");

        for(int i=0;i<oppriList.size();i++){
            permissionSQL.append("OR");
            String permissionCond=oppriList.getString(i);//权限配置条件
            if(permissionCond.equals("CURORG")){   //本单位
                permissionSQL.append(String.format("(%s='%s')",orgField,AuthenticationUser.getAuthenticationUser().getOrgid()));
            }
            else if(permissionCond.equals("PORG")){//上级单位
                permissionSQL.append(String.format(" %s in(%s) ", orgField, formatStringArr(orgParent)));
            }
            else if(permissionCond.equals("SORG")){//下级单位
                permissionSQL.append(String.format(" %s in(%s) ", orgField, formatStringArr(orgChild)));
            }
            else if(permissionCond.equals("CREATEMAN")){//建立人
                permissionSQL.append(String.format("(%s='%s')",createManField,AuthenticationUser.getAuthenticationUser().getUserid()));
            }
            else if(permissionCond.equals("CURORGDEPT")){//本部门
                permissionSQL.append(String.format("(%s='%s')",orgDeptField,AuthenticationUser.getAuthenticationUser().getMdeptid()));
            }
            else if(permissionCond.equals("PORGDEPT")){//上级部门
                permissionSQL.append(String.format(" %s in (%s) ", orgDeptField, formatStringArr(orgDeptParent)));
            }
            else if(permissionCond.equals("SORGDEPT")){//下级部门
                permissionSQL.append(String.format(" %s in (%s) ", orgDeptField, formatStringArr(orgDeptChild)));
            }
            else if(permissionCond.equals("ALL")){//全部数据
                  permissionSQL.append("(1=1)");
            }
            else{
                permissionSQL.append(nPermissionSQL);
            }
        }
        if(StringUtils.isEmpty(permissionSQL.toString()))
            return "";
        String resultCond=parseResult(permissionSQL, "OR");
        return  resultCond;
    }

    /**
     * 构造 wrapper
     * @param whereCond
     * @return
     */
    private QueryWrapper getPermissionWrapper(String whereCond){

        QueryWrapper permissionWrapper=new QueryWrapper();
        if(!StringUtils.isEmpty(whereCond)){
            permissionWrapper.apply(whereCond);
        }
        return permissionWrapper;
    }

    /**
     * 获取实体权限字段 orgid/orgsecid
     * @param entityBase
     * @return
     */
    private Map<String,String> getPermissionField(EntityBase entityBase){

        Map<String,String> permissionFiled=new HashMap<>();
        String orgField="orgid";  //组织属性
        String orgDeptField="orgsecid"; //部门属性
        String createManField="createman"; //创建人属性
        String keyField="";//主键属性

        DEFieldCacheMap.getFieldMap(entityBase.getClass().getName());
        Map <Field, DEField> preFields= SearchDEField(entityBase.getClass().getName()); //从缓存中获取当前类预置属性

        for (Map.Entry<Field,DEField> entry : preFields.entrySet()){
            Field preField=entry.getKey();//获取注解字段
            DEField fieldAnnotation=entry.getValue();//获取注解值
            DEPredefinedFieldType prefieldType=fieldAnnotation.preType();
            if(prefieldType==prefieldType.ORGID)//用户配置系统预置属性-组织机构标识
                orgField=preField.getName();
            if(prefieldType==prefieldType.ORGSECTORID)//用户配置系统预置属性-部门标识
                orgDeptField=preField.getName();
            if(fieldAnnotation.isKeyField())//用户配置系统预置属性-部门标识
                keyField=preField.getName();
        }
        permissionFiled.put("orgfield",orgField);
        permissionFiled.put("orgsecfield",orgDeptField);
        permissionFiled.put("createmanfield",createManField);
        permissionFiled.put("keyfield",keyField);
        return permissionFiled;
    }

    /**
     *获取含有@DEField注解的实体属性
     * @param className do对象类名
     * @return
     */
    private Map <Field, DEField> SearchDEField(String className){

        List<Field> fields =  DEFieldCacheMap.getFields(className);
        Map <Field, DEField> deFieldMap =new HashMap<>();
        for(Field field:fields){
            DEField deField=field.getAnnotation(DEField.class);
            if(!ObjectUtils.isEmpty(deField)) {
                deFieldMap.put(field,deField);
            }
        }
        return deFieldMap;
    }

    /**
     * 转换[a,b]格式字符串到 'a','b'格式
     *
     * @return
     */
    private String formatStringArr(JSONArray array) {

        String[] arr = array.toArray(new String[array.size()]);
        return "'" + String.join("','", arr) + "'";
    }

    /**
     * 格式转换
     *
     * @param cond
     * @param operator
     * @return
     */
    private String parseResult(StringBuffer cond, String operator) {

        String resultCond = cond.toString();
        if (resultCond.startsWith(operator))
            resultCond = resultCond.replaceFirst(operator, "");
        if (resultCond.endsWith(operator))
            resultCond = resultCond.substring(0, resultCond.lastIndexOf(operator));
        return resultCond;
    }

}