package cn.ibizlab.api.rest;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.math.BigInteger;
import java.util.HashMap;
import lombok.extern.slf4j.Slf4j;
import com.alibaba.fastjson.JSONObject;
import javax.servlet.ServletRequest;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cglib.beans.BeanCopier;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import org.springframework.http.HttpStatus;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.util.StringUtils;
import org.springframework.context.annotation.Lazy;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.security.access.prepost.PostAuthorize;
import org.springframework.validation.annotation.Validated;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import cn.ibizlab.api.dto.*;
import cn.ibizlab.api.mapping.*;
import cn.ibizlab.core.lite.domain.MetaDataSet;
import cn.ibizlab.core.lite.service.IMetaDataSetService;
import cn.ibizlab.core.lite.filter.MetaDataSetSearchContext;
import cn.ibizlab.util.annotation.VersionCheck;

@Slf4j
@Api(tags = {"数据集" })
@RestController("api-metadataset")
@RequestMapping("")
public class MetaDataSetResource {

    @Autowired
    public IMetaDataSetService metadatasetService;

    @Autowired
    @Lazy
    public MetaDataSetMapping metadatasetMapping;

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-MetaDataSet-Create-all')")
    @ApiOperation(value = "新建数据集", tags = {"数据集" },  notes = "新建数据集")
	@RequestMapping(method = RequestMethod.POST, value = "/metadatasets")
    public ResponseEntity<MetaDataSetDTO> create(@Validated @RequestBody MetaDataSetDTO metadatasetdto) {
        MetaDataSet domain = metadatasetMapping.toDomain(metadatasetdto);
		metadatasetService.create(domain);
        MetaDataSetDTO dto = metadatasetMapping.toDto(domain);
		return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-MetaDataSet-Create-all')")
    @ApiOperation(value = "批量新建数据集", tags = {"数据集" },  notes = "批量新建数据集")
	@RequestMapping(method = RequestMethod.POST, value = "/metadatasets/batch")
    public ResponseEntity<Boolean> createBatch(@RequestBody List<MetaDataSetDTO> metadatasetdtos) {
        metadatasetService.createBatch(metadatasetMapping.toDomain(metadatasetdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-MetaDataSet-Update-all')")
    @ApiOperation(value = "更新数据集", tags = {"数据集" },  notes = "更新数据集")
	@RequestMapping(method = RequestMethod.PUT, value = "/metadatasets/{metadataset_id}")
    public ResponseEntity<MetaDataSetDTO> update(@PathVariable("metadataset_id") String metadataset_id, @RequestBody MetaDataSetDTO metadatasetdto) {
		MetaDataSet domain  = metadatasetMapping.toDomain(metadatasetdto);
        domain .setDatasetId(metadataset_id);
		metadatasetService.update(domain );
		MetaDataSetDTO dto = metadatasetMapping.toDto(domain );
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-MetaDataSet-Update-all')")
    @ApiOperation(value = "批量更新数据集", tags = {"数据集" },  notes = "批量更新数据集")
	@RequestMapping(method = RequestMethod.PUT, value = "/metadatasets/batch")
    public ResponseEntity<Boolean> updateBatch(@RequestBody List<MetaDataSetDTO> metadatasetdtos) {
        metadatasetService.updateBatch(metadatasetMapping.toDomain(metadatasetdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-MetaDataSet-Remove-all')")
    @ApiOperation(value = "删除数据集", tags = {"数据集" },  notes = "删除数据集")
	@RequestMapping(method = RequestMethod.DELETE, value = "/metadatasets/{metadataset_id}")
    public ResponseEntity<Boolean> remove(@PathVariable("metadataset_id") String metadataset_id) {
         return ResponseEntity.status(HttpStatus.OK).body(metadatasetService.remove(metadataset_id));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-MetaDataSet-Remove-all')")
    @ApiOperation(value = "批量删除数据集", tags = {"数据集" },  notes = "批量删除数据集")
	@RequestMapping(method = RequestMethod.DELETE, value = "/metadatasets/batch")
    public ResponseEntity<Boolean> removeBatch(@RequestBody List<String> ids) {
        metadatasetService.removeBatch(ids);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-MetaDataSet-Get-all')")
    @ApiOperation(value = "获取数据集", tags = {"数据集" },  notes = "获取数据集")
	@RequestMapping(method = RequestMethod.GET, value = "/metadatasets/{metadataset_id}")
    public ResponseEntity<MetaDataSetDTO> get(@PathVariable("metadataset_id") String metadataset_id) {
        MetaDataSet domain = metadatasetService.get(metadataset_id);
        MetaDataSetDTO dto = metadatasetMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "获取数据集草稿", tags = {"数据集" },  notes = "获取数据集草稿")
	@RequestMapping(method = RequestMethod.GET, value = "/metadatasets/getdraft")
    public ResponseEntity<MetaDataSetDTO> getDraft(MetaDataSetDTO dto) {
        MetaDataSet domain = metadatasetMapping.toDomain(dto);
        return ResponseEntity.status(HttpStatus.OK).body(metadatasetMapping.toDto(metadatasetService.getDraft(domain)));
    }

    @ApiOperation(value = "检查数据集", tags = {"数据集" },  notes = "检查数据集")
	@RequestMapping(method = RequestMethod.POST, value = "/metadatasets/checkkey")
    public ResponseEntity<Boolean> checkKey(@RequestBody MetaDataSetDTO metadatasetdto) {
        return  ResponseEntity.status(HttpStatus.OK).body(metadatasetService.checkKey(metadatasetMapping.toDomain(metadatasetdto)));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-MetaDataSet-Save-all')")
    @ApiOperation(value = "保存数据集", tags = {"数据集" },  notes = "保存数据集")
	@RequestMapping(method = RequestMethod.POST, value = "/metadatasets/save")
    public ResponseEntity<Boolean> save(@RequestBody MetaDataSetDTO metadatasetdto) {
        return ResponseEntity.status(HttpStatus.OK).body(metadatasetService.save(metadatasetMapping.toDomain(metadatasetdto)));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-MetaDataSet-Save-all')")
    @ApiOperation(value = "批量保存数据集", tags = {"数据集" },  notes = "批量保存数据集")
	@RequestMapping(method = RequestMethod.POST, value = "/metadatasets/savebatch")
    public ResponseEntity<Boolean> saveBatch(@RequestBody List<MetaDataSetDTO> metadatasetdtos) {
        metadatasetService.saveBatch(metadatasetMapping.toDomain(metadatasetdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-MetaDataSet-searchDefault-all')")
	@ApiOperation(value = "获取数据集", tags = {"数据集" } ,notes = "获取数据集")
    @RequestMapping(method= RequestMethod.GET , value="/metadatasets/fetchdefault")
	public ResponseEntity<List<MetaDataSetDTO>> fetchDefault(MetaDataSetSearchContext context) {
        Page<MetaDataSet> domains = metadatasetService.searchDefault(context) ;
        List<MetaDataSetDTO> list = metadatasetMapping.toDto(domains.getContent());
        return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(list);
	}

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-MetaDataSet-searchDefault-all')")
	@ApiOperation(value = "查询数据集", tags = {"数据集" } ,notes = "查询数据集")
    @RequestMapping(method= RequestMethod.POST , value="/metadatasets/searchdefault")
	public ResponseEntity<Page<MetaDataSetDTO>> searchDefault(@RequestBody MetaDataSetSearchContext context) {
        Page<MetaDataSet> domains = metadatasetService.searchDefault(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(metadatasetMapping.toDto(domains.getContent()), context.getPageable(), domains.getTotalElements()));
	}


    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-MetaDataSet-Create-all')")
    @ApiOperation(value = "根据实体建立数据集", tags = {"数据集" },  notes = "根据实体建立数据集")
	@RequestMapping(method = RequestMethod.POST, value = "/metaentities/{metaentity_id}/metadatasets")
    public ResponseEntity<MetaDataSetDTO> createByMetaEntity(@PathVariable("metaentity_id") String metaentity_id, @RequestBody MetaDataSetDTO metadatasetdto) {
        MetaDataSet domain = metadatasetMapping.toDomain(metadatasetdto);
        domain.setEntityId(metaentity_id);
		metadatasetService.create(domain);
        MetaDataSetDTO dto = metadatasetMapping.toDto(domain);
		return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-MetaDataSet-Create-all')")
    @ApiOperation(value = "根据实体批量建立数据集", tags = {"数据集" },  notes = "根据实体批量建立数据集")
	@RequestMapping(method = RequestMethod.POST, value = "/metaentities/{metaentity_id}/metadatasets/batch")
    public ResponseEntity<Boolean> createBatchByMetaEntity(@PathVariable("metaentity_id") String metaentity_id, @RequestBody List<MetaDataSetDTO> metadatasetdtos) {
        List<MetaDataSet> domainlist=metadatasetMapping.toDomain(metadatasetdtos);
        for(MetaDataSet domain:domainlist){
            domain.setEntityId(metaentity_id);
        }
        metadatasetService.createBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-MetaDataSet-Update-all')")
    @ApiOperation(value = "根据实体更新数据集", tags = {"数据集" },  notes = "根据实体更新数据集")
	@RequestMapping(method = RequestMethod.PUT, value = "/metaentities/{metaentity_id}/metadatasets/{metadataset_id}")
    public ResponseEntity<MetaDataSetDTO> updateByMetaEntity(@PathVariable("metaentity_id") String metaentity_id, @PathVariable("metadataset_id") String metadataset_id, @RequestBody MetaDataSetDTO metadatasetdto) {
        MetaDataSet domain = metadatasetMapping.toDomain(metadatasetdto);
        domain.setEntityId(metaentity_id);
        domain.setDatasetId(metadataset_id);
		metadatasetService.update(domain);
        MetaDataSetDTO dto = metadatasetMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-MetaDataSet-Update-all')")
    @ApiOperation(value = "根据实体批量更新数据集", tags = {"数据集" },  notes = "根据实体批量更新数据集")
	@RequestMapping(method = RequestMethod.PUT, value = "/metaentities/{metaentity_id}/metadatasets/batch")
    public ResponseEntity<Boolean> updateBatchByMetaEntity(@PathVariable("metaentity_id") String metaentity_id, @RequestBody List<MetaDataSetDTO> metadatasetdtos) {
        List<MetaDataSet> domainlist=metadatasetMapping.toDomain(metadatasetdtos);
        for(MetaDataSet domain:domainlist){
            domain.setEntityId(metaentity_id);
        }
        metadatasetService.updateBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-MetaDataSet-Remove-all')")
    @ApiOperation(value = "根据实体删除数据集", tags = {"数据集" },  notes = "根据实体删除数据集")
	@RequestMapping(method = RequestMethod.DELETE, value = "/metaentities/{metaentity_id}/metadatasets/{metadataset_id}")
    public ResponseEntity<Boolean> removeByMetaEntity(@PathVariable("metaentity_id") String metaentity_id, @PathVariable("metadataset_id") String metadataset_id) {
		return ResponseEntity.status(HttpStatus.OK).body(metadatasetService.remove(metadataset_id));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-MetaDataSet-Remove-all')")
    @ApiOperation(value = "根据实体批量删除数据集", tags = {"数据集" },  notes = "根据实体批量删除数据集")
	@RequestMapping(method = RequestMethod.DELETE, value = "/metaentities/{metaentity_id}/metadatasets/batch")
    public ResponseEntity<Boolean> removeBatchByMetaEntity(@RequestBody List<String> ids) {
        metadatasetService.removeBatch(ids);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-MetaDataSet-Get-all')")
    @ApiOperation(value = "根据实体获取数据集", tags = {"数据集" },  notes = "根据实体获取数据集")
	@RequestMapping(method = RequestMethod.GET, value = "/metaentities/{metaentity_id}/metadatasets/{metadataset_id}")
    public ResponseEntity<MetaDataSetDTO> getByMetaEntity(@PathVariable("metaentity_id") String metaentity_id, @PathVariable("metadataset_id") String metadataset_id) {
        MetaDataSet domain = metadatasetService.get(metadataset_id);
        MetaDataSetDTO dto = metadatasetMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "根据实体获取数据集草稿", tags = {"数据集" },  notes = "根据实体获取数据集草稿")
    @RequestMapping(method = RequestMethod.GET, value = "/metaentities/{metaentity_id}/metadatasets/getdraft")
    public ResponseEntity<MetaDataSetDTO> getDraftByMetaEntity(@PathVariable("metaentity_id") String metaentity_id, MetaDataSetDTO dto) {
        MetaDataSet domain = metadatasetMapping.toDomain(dto);
        domain.setEntityId(metaentity_id);
        return ResponseEntity.status(HttpStatus.OK).body(metadatasetMapping.toDto(metadatasetService.getDraft(domain)));
    }

    @ApiOperation(value = "根据实体检查数据集", tags = {"数据集" },  notes = "根据实体检查数据集")
	@RequestMapping(method = RequestMethod.POST, value = "/metaentities/{metaentity_id}/metadatasets/checkkey")
    public ResponseEntity<Boolean> checkKeyByMetaEntity(@PathVariable("metaentity_id") String metaentity_id, @RequestBody MetaDataSetDTO metadatasetdto) {
        return  ResponseEntity.status(HttpStatus.OK).body(metadatasetService.checkKey(metadatasetMapping.toDomain(metadatasetdto)));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-MetaDataSet-Save-all')")
    @ApiOperation(value = "根据实体保存数据集", tags = {"数据集" },  notes = "根据实体保存数据集")
	@RequestMapping(method = RequestMethod.POST, value = "/metaentities/{metaentity_id}/metadatasets/save")
    public ResponseEntity<Boolean> saveByMetaEntity(@PathVariable("metaentity_id") String metaentity_id, @RequestBody MetaDataSetDTO metadatasetdto) {
        MetaDataSet domain = metadatasetMapping.toDomain(metadatasetdto);
        domain.setEntityId(metaentity_id);
        return ResponseEntity.status(HttpStatus.OK).body(metadatasetService.save(domain));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-MetaDataSet-Save-all')")
    @ApiOperation(value = "根据实体批量保存数据集", tags = {"数据集" },  notes = "根据实体批量保存数据集")
	@RequestMapping(method = RequestMethod.POST, value = "/metaentities/{metaentity_id}/metadatasets/savebatch")
    public ResponseEntity<Boolean> saveBatchByMetaEntity(@PathVariable("metaentity_id") String metaentity_id, @RequestBody List<MetaDataSetDTO> metadatasetdtos) {
        List<MetaDataSet> domainlist=metadatasetMapping.toDomain(metadatasetdtos);
        for(MetaDataSet domain:domainlist){
             domain.setEntityId(metaentity_id);
        }
        metadatasetService.saveBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-MetaDataSet-searchDefault-all')")
	@ApiOperation(value = "根据实体获取数据集", tags = {"数据集" } ,notes = "根据实体获取数据集")
    @RequestMapping(method= RequestMethod.GET , value="/metaentities/{metaentity_id}/metadatasets/fetchdefault")
	public ResponseEntity<List<MetaDataSetDTO>> fetchMetaDataSetDefaultByMetaEntity(@PathVariable("metaentity_id") String metaentity_id,MetaDataSetSearchContext context) {
        context.setN_entityid_eq(metaentity_id);
        Page<MetaDataSet> domains = metadatasetService.searchDefault(context) ;
        List<MetaDataSetDTO> list = metadatasetMapping.toDto(domains.getContent());
	    return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(list);
	}

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-MetaDataSet-searchDefault-all')")
	@ApiOperation(value = "根据实体查询数据集", tags = {"数据集" } ,notes = "根据实体查询数据集")
    @RequestMapping(method= RequestMethod.POST , value="/metaentities/{metaentity_id}/metadatasets/searchdefault")
	public ResponseEntity<Page<MetaDataSetDTO>> searchMetaDataSetDefaultByMetaEntity(@PathVariable("metaentity_id") String metaentity_id, @RequestBody MetaDataSetSearchContext context) {
        context.setN_entityid_eq(metaentity_id);
        Page<MetaDataSet> domains = metadatasetService.searchDefault(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(metadatasetMapping.toDto(domains.getContent()), context.getPageable(), domains.getTotalElements()));
	}
    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-MetaDataSet-Create-all')")
    @ApiOperation(value = "根据系统实体建立数据集", tags = {"数据集" },  notes = "根据系统实体建立数据集")
	@RequestMapping(method = RequestMethod.POST, value = "/dstsystems/{dstsystem_id}/metaentities/{metaentity_id}/metadatasets")
    public ResponseEntity<MetaDataSetDTO> createByDstSystemMetaEntity(@PathVariable("dstsystem_id") String dstsystem_id, @PathVariable("metaentity_id") String metaentity_id, @RequestBody MetaDataSetDTO metadatasetdto) {
        MetaDataSet domain = metadatasetMapping.toDomain(metadatasetdto);
        domain.setEntityId(metaentity_id);
		metadatasetService.create(domain);
        MetaDataSetDTO dto = metadatasetMapping.toDto(domain);
		return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-MetaDataSet-Create-all')")
    @ApiOperation(value = "根据系统实体批量建立数据集", tags = {"数据集" },  notes = "根据系统实体批量建立数据集")
	@RequestMapping(method = RequestMethod.POST, value = "/dstsystems/{dstsystem_id}/metaentities/{metaentity_id}/metadatasets/batch")
    public ResponseEntity<Boolean> createBatchByDstSystemMetaEntity(@PathVariable("dstsystem_id") String dstsystem_id, @PathVariable("metaentity_id") String metaentity_id, @RequestBody List<MetaDataSetDTO> metadatasetdtos) {
        List<MetaDataSet> domainlist=metadatasetMapping.toDomain(metadatasetdtos);
        for(MetaDataSet domain:domainlist){
            domain.setEntityId(metaentity_id);
        }
        metadatasetService.createBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-MetaDataSet-Update-all')")
    @ApiOperation(value = "根据系统实体更新数据集", tags = {"数据集" },  notes = "根据系统实体更新数据集")
	@RequestMapping(method = RequestMethod.PUT, value = "/dstsystems/{dstsystem_id}/metaentities/{metaentity_id}/metadatasets/{metadataset_id}")
    public ResponseEntity<MetaDataSetDTO> updateByDstSystemMetaEntity(@PathVariable("dstsystem_id") String dstsystem_id, @PathVariable("metaentity_id") String metaentity_id, @PathVariable("metadataset_id") String metadataset_id, @RequestBody MetaDataSetDTO metadatasetdto) {
        MetaDataSet domain = metadatasetMapping.toDomain(metadatasetdto);
        domain.setEntityId(metaentity_id);
        domain.setDatasetId(metadataset_id);
		metadatasetService.update(domain);
        MetaDataSetDTO dto = metadatasetMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-MetaDataSet-Update-all')")
    @ApiOperation(value = "根据系统实体批量更新数据集", tags = {"数据集" },  notes = "根据系统实体批量更新数据集")
	@RequestMapping(method = RequestMethod.PUT, value = "/dstsystems/{dstsystem_id}/metaentities/{metaentity_id}/metadatasets/batch")
    public ResponseEntity<Boolean> updateBatchByDstSystemMetaEntity(@PathVariable("dstsystem_id") String dstsystem_id, @PathVariable("metaentity_id") String metaentity_id, @RequestBody List<MetaDataSetDTO> metadatasetdtos) {
        List<MetaDataSet> domainlist=metadatasetMapping.toDomain(metadatasetdtos);
        for(MetaDataSet domain:domainlist){
            domain.setEntityId(metaentity_id);
        }
        metadatasetService.updateBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-MetaDataSet-Remove-all')")
    @ApiOperation(value = "根据系统实体删除数据集", tags = {"数据集" },  notes = "根据系统实体删除数据集")
	@RequestMapping(method = RequestMethod.DELETE, value = "/dstsystems/{dstsystem_id}/metaentities/{metaentity_id}/metadatasets/{metadataset_id}")
    public ResponseEntity<Boolean> removeByDstSystemMetaEntity(@PathVariable("dstsystem_id") String dstsystem_id, @PathVariable("metaentity_id") String metaentity_id, @PathVariable("metadataset_id") String metadataset_id) {
		return ResponseEntity.status(HttpStatus.OK).body(metadatasetService.remove(metadataset_id));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-MetaDataSet-Remove-all')")
    @ApiOperation(value = "根据系统实体批量删除数据集", tags = {"数据集" },  notes = "根据系统实体批量删除数据集")
	@RequestMapping(method = RequestMethod.DELETE, value = "/dstsystems/{dstsystem_id}/metaentities/{metaentity_id}/metadatasets/batch")
    public ResponseEntity<Boolean> removeBatchByDstSystemMetaEntity(@RequestBody List<String> ids) {
        metadatasetService.removeBatch(ids);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-MetaDataSet-Get-all')")
    @ApiOperation(value = "根据系统实体获取数据集", tags = {"数据集" },  notes = "根据系统实体获取数据集")
	@RequestMapping(method = RequestMethod.GET, value = "/dstsystems/{dstsystem_id}/metaentities/{metaentity_id}/metadatasets/{metadataset_id}")
    public ResponseEntity<MetaDataSetDTO> getByDstSystemMetaEntity(@PathVariable("dstsystem_id") String dstsystem_id, @PathVariable("metaentity_id") String metaentity_id, @PathVariable("metadataset_id") String metadataset_id) {
        MetaDataSet domain = metadatasetService.get(metadataset_id);
        MetaDataSetDTO dto = metadatasetMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "根据系统实体获取数据集草稿", tags = {"数据集" },  notes = "根据系统实体获取数据集草稿")
    @RequestMapping(method = RequestMethod.GET, value = "/dstsystems/{dstsystem_id}/metaentities/{metaentity_id}/metadatasets/getdraft")
    public ResponseEntity<MetaDataSetDTO> getDraftByDstSystemMetaEntity(@PathVariable("dstsystem_id") String dstsystem_id, @PathVariable("metaentity_id") String metaentity_id, MetaDataSetDTO dto) {
        MetaDataSet domain = metadatasetMapping.toDomain(dto);
        domain.setEntityId(metaentity_id);
        return ResponseEntity.status(HttpStatus.OK).body(metadatasetMapping.toDto(metadatasetService.getDraft(domain)));
    }

    @ApiOperation(value = "根据系统实体检查数据集", tags = {"数据集" },  notes = "根据系统实体检查数据集")
	@RequestMapping(method = RequestMethod.POST, value = "/dstsystems/{dstsystem_id}/metaentities/{metaentity_id}/metadatasets/checkkey")
    public ResponseEntity<Boolean> checkKeyByDstSystemMetaEntity(@PathVariable("dstsystem_id") String dstsystem_id, @PathVariable("metaentity_id") String metaentity_id, @RequestBody MetaDataSetDTO metadatasetdto) {
        return  ResponseEntity.status(HttpStatus.OK).body(metadatasetService.checkKey(metadatasetMapping.toDomain(metadatasetdto)));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-MetaDataSet-Save-all')")
    @ApiOperation(value = "根据系统实体保存数据集", tags = {"数据集" },  notes = "根据系统实体保存数据集")
	@RequestMapping(method = RequestMethod.POST, value = "/dstsystems/{dstsystem_id}/metaentities/{metaentity_id}/metadatasets/save")
    public ResponseEntity<Boolean> saveByDstSystemMetaEntity(@PathVariable("dstsystem_id") String dstsystem_id, @PathVariable("metaentity_id") String metaentity_id, @RequestBody MetaDataSetDTO metadatasetdto) {
        MetaDataSet domain = metadatasetMapping.toDomain(metadatasetdto);
        domain.setEntityId(metaentity_id);
        return ResponseEntity.status(HttpStatus.OK).body(metadatasetService.save(domain));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-MetaDataSet-Save-all')")
    @ApiOperation(value = "根据系统实体批量保存数据集", tags = {"数据集" },  notes = "根据系统实体批量保存数据集")
	@RequestMapping(method = RequestMethod.POST, value = "/dstsystems/{dstsystem_id}/metaentities/{metaentity_id}/metadatasets/savebatch")
    public ResponseEntity<Boolean> saveBatchByDstSystemMetaEntity(@PathVariable("dstsystem_id") String dstsystem_id, @PathVariable("metaentity_id") String metaentity_id, @RequestBody List<MetaDataSetDTO> metadatasetdtos) {
        List<MetaDataSet> domainlist=metadatasetMapping.toDomain(metadatasetdtos);
        for(MetaDataSet domain:domainlist){
             domain.setEntityId(metaentity_id);
        }
        metadatasetService.saveBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-MetaDataSet-searchDefault-all')")
	@ApiOperation(value = "根据系统实体获取数据集", tags = {"数据集" } ,notes = "根据系统实体获取数据集")
    @RequestMapping(method= RequestMethod.GET , value="/dstsystems/{dstsystem_id}/metaentities/{metaentity_id}/metadatasets/fetchdefault")
	public ResponseEntity<List<MetaDataSetDTO>> fetchMetaDataSetDefaultByDstSystemMetaEntity(@PathVariable("dstsystem_id") String dstsystem_id, @PathVariable("metaentity_id") String metaentity_id,MetaDataSetSearchContext context) {
        context.setN_entityid_eq(metaentity_id);
        Page<MetaDataSet> domains = metadatasetService.searchDefault(context) ;
        List<MetaDataSetDTO> list = metadatasetMapping.toDto(domains.getContent());
	    return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(list);
	}

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-MetaDataSet-searchDefault-all')")
	@ApiOperation(value = "根据系统实体查询数据集", tags = {"数据集" } ,notes = "根据系统实体查询数据集")
    @RequestMapping(method= RequestMethod.POST , value="/dstsystems/{dstsystem_id}/metaentities/{metaentity_id}/metadatasets/searchdefault")
	public ResponseEntity<Page<MetaDataSetDTO>> searchMetaDataSetDefaultByDstSystemMetaEntity(@PathVariable("dstsystem_id") String dstsystem_id, @PathVariable("metaentity_id") String metaentity_id, @RequestBody MetaDataSetSearchContext context) {
        context.setN_entityid_eq(metaentity_id);
        Page<MetaDataSet> domains = metadatasetService.searchDefault(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(metadatasetMapping.toDto(domains.getContent()), context.getPageable(), domains.getTotalElements()));
	}
}

