package cn.ibizlab.api.rest;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.math.BigInteger;
import java.util.HashMap;
import lombok.extern.slf4j.Slf4j;
import com.alibaba.fastjson.JSONObject;
import javax.servlet.ServletRequest;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cglib.beans.BeanCopier;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import org.springframework.http.HttpStatus;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.util.StringUtils;
import org.springframework.context.annotation.Lazy;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.security.access.prepost.PostAuthorize;
import org.springframework.validation.annotation.Validated;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import cn.ibizlab.api.dto.*;
import cn.ibizlab.api.mapping.*;
import cn.ibizlab.core.lite.domain.DstConfig;
import cn.ibizlab.core.lite.service.IDstConfigService;
import cn.ibizlab.core.lite.filter.DstConfigSearchContext;
import cn.ibizlab.util.annotation.VersionCheck;

@Slf4j
@Api(tags = {"配置" })
@RestController("api-dstconfig")
@RequestMapping("")
public class DstConfigResource {

    @Autowired
    public IDstConfigService dstconfigService;

    @Autowired
    @Lazy
    public DstConfigMapping dstconfigMapping;

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstConfig-Create-all')")
    @ApiOperation(value = "新建配置", tags = {"配置" },  notes = "新建配置")
	@RequestMapping(method = RequestMethod.POST, value = "/dstconfigs")
    public ResponseEntity<DstConfigDTO> create(@Validated @RequestBody DstConfigDTO dstconfigdto) {
        DstConfig domain = dstconfigMapping.toDomain(dstconfigdto);
		dstconfigService.create(domain);
        DstConfigDTO dto = dstconfigMapping.toDto(domain);
		return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstConfig-Create-all')")
    @ApiOperation(value = "批量新建配置", tags = {"配置" },  notes = "批量新建配置")
	@RequestMapping(method = RequestMethod.POST, value = "/dstconfigs/batch")
    public ResponseEntity<Boolean> createBatch(@RequestBody List<DstConfigDTO> dstconfigdtos) {
        dstconfigService.createBatch(dstconfigMapping.toDomain(dstconfigdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @VersionCheck(entity = "dstconfig" , versionfield = "updateDate")
    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstConfig-Update-all')")
    @ApiOperation(value = "更新配置", tags = {"配置" },  notes = "更新配置")
	@RequestMapping(method = RequestMethod.PUT, value = "/dstconfigs/{dstconfig_id}")
    public ResponseEntity<DstConfigDTO> update(@PathVariable("dstconfig_id") String dstconfig_id, @RequestBody DstConfigDTO dstconfigdto) {
		DstConfig domain  = dstconfigMapping.toDomain(dstconfigdto);
        domain .setCfgId(dstconfig_id);
		dstconfigService.update(domain );
		DstConfigDTO dto = dstconfigMapping.toDto(domain );
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstConfig-Update-all')")
    @ApiOperation(value = "批量更新配置", tags = {"配置" },  notes = "批量更新配置")
	@RequestMapping(method = RequestMethod.PUT, value = "/dstconfigs/batch")
    public ResponseEntity<Boolean> updateBatch(@RequestBody List<DstConfigDTO> dstconfigdtos) {
        dstconfigService.updateBatch(dstconfigMapping.toDomain(dstconfigdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstConfig-Remove-all')")
    @ApiOperation(value = "删除配置", tags = {"配置" },  notes = "删除配置")
	@RequestMapping(method = RequestMethod.DELETE, value = "/dstconfigs/{dstconfig_id}")
    public ResponseEntity<Boolean> remove(@PathVariable("dstconfig_id") String dstconfig_id) {
         return ResponseEntity.status(HttpStatus.OK).body(dstconfigService.remove(dstconfig_id));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstConfig-Remove-all')")
    @ApiOperation(value = "批量删除配置", tags = {"配置" },  notes = "批量删除配置")
	@RequestMapping(method = RequestMethod.DELETE, value = "/dstconfigs/batch")
    public ResponseEntity<Boolean> removeBatch(@RequestBody List<String> ids) {
        dstconfigService.removeBatch(ids);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstConfig-Get-all')")
    @ApiOperation(value = "获取配置", tags = {"配置" },  notes = "获取配置")
	@RequestMapping(method = RequestMethod.GET, value = "/dstconfigs/{dstconfig_id}")
    public ResponseEntity<DstConfigDTO> get(@PathVariable("dstconfig_id") String dstconfig_id) {
        DstConfig domain = dstconfigService.get(dstconfig_id);
        DstConfigDTO dto = dstconfigMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "获取配置草稿", tags = {"配置" },  notes = "获取配置草稿")
	@RequestMapping(method = RequestMethod.GET, value = "/dstconfigs/getdraft")
    public ResponseEntity<DstConfigDTO> getDraft(DstConfigDTO dto) {
        DstConfig domain = dstconfigMapping.toDomain(dto);
        return ResponseEntity.status(HttpStatus.OK).body(dstconfigMapping.toDto(dstconfigService.getDraft(domain)));
    }

    @ApiOperation(value = "检查配置", tags = {"配置" },  notes = "检查配置")
	@RequestMapping(method = RequestMethod.POST, value = "/dstconfigs/checkkey")
    public ResponseEntity<Boolean> checkKey(@RequestBody DstConfigDTO dstconfigdto) {
        return  ResponseEntity.status(HttpStatus.OK).body(dstconfigService.checkKey(dstconfigMapping.toDomain(dstconfigdto)));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstConfig-Reset-all')")
    @ApiOperation(value = "reset", tags = {"配置" },  notes = "reset")
	@RequestMapping(method = RequestMethod.POST, value = "/dstconfigs/{dstconfig_id}/reset")
    public ResponseEntity<DstConfigDTO> reset(@PathVariable("dstconfig_id") String dstconfig_id, @RequestBody DstConfigDTO dstconfigdto) {
        DstConfig domain = dstconfigMapping.toDomain(dstconfigdto);
        domain.setCfgId(dstconfig_id);
        domain = dstconfigService.reset(domain);
        dstconfigdto = dstconfigMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dstconfigdto);
    }
    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstConfig-Reset-all')")
    @ApiOperation(value = "批量处理[reset]", tags = {"配置" },  notes = "批量处理[reset]")
	@RequestMapping(method = RequestMethod.POST, value = "/dstconfigs/resetbatch")
    public ResponseEntity<Boolean> resetBatch(@RequestBody List<DstConfigDTO> dstconfigdtos) {
        List<DstConfig> domains = dstconfigMapping.toDomain(dstconfigdtos);
        boolean result = dstconfigService.resetBatch(domains);
        return ResponseEntity.status(HttpStatus.OK).body(result);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstConfig-Save-all')")
    @ApiOperation(value = "保存配置", tags = {"配置" },  notes = "保存配置")
	@RequestMapping(method = RequestMethod.POST, value = "/dstconfigs/save")
    public ResponseEntity<Boolean> save(@RequestBody DstConfigDTO dstconfigdto) {
        return ResponseEntity.status(HttpStatus.OK).body(dstconfigService.save(dstconfigMapping.toDomain(dstconfigdto)));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstConfig-Save-all')")
    @ApiOperation(value = "批量保存配置", tags = {"配置" },  notes = "批量保存配置")
	@RequestMapping(method = RequestMethod.POST, value = "/dstconfigs/savebatch")
    public ResponseEntity<Boolean> saveBatch(@RequestBody List<DstConfigDTO> dstconfigdtos) {
        dstconfigService.saveBatch(dstconfigMapping.toDomain(dstconfigdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstConfig-searchDefault-all')")
	@ApiOperation(value = "获取数据集", tags = {"配置" } ,notes = "获取数据集")
    @RequestMapping(method= RequestMethod.GET , value="/dstconfigs/fetchdefault")
	public ResponseEntity<List<DstConfigDTO>> fetchDefault(DstConfigSearchContext context) {
        Page<DstConfig> domains = dstconfigService.searchDefault(context) ;
        List<DstConfigDTO> list = dstconfigMapping.toDto(domains.getContent());
        return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(list);
	}

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstConfig-searchDefault-all')")
	@ApiOperation(value = "查询数据集", tags = {"配置" } ,notes = "查询数据集")
    @RequestMapping(method= RequestMethod.POST , value="/dstconfigs/searchdefault")
	public ResponseEntity<Page<DstConfigDTO>> searchDefault(@RequestBody DstConfigSearchContext context) {
        Page<DstConfig> domains = dstconfigService.searchDefault(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(dstconfigMapping.toDto(domains.getContent()), context.getPageable(), domains.getTotalElements()));
	}


}

