package cn.ibizlab.api.rest;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.math.BigInteger;
import java.util.HashMap;
import lombok.extern.slf4j.Slf4j;
import com.alibaba.fastjson.JSONObject;
import javax.servlet.ServletRequest;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cglib.beans.BeanCopier;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import org.springframework.http.HttpStatus;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.util.StringUtils;
import org.springframework.context.annotation.Lazy;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.security.access.prepost.PostAuthorize;
import org.springframework.validation.annotation.Validated;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import cn.ibizlab.api.dto.*;
import cn.ibizlab.api.mapping.*;
import cn.ibizlab.core.avuedata.domain.BladeVisual;
import cn.ibizlab.core.avuedata.service.IBladeVisualService;
import cn.ibizlab.core.avuedata.filter.BladeVisualSearchContext;
import cn.ibizlab.util.annotation.VersionCheck;

@Slf4j
@Api(tags = {"可视化表" })
@RestController("api-bladevisual")
@RequestMapping("")
public class BladeVisualResource {

    @Autowired
    public IBladeVisualService bladevisualService;

    @Autowired
    @Lazy
    public BladeVisualMapping bladevisualMapping;

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-BladeVisual-Create-all')")
    @ApiOperation(value = "新建可视化表", tags = {"可视化表" },  notes = "新建可视化表")
	@RequestMapping(method = RequestMethod.POST, value = "/bladevisuals")
    public ResponseEntity<BladeVisualDTO> create(@Validated @RequestBody BladeVisualDTO bladevisualdto) {
        BladeVisual domain = bladevisualMapping.toDomain(bladevisualdto);
		bladevisualService.create(domain);
        BladeVisualDTO dto = bladevisualMapping.toDto(domain);
		return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-BladeVisual-Create-all')")
    @ApiOperation(value = "批量新建可视化表", tags = {"可视化表" },  notes = "批量新建可视化表")
	@RequestMapping(method = RequestMethod.POST, value = "/bladevisuals/batch")
    public ResponseEntity<Boolean> createBatch(@RequestBody List<BladeVisualDTO> bladevisualdtos) {
        bladevisualService.createBatch(bladevisualMapping.toDomain(bladevisualdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @VersionCheck(entity = "bladevisual" , versionfield = "updatetime")
    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-BladeVisual-Update-all')")
    @ApiOperation(value = "更新可视化表", tags = {"可视化表" },  notes = "更新可视化表")
	@RequestMapping(method = RequestMethod.PUT, value = "/bladevisuals/{bladevisual_id}")
    public ResponseEntity<BladeVisualDTO> update(@PathVariable("bladevisual_id") Long bladevisual_id, @RequestBody BladeVisualDTO bladevisualdto) {
		BladeVisual domain  = bladevisualMapping.toDomain(bladevisualdto);
        domain .setId(bladevisual_id);
		bladevisualService.update(domain );
		BladeVisualDTO dto = bladevisualMapping.toDto(domain );
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-BladeVisual-Update-all')")
    @ApiOperation(value = "批量更新可视化表", tags = {"可视化表" },  notes = "批量更新可视化表")
	@RequestMapping(method = RequestMethod.PUT, value = "/bladevisuals/batch")
    public ResponseEntity<Boolean> updateBatch(@RequestBody List<BladeVisualDTO> bladevisualdtos) {
        bladevisualService.updateBatch(bladevisualMapping.toDomain(bladevisualdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-BladeVisual-Remove-all')")
    @ApiOperation(value = "删除可视化表", tags = {"可视化表" },  notes = "删除可视化表")
	@RequestMapping(method = RequestMethod.DELETE, value = "/bladevisuals/{bladevisual_id}")
    public ResponseEntity<Boolean> remove(@PathVariable("bladevisual_id") Long bladevisual_id) {
         return ResponseEntity.status(HttpStatus.OK).body(bladevisualService.remove(bladevisual_id));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-BladeVisual-Remove-all')")
    @ApiOperation(value = "批量删除可视化表", tags = {"可视化表" },  notes = "批量删除可视化表")
	@RequestMapping(method = RequestMethod.DELETE, value = "/bladevisuals/batch")
    public ResponseEntity<Boolean> removeBatch(@RequestBody List<Long> ids) {
        bladevisualService.removeBatch(ids);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-BladeVisual-Get-all')")
    @ApiOperation(value = "获取可视化表", tags = {"可视化表" },  notes = "获取可视化表")
	@RequestMapping(method = RequestMethod.GET, value = "/bladevisuals/{bladevisual_id}")
    public ResponseEntity<BladeVisualDTO> get(@PathVariable("bladevisual_id") Long bladevisual_id) {
        BladeVisual domain = bladevisualService.get(bladevisual_id);
        BladeVisualDTO dto = bladevisualMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "获取可视化表草稿", tags = {"可视化表" },  notes = "获取可视化表草稿")
	@RequestMapping(method = RequestMethod.GET, value = "/bladevisuals/getdraft")
    public ResponseEntity<BladeVisualDTO> getDraft() {
        return ResponseEntity.status(HttpStatus.OK).body(bladevisualMapping.toDto(bladevisualService.getDraft(new BladeVisual())));
    }

    @ApiOperation(value = "检查可视化表", tags = {"可视化表" },  notes = "检查可视化表")
	@RequestMapping(method = RequestMethod.POST, value = "/bladevisuals/checkkey")
    public ResponseEntity<Boolean> checkKey(@RequestBody BladeVisualDTO bladevisualdto) {
        return  ResponseEntity.status(HttpStatus.OK).body(bladevisualService.checkKey(bladevisualMapping.toDomain(bladevisualdto)));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-BladeVisual-Save-all')")
    @ApiOperation(value = "保存可视化表", tags = {"可视化表" },  notes = "保存可视化表")
	@RequestMapping(method = RequestMethod.POST, value = "/bladevisuals/save")
    public ResponseEntity<Boolean> save(@RequestBody BladeVisualDTO bladevisualdto) {
        return ResponseEntity.status(HttpStatus.OK).body(bladevisualService.save(bladevisualMapping.toDomain(bladevisualdto)));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-BladeVisual-Save-all')")
    @ApiOperation(value = "批量保存可视化表", tags = {"可视化表" },  notes = "批量保存可视化表")
	@RequestMapping(method = RequestMethod.POST, value = "/bladevisuals/savebatch")
    public ResponseEntity<Boolean> saveBatch(@RequestBody List<BladeVisualDTO> bladevisualdtos) {
        bladevisualService.saveBatch(bladevisualMapping.toDomain(bladevisualdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-BladeVisual-searchDefault-all')")
	@ApiOperation(value = "获取数据集", tags = {"可视化表" } ,notes = "获取数据集")
    @RequestMapping(method= RequestMethod.GET , value="/bladevisuals/fetchdefault")
	public ResponseEntity<List<BladeVisualDTO>> fetchDefault(BladeVisualSearchContext context) {
        Page<BladeVisual> domains = bladevisualService.searchDefault(context) ;
        List<BladeVisualDTO> list = bladevisualMapping.toDto(domains.getContent());
        return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(list);
	}

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-BladeVisual-searchDefault-all')")
	@ApiOperation(value = "查询数据集", tags = {"可视化表" } ,notes = "查询数据集")
    @RequestMapping(method= RequestMethod.POST , value="/bladevisuals/searchdefault")
	public ResponseEntity<Page<BladeVisualDTO>> searchDefault(@RequestBody BladeVisualSearchContext context) {
        Page<BladeVisual> domains = bladevisualService.searchDefault(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(bladevisualMapping.toDto(domains.getContent()), context.getPageable(), domains.getTotalElements()));
	}


}

