package cn.ibizlab.api.rest;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.math.BigInteger;
import java.util.HashMap;
import lombok.extern.slf4j.Slf4j;
import com.alibaba.fastjson.JSONObject;
import javax.servlet.ServletRequest;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cglib.beans.BeanCopier;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import org.springframework.http.HttpStatus;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.util.StringUtils;
import org.springframework.context.annotation.Lazy;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.security.access.prepost.PostAuthorize;
import org.springframework.validation.annotation.Validated;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import cn.ibizlab.api.dto.*;
import cn.ibizlab.api.mapping.*;
import cn.ibizlab.core.lite.domain.DstComponent;
import cn.ibizlab.core.lite.service.IDstComponentService;
import cn.ibizlab.core.lite.filter.DstComponentSearchContext;
import cn.ibizlab.util.annotation.VersionCheck;

@Slf4j
@Api(tags = {"组件" })
@RestController("api-dstcomponent")
@RequestMapping("")
public class DstComponentResource {

    @Autowired
    public IDstComponentService dstcomponentService;

    @Autowired
    @Lazy
    public DstComponentMapping dstcomponentMapping;

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstComponent-Create-all')")
    @ApiOperation(value = "新建组件", tags = {"组件" },  notes = "新建组件")
	@RequestMapping(method = RequestMethod.POST, value = "/dstcomponents")
    public ResponseEntity<DstComponentDTO> create(@Validated @RequestBody DstComponentDTO dstcomponentdto) {
        DstComponent domain = dstcomponentMapping.toDomain(dstcomponentdto);
		dstcomponentService.create(domain);
        DstComponentDTO dto = dstcomponentMapping.toDto(domain);
		return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstComponent-Create-all')")
    @ApiOperation(value = "批量新建组件", tags = {"组件" },  notes = "批量新建组件")
	@RequestMapping(method = RequestMethod.POST, value = "/dstcomponents/batch")
    public ResponseEntity<Boolean> createBatch(@RequestBody List<DstComponentDTO> dstcomponentdtos) {
        dstcomponentService.createBatch(dstcomponentMapping.toDomain(dstcomponentdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @VersionCheck(entity = "dstcomponent" , versionfield = "updatedate")
    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstComponent-Update-all')")
    @ApiOperation(value = "更新组件", tags = {"组件" },  notes = "更新组件")
	@RequestMapping(method = RequestMethod.PUT, value = "/dstcomponents/{dstcomponent_id}")
    public ResponseEntity<DstComponentDTO> update(@PathVariable("dstcomponent_id") String dstcomponent_id, @RequestBody DstComponentDTO dstcomponentdto) {
		DstComponent domain  = dstcomponentMapping.toDomain(dstcomponentdto);
        domain .setId(dstcomponent_id);
		dstcomponentService.update(domain );
		DstComponentDTO dto = dstcomponentMapping.toDto(domain );
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstComponent-Update-all')")
    @ApiOperation(value = "批量更新组件", tags = {"组件" },  notes = "批量更新组件")
	@RequestMapping(method = RequestMethod.PUT, value = "/dstcomponents/batch")
    public ResponseEntity<Boolean> updateBatch(@RequestBody List<DstComponentDTO> dstcomponentdtos) {
        dstcomponentService.updateBatch(dstcomponentMapping.toDomain(dstcomponentdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstComponent-Remove-all')")
    @ApiOperation(value = "删除组件", tags = {"组件" },  notes = "删除组件")
	@RequestMapping(method = RequestMethod.DELETE, value = "/dstcomponents/{dstcomponent_id}")
    public ResponseEntity<Boolean> remove(@PathVariable("dstcomponent_id") String dstcomponent_id) {
         return ResponseEntity.status(HttpStatus.OK).body(dstcomponentService.remove(dstcomponent_id));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstComponent-Remove-all')")
    @ApiOperation(value = "批量删除组件", tags = {"组件" },  notes = "批量删除组件")
	@RequestMapping(method = RequestMethod.DELETE, value = "/dstcomponents/batch")
    public ResponseEntity<Boolean> removeBatch(@RequestBody List<String> ids) {
        dstcomponentService.removeBatch(ids);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstComponent-Get-all')")
    @ApiOperation(value = "获取组件", tags = {"组件" },  notes = "获取组件")
	@RequestMapping(method = RequestMethod.GET, value = "/dstcomponents/{dstcomponent_id}")
    public ResponseEntity<DstComponentDTO> get(@PathVariable("dstcomponent_id") String dstcomponent_id) {
        DstComponent domain = dstcomponentService.get(dstcomponent_id);
        DstComponentDTO dto = dstcomponentMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "获取组件草稿", tags = {"组件" },  notes = "获取组件草稿")
	@RequestMapping(method = RequestMethod.GET, value = "/dstcomponents/getdraft")
    public ResponseEntity<DstComponentDTO> getDraft() {
        return ResponseEntity.status(HttpStatus.OK).body(dstcomponentMapping.toDto(dstcomponentService.getDraft(new DstComponent())));
    }

    @ApiOperation(value = "检查组件", tags = {"组件" },  notes = "检查组件")
	@RequestMapping(method = RequestMethod.POST, value = "/dstcomponents/checkkey")
    public ResponseEntity<Boolean> checkKey(@RequestBody DstComponentDTO dstcomponentdto) {
        return  ResponseEntity.status(HttpStatus.OK).body(dstcomponentService.checkKey(dstcomponentMapping.toDomain(dstcomponentdto)));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstComponent-Save-all')")
    @ApiOperation(value = "保存组件", tags = {"组件" },  notes = "保存组件")
	@RequestMapping(method = RequestMethod.POST, value = "/dstcomponents/save")
    public ResponseEntity<Boolean> save(@RequestBody DstComponentDTO dstcomponentdto) {
        return ResponseEntity.status(HttpStatus.OK).body(dstcomponentService.save(dstcomponentMapping.toDomain(dstcomponentdto)));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstComponent-Save-all')")
    @ApiOperation(value = "批量保存组件", tags = {"组件" },  notes = "批量保存组件")
	@RequestMapping(method = RequestMethod.POST, value = "/dstcomponents/savebatch")
    public ResponseEntity<Boolean> saveBatch(@RequestBody List<DstComponentDTO> dstcomponentdtos) {
        dstcomponentService.saveBatch(dstcomponentMapping.toDomain(dstcomponentdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstComponent-Sync-all')")
    @ApiOperation(value = "Sync", tags = {"组件" },  notes = "Sync")
	@RequestMapping(method = RequestMethod.POST, value = "/dstcomponents/{dstcomponent_id}/sync")
    public ResponseEntity<DstComponentDTO> sync(@PathVariable("dstcomponent_id") String dstcomponent_id, @RequestBody DstComponentDTO dstcomponentdto) {
        DstComponent domain = dstcomponentMapping.toDomain(dstcomponentdto);
        domain.setId(dstcomponent_id);
        domain = dstcomponentService.sync(domain);
        dstcomponentdto = dstcomponentMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dstcomponentdto);
    }
    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstComponent-Sync-all')")
    @ApiOperation(value = "批量处理[Sync]", tags = {"组件" },  notes = "批量处理[Sync]")
	@RequestMapping(method = RequestMethod.POST, value = "/dstcomponents/syncbatch")
    public ResponseEntity<Boolean> syncBatch(@RequestBody List<DstComponentDTO> dstcomponentdtos) {
        List<DstComponent> domains = dstcomponentMapping.toDomain(dstcomponentdtos);
        boolean result = dstcomponentService.syncBatch(domains);
        return ResponseEntity.status(HttpStatus.OK).body(result);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstComponent-searchDefault-all')")
	@ApiOperation(value = "获取数据集", tags = {"组件" } ,notes = "获取数据集")
    @RequestMapping(method= RequestMethod.GET , value="/dstcomponents/fetchdefault")
	public ResponseEntity<List<DstComponentDTO>> fetchDefault(DstComponentSearchContext context) {
        Page<DstComponent> domains = dstcomponentService.searchDefault(context) ;
        List<DstComponentDTO> list = dstcomponentMapping.toDto(domains.getContent());
        return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(list);
	}

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstComponent-searchDefault-all')")
	@ApiOperation(value = "查询数据集", tags = {"组件" } ,notes = "查询数据集")
    @RequestMapping(method= RequestMethod.POST , value="/dstcomponents/searchdefault")
	public ResponseEntity<Page<DstComponentDTO>> searchDefault(@RequestBody DstComponentSearchContext context) {
        Page<DstComponent> domains = dstcomponentService.searchDefault(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(dstcomponentMapping.toDto(domains.getContent()), context.getPageable(), domains.getTotalElements()));
	}


    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstComponent-Create-all')")
    @ApiOperation(value = "根据应用建立组件", tags = {"组件" },  notes = "根据应用建立组件")
	@RequestMapping(method = RequestMethod.POST, value = "/dstapps/{dstapp_id}/dstcomponents")
    public ResponseEntity<DstComponentDTO> createByDstApp(@PathVariable("dstapp_id") String dstapp_id, @RequestBody DstComponentDTO dstcomponentdto) {
        DstComponent domain = dstcomponentMapping.toDomain(dstcomponentdto);
        domain.setAppId(dstapp_id);
		dstcomponentService.create(domain);
        DstComponentDTO dto = dstcomponentMapping.toDto(domain);
		return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstComponent-Create-all')")
    @ApiOperation(value = "根据应用批量建立组件", tags = {"组件" },  notes = "根据应用批量建立组件")
	@RequestMapping(method = RequestMethod.POST, value = "/dstapps/{dstapp_id}/dstcomponents/batch")
    public ResponseEntity<Boolean> createBatchByDstApp(@PathVariable("dstapp_id") String dstapp_id, @RequestBody List<DstComponentDTO> dstcomponentdtos) {
        List<DstComponent> domainlist=dstcomponentMapping.toDomain(dstcomponentdtos);
        for(DstComponent domain:domainlist){
            domain.setAppId(dstapp_id);
        }
        dstcomponentService.createBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @VersionCheck(entity = "dstcomponent" , versionfield = "updatedate")
    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstComponent-Update-all')")
    @ApiOperation(value = "根据应用更新组件", tags = {"组件" },  notes = "根据应用更新组件")
	@RequestMapping(method = RequestMethod.PUT, value = "/dstapps/{dstapp_id}/dstcomponents/{dstcomponent_id}")
    public ResponseEntity<DstComponentDTO> updateByDstApp(@PathVariable("dstapp_id") String dstapp_id, @PathVariable("dstcomponent_id") String dstcomponent_id, @RequestBody DstComponentDTO dstcomponentdto) {
        DstComponent domain = dstcomponentMapping.toDomain(dstcomponentdto);
        domain.setAppId(dstapp_id);
        domain.setId(dstcomponent_id);
		dstcomponentService.update(domain);
        DstComponentDTO dto = dstcomponentMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstComponent-Update-all')")
    @ApiOperation(value = "根据应用批量更新组件", tags = {"组件" },  notes = "根据应用批量更新组件")
	@RequestMapping(method = RequestMethod.PUT, value = "/dstapps/{dstapp_id}/dstcomponents/batch")
    public ResponseEntity<Boolean> updateBatchByDstApp(@PathVariable("dstapp_id") String dstapp_id, @RequestBody List<DstComponentDTO> dstcomponentdtos) {
        List<DstComponent> domainlist=dstcomponentMapping.toDomain(dstcomponentdtos);
        for(DstComponent domain:domainlist){
            domain.setAppId(dstapp_id);
        }
        dstcomponentService.updateBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstComponent-Remove-all')")
    @ApiOperation(value = "根据应用删除组件", tags = {"组件" },  notes = "根据应用删除组件")
	@RequestMapping(method = RequestMethod.DELETE, value = "/dstapps/{dstapp_id}/dstcomponents/{dstcomponent_id}")
    public ResponseEntity<Boolean> removeByDstApp(@PathVariable("dstapp_id") String dstapp_id, @PathVariable("dstcomponent_id") String dstcomponent_id) {
		return ResponseEntity.status(HttpStatus.OK).body(dstcomponentService.remove(dstcomponent_id));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstComponent-Remove-all')")
    @ApiOperation(value = "根据应用批量删除组件", tags = {"组件" },  notes = "根据应用批量删除组件")
	@RequestMapping(method = RequestMethod.DELETE, value = "/dstapps/{dstapp_id}/dstcomponents/batch")
    public ResponseEntity<Boolean> removeBatchByDstApp(@RequestBody List<String> ids) {
        dstcomponentService.removeBatch(ids);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstComponent-Get-all')")
    @ApiOperation(value = "根据应用获取组件", tags = {"组件" },  notes = "根据应用获取组件")
	@RequestMapping(method = RequestMethod.GET, value = "/dstapps/{dstapp_id}/dstcomponents/{dstcomponent_id}")
    public ResponseEntity<DstComponentDTO> getByDstApp(@PathVariable("dstapp_id") String dstapp_id, @PathVariable("dstcomponent_id") String dstcomponent_id) {
        DstComponent domain = dstcomponentService.get(dstcomponent_id);
        DstComponentDTO dto = dstcomponentMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "根据应用获取组件草稿", tags = {"组件" },  notes = "根据应用获取组件草稿")
    @RequestMapping(method = RequestMethod.GET, value = "/dstapps/{dstapp_id}/dstcomponents/getdraft")
    public ResponseEntity<DstComponentDTO> getDraftByDstApp(@PathVariable("dstapp_id") String dstapp_id) {
        DstComponent domain = new DstComponent();
        domain.setAppId(dstapp_id);
        return ResponseEntity.status(HttpStatus.OK).body(dstcomponentMapping.toDto(dstcomponentService.getDraft(domain)));
    }

    @ApiOperation(value = "根据应用检查组件", tags = {"组件" },  notes = "根据应用检查组件")
	@RequestMapping(method = RequestMethod.POST, value = "/dstapps/{dstapp_id}/dstcomponents/checkkey")
    public ResponseEntity<Boolean> checkKeyByDstApp(@PathVariable("dstapp_id") String dstapp_id, @RequestBody DstComponentDTO dstcomponentdto) {
        return  ResponseEntity.status(HttpStatus.OK).body(dstcomponentService.checkKey(dstcomponentMapping.toDomain(dstcomponentdto)));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstComponent-Save-all')")
    @ApiOperation(value = "根据应用保存组件", tags = {"组件" },  notes = "根据应用保存组件")
	@RequestMapping(method = RequestMethod.POST, value = "/dstapps/{dstapp_id}/dstcomponents/save")
    public ResponseEntity<Boolean> saveByDstApp(@PathVariable("dstapp_id") String dstapp_id, @RequestBody DstComponentDTO dstcomponentdto) {
        DstComponent domain = dstcomponentMapping.toDomain(dstcomponentdto);
        domain.setAppId(dstapp_id);
        return ResponseEntity.status(HttpStatus.OK).body(dstcomponentService.save(domain));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstComponent-Save-all')")
    @ApiOperation(value = "根据应用批量保存组件", tags = {"组件" },  notes = "根据应用批量保存组件")
	@RequestMapping(method = RequestMethod.POST, value = "/dstapps/{dstapp_id}/dstcomponents/savebatch")
    public ResponseEntity<Boolean> saveBatchByDstApp(@PathVariable("dstapp_id") String dstapp_id, @RequestBody List<DstComponentDTO> dstcomponentdtos) {
        List<DstComponent> domainlist=dstcomponentMapping.toDomain(dstcomponentdtos);
        for(DstComponent domain:domainlist){
             domain.setAppId(dstapp_id);
        }
        dstcomponentService.saveBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstComponent-Sync-all')")
    @ApiOperation(value = "根据应用组件", tags = {"组件" },  notes = "根据应用组件")
	@RequestMapping(method = RequestMethod.POST, value = "/dstapps/{dstapp_id}/dstcomponents/{dstcomponent_id}/sync")
    public ResponseEntity<DstComponentDTO> syncByDstApp(@PathVariable("dstapp_id") String dstapp_id, @PathVariable("dstcomponent_id") String dstcomponent_id, @RequestBody DstComponentDTO dstcomponentdto) {
        DstComponent domain = dstcomponentMapping.toDomain(dstcomponentdto);
        domain.setAppId(dstapp_id);
        domain = dstcomponentService.sync(domain) ;
        dstcomponentdto = dstcomponentMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dstcomponentdto);
    }
    @ApiOperation(value = "批量处理[根据应用组件]", tags = {"组件" },  notes = "批量处理[根据应用组件]")
	@RequestMapping(method = RequestMethod.POST, value = "/dstapps/{dstapp_id}/dstcomponents/syncbatch")
    public ResponseEntity<Boolean> syncByDstApp(@PathVariable("dstapp_id") String dstapp_id, @RequestBody List<DstComponentDTO> dstcomponentdtos) {
        List<DstComponent> domains = dstcomponentMapping.toDomain(dstcomponentdtos);
        boolean result = dstcomponentService.syncBatch(domains);
        return ResponseEntity.status(HttpStatus.OK).body(result);
    }
    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstComponent-searchDefault-all')")
	@ApiOperation(value = "根据应用获取数据集", tags = {"组件" } ,notes = "根据应用获取数据集")
    @RequestMapping(method= RequestMethod.GET , value="/dstapps/{dstapp_id}/dstcomponents/fetchdefault")
	public ResponseEntity<List<DstComponentDTO>> fetchDstComponentDefaultByDstApp(@PathVariable("dstapp_id") String dstapp_id,DstComponentSearchContext context) {
        context.setN_appid_eq(dstapp_id);
        Page<DstComponent> domains = dstcomponentService.searchDefault(context) ;
        List<DstComponentDTO> list = dstcomponentMapping.toDto(domains.getContent());
	    return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(list);
	}

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstComponent-searchDefault-all')")
	@ApiOperation(value = "根据应用查询数据集", tags = {"组件" } ,notes = "根据应用查询数据集")
    @RequestMapping(method= RequestMethod.POST , value="/dstapps/{dstapp_id}/dstcomponents/searchdefault")
	public ResponseEntity<Page<DstComponentDTO>> searchDstComponentDefaultByDstApp(@PathVariable("dstapp_id") String dstapp_id, @RequestBody DstComponentSearchContext context) {
        context.setN_appid_eq(dstapp_id);
        Page<DstComponent> domains = dstcomponentService.searchDefault(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(dstcomponentMapping.toDto(domains.getContent()), context.getPageable(), domains.getTotalElements()));
	}
    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstComponent-Create-all')")
    @ApiOperation(value = "根据系统应用建立组件", tags = {"组件" },  notes = "根据系统应用建立组件")
	@RequestMapping(method = RequestMethod.POST, value = "/dstsystems/{dstsystem_id}/dstapps/{dstapp_id}/dstcomponents")
    public ResponseEntity<DstComponentDTO> createByDstSystemDstApp(@PathVariable("dstsystem_id") String dstsystem_id, @PathVariable("dstapp_id") String dstapp_id, @RequestBody DstComponentDTO dstcomponentdto) {
        DstComponent domain = dstcomponentMapping.toDomain(dstcomponentdto);
        domain.setAppId(dstapp_id);
		dstcomponentService.create(domain);
        DstComponentDTO dto = dstcomponentMapping.toDto(domain);
		return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstComponent-Create-all')")
    @ApiOperation(value = "根据系统应用批量建立组件", tags = {"组件" },  notes = "根据系统应用批量建立组件")
	@RequestMapping(method = RequestMethod.POST, value = "/dstsystems/{dstsystem_id}/dstapps/{dstapp_id}/dstcomponents/batch")
    public ResponseEntity<Boolean> createBatchByDstSystemDstApp(@PathVariable("dstsystem_id") String dstsystem_id, @PathVariable("dstapp_id") String dstapp_id, @RequestBody List<DstComponentDTO> dstcomponentdtos) {
        List<DstComponent> domainlist=dstcomponentMapping.toDomain(dstcomponentdtos);
        for(DstComponent domain:domainlist){
            domain.setAppId(dstapp_id);
        }
        dstcomponentService.createBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @VersionCheck(entity = "dstcomponent" , versionfield = "updatedate")
    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstComponent-Update-all')")
    @ApiOperation(value = "根据系统应用更新组件", tags = {"组件" },  notes = "根据系统应用更新组件")
	@RequestMapping(method = RequestMethod.PUT, value = "/dstsystems/{dstsystem_id}/dstapps/{dstapp_id}/dstcomponents/{dstcomponent_id}")
    public ResponseEntity<DstComponentDTO> updateByDstSystemDstApp(@PathVariable("dstsystem_id") String dstsystem_id, @PathVariable("dstapp_id") String dstapp_id, @PathVariable("dstcomponent_id") String dstcomponent_id, @RequestBody DstComponentDTO dstcomponentdto) {
        DstComponent domain = dstcomponentMapping.toDomain(dstcomponentdto);
        domain.setAppId(dstapp_id);
        domain.setId(dstcomponent_id);
		dstcomponentService.update(domain);
        DstComponentDTO dto = dstcomponentMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstComponent-Update-all')")
    @ApiOperation(value = "根据系统应用批量更新组件", tags = {"组件" },  notes = "根据系统应用批量更新组件")
	@RequestMapping(method = RequestMethod.PUT, value = "/dstsystems/{dstsystem_id}/dstapps/{dstapp_id}/dstcomponents/batch")
    public ResponseEntity<Boolean> updateBatchByDstSystemDstApp(@PathVariable("dstsystem_id") String dstsystem_id, @PathVariable("dstapp_id") String dstapp_id, @RequestBody List<DstComponentDTO> dstcomponentdtos) {
        List<DstComponent> domainlist=dstcomponentMapping.toDomain(dstcomponentdtos);
        for(DstComponent domain:domainlist){
            domain.setAppId(dstapp_id);
        }
        dstcomponentService.updateBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstComponent-Remove-all')")
    @ApiOperation(value = "根据系统应用删除组件", tags = {"组件" },  notes = "根据系统应用删除组件")
	@RequestMapping(method = RequestMethod.DELETE, value = "/dstsystems/{dstsystem_id}/dstapps/{dstapp_id}/dstcomponents/{dstcomponent_id}")
    public ResponseEntity<Boolean> removeByDstSystemDstApp(@PathVariable("dstsystem_id") String dstsystem_id, @PathVariable("dstapp_id") String dstapp_id, @PathVariable("dstcomponent_id") String dstcomponent_id) {
		return ResponseEntity.status(HttpStatus.OK).body(dstcomponentService.remove(dstcomponent_id));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstComponent-Remove-all')")
    @ApiOperation(value = "根据系统应用批量删除组件", tags = {"组件" },  notes = "根据系统应用批量删除组件")
	@RequestMapping(method = RequestMethod.DELETE, value = "/dstsystems/{dstsystem_id}/dstapps/{dstapp_id}/dstcomponents/batch")
    public ResponseEntity<Boolean> removeBatchByDstSystemDstApp(@RequestBody List<String> ids) {
        dstcomponentService.removeBatch(ids);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstComponent-Get-all')")
    @ApiOperation(value = "根据系统应用获取组件", tags = {"组件" },  notes = "根据系统应用获取组件")
	@RequestMapping(method = RequestMethod.GET, value = "/dstsystems/{dstsystem_id}/dstapps/{dstapp_id}/dstcomponents/{dstcomponent_id}")
    public ResponseEntity<DstComponentDTO> getByDstSystemDstApp(@PathVariable("dstsystem_id") String dstsystem_id, @PathVariable("dstapp_id") String dstapp_id, @PathVariable("dstcomponent_id") String dstcomponent_id) {
        DstComponent domain = dstcomponentService.get(dstcomponent_id);
        DstComponentDTO dto = dstcomponentMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "根据系统应用获取组件草稿", tags = {"组件" },  notes = "根据系统应用获取组件草稿")
    @RequestMapping(method = RequestMethod.GET, value = "/dstsystems/{dstsystem_id}/dstapps/{dstapp_id}/dstcomponents/getdraft")
    public ResponseEntity<DstComponentDTO> getDraftByDstSystemDstApp(@PathVariable("dstsystem_id") String dstsystem_id, @PathVariable("dstapp_id") String dstapp_id) {
        DstComponent domain = new DstComponent();
        domain.setAppId(dstapp_id);
        return ResponseEntity.status(HttpStatus.OK).body(dstcomponentMapping.toDto(dstcomponentService.getDraft(domain)));
    }

    @ApiOperation(value = "根据系统应用检查组件", tags = {"组件" },  notes = "根据系统应用检查组件")
	@RequestMapping(method = RequestMethod.POST, value = "/dstsystems/{dstsystem_id}/dstapps/{dstapp_id}/dstcomponents/checkkey")
    public ResponseEntity<Boolean> checkKeyByDstSystemDstApp(@PathVariable("dstsystem_id") String dstsystem_id, @PathVariable("dstapp_id") String dstapp_id, @RequestBody DstComponentDTO dstcomponentdto) {
        return  ResponseEntity.status(HttpStatus.OK).body(dstcomponentService.checkKey(dstcomponentMapping.toDomain(dstcomponentdto)));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstComponent-Save-all')")
    @ApiOperation(value = "根据系统应用保存组件", tags = {"组件" },  notes = "根据系统应用保存组件")
	@RequestMapping(method = RequestMethod.POST, value = "/dstsystems/{dstsystem_id}/dstapps/{dstapp_id}/dstcomponents/save")
    public ResponseEntity<Boolean> saveByDstSystemDstApp(@PathVariable("dstsystem_id") String dstsystem_id, @PathVariable("dstapp_id") String dstapp_id, @RequestBody DstComponentDTO dstcomponentdto) {
        DstComponent domain = dstcomponentMapping.toDomain(dstcomponentdto);
        domain.setAppId(dstapp_id);
        return ResponseEntity.status(HttpStatus.OK).body(dstcomponentService.save(domain));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstComponent-Save-all')")
    @ApiOperation(value = "根据系统应用批量保存组件", tags = {"组件" },  notes = "根据系统应用批量保存组件")
	@RequestMapping(method = RequestMethod.POST, value = "/dstsystems/{dstsystem_id}/dstapps/{dstapp_id}/dstcomponents/savebatch")
    public ResponseEntity<Boolean> saveBatchByDstSystemDstApp(@PathVariable("dstsystem_id") String dstsystem_id, @PathVariable("dstapp_id") String dstapp_id, @RequestBody List<DstComponentDTO> dstcomponentdtos) {
        List<DstComponent> domainlist=dstcomponentMapping.toDomain(dstcomponentdtos);
        for(DstComponent domain:domainlist){
             domain.setAppId(dstapp_id);
        }
        dstcomponentService.saveBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstComponent-Sync-all')")
    @ApiOperation(value = "根据系统应用组件", tags = {"组件" },  notes = "根据系统应用组件")
	@RequestMapping(method = RequestMethod.POST, value = "/dstsystems/{dstsystem_id}/dstapps/{dstapp_id}/dstcomponents/{dstcomponent_id}/sync")
    public ResponseEntity<DstComponentDTO> syncByDstSystemDstApp(@PathVariable("dstsystem_id") String dstsystem_id, @PathVariable("dstapp_id") String dstapp_id, @PathVariable("dstcomponent_id") String dstcomponent_id, @RequestBody DstComponentDTO dstcomponentdto) {
        DstComponent domain = dstcomponentMapping.toDomain(dstcomponentdto);
        domain.setAppId(dstapp_id);
        domain = dstcomponentService.sync(domain) ;
        dstcomponentdto = dstcomponentMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dstcomponentdto);
    }
    @ApiOperation(value = "批量处理[根据系统应用组件]", tags = {"组件" },  notes = "批量处理[根据系统应用组件]")
	@RequestMapping(method = RequestMethod.POST, value = "/dstsystems/{dstsystem_id}/dstapps/{dstapp_id}/dstcomponents/syncbatch")
    public ResponseEntity<Boolean> syncByDstSystemDstApp(@PathVariable("dstsystem_id") String dstsystem_id, @PathVariable("dstapp_id") String dstapp_id, @RequestBody List<DstComponentDTO> dstcomponentdtos) {
        List<DstComponent> domains = dstcomponentMapping.toDomain(dstcomponentdtos);
        boolean result = dstcomponentService.syncBatch(domains);
        return ResponseEntity.status(HttpStatus.OK).body(result);
    }
    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstComponent-searchDefault-all')")
	@ApiOperation(value = "根据系统应用获取数据集", tags = {"组件" } ,notes = "根据系统应用获取数据集")
    @RequestMapping(method= RequestMethod.GET , value="/dstsystems/{dstsystem_id}/dstapps/{dstapp_id}/dstcomponents/fetchdefault")
	public ResponseEntity<List<DstComponentDTO>> fetchDstComponentDefaultByDstSystemDstApp(@PathVariable("dstsystem_id") String dstsystem_id, @PathVariable("dstapp_id") String dstapp_id,DstComponentSearchContext context) {
        context.setN_appid_eq(dstapp_id);
        Page<DstComponent> domains = dstcomponentService.searchDefault(context) ;
        List<DstComponentDTO> list = dstcomponentMapping.toDto(domains.getContent());
	    return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(list);
	}

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-DstComponent-searchDefault-all')")
	@ApiOperation(value = "根据系统应用查询数据集", tags = {"组件" } ,notes = "根据系统应用查询数据集")
    @RequestMapping(method= RequestMethod.POST , value="/dstsystems/{dstsystem_id}/dstapps/{dstapp_id}/dstcomponents/searchdefault")
	public ResponseEntity<Page<DstComponentDTO>> searchDstComponentDefaultByDstSystemDstApp(@PathVariable("dstsystem_id") String dstsystem_id, @PathVariable("dstapp_id") String dstapp_id, @RequestBody DstComponentSearchContext context) {
        context.setN_appid_eq(dstapp_id);
        Page<DstComponent> domains = dstcomponentService.searchDefault(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(dstcomponentMapping.toDto(domains.getContent()), context.getPageable(), domains.getTotalElements()));
	}
}

