package cn.ibizlab.api.rest;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.math.BigInteger;
import java.util.HashMap;
import lombok.extern.slf4j.Slf4j;
import com.alibaba.fastjson.JSONObject;
import javax.servlet.ServletRequest;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cglib.beans.BeanCopier;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import org.springframework.http.HttpStatus;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.util.StringUtils;
import org.springframework.context.annotation.Lazy;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.security.access.prepost.PostAuthorize;
import org.springframework.validation.annotation.Validated;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import cn.ibizlab.api.dto.*;
import cn.ibizlab.api.mapping.*;
import cn.ibizlab.core.avuedata.domain.BladeVisualMap;
import cn.ibizlab.core.avuedata.service.IBladeVisualMapService;
import cn.ibizlab.core.avuedata.filter.BladeVisualMapSearchContext;
import cn.ibizlab.util.annotation.VersionCheck;

@Slf4j
@Api(tags = {"可视化地图配置表" })
@RestController("api-bladevisualmap")
@RequestMapping("")
public class BladeVisualMapResource {

    @Autowired
    public IBladeVisualMapService bladevisualmapService;

    @Autowired
    @Lazy
    public BladeVisualMapMapping bladevisualmapMapping;

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-BladeVisualMap-Create-all')")
    @ApiOperation(value = "新建可视化地图配置表", tags = {"可视化地图配置表" },  notes = "新建可视化地图配置表")
	@RequestMapping(method = RequestMethod.POST, value = "/bladevisualmaps")
    public ResponseEntity<BladeVisualMapDTO> create(@Validated @RequestBody BladeVisualMapDTO bladevisualmapdto) {
        BladeVisualMap domain = bladevisualmapMapping.toDomain(bladevisualmapdto);
		bladevisualmapService.create(domain);
        BladeVisualMapDTO dto = bladevisualmapMapping.toDto(domain);
		return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-BladeVisualMap-Create-all')")
    @ApiOperation(value = "批量新建可视化地图配置表", tags = {"可视化地图配置表" },  notes = "批量新建可视化地图配置表")
	@RequestMapping(method = RequestMethod.POST, value = "/bladevisualmaps/batch")
    public ResponseEntity<Boolean> createBatch(@RequestBody List<BladeVisualMapDTO> bladevisualmapdtos) {
        bladevisualmapService.createBatch(bladevisualmapMapping.toDomain(bladevisualmapdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-BladeVisualMap-Update-all')")
    @ApiOperation(value = "更新可视化地图配置表", tags = {"可视化地图配置表" },  notes = "更新可视化地图配置表")
	@RequestMapping(method = RequestMethod.PUT, value = "/bladevisualmaps/{bladevisualmap_id}")
    public ResponseEntity<BladeVisualMapDTO> update(@PathVariable("bladevisualmap_id") Long bladevisualmap_id, @RequestBody BladeVisualMapDTO bladevisualmapdto) {
		BladeVisualMap domain  = bladevisualmapMapping.toDomain(bladevisualmapdto);
        domain .setId(bladevisualmap_id);
		bladevisualmapService.update(domain );
		BladeVisualMapDTO dto = bladevisualmapMapping.toDto(domain );
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-BladeVisualMap-Update-all')")
    @ApiOperation(value = "批量更新可视化地图配置表", tags = {"可视化地图配置表" },  notes = "批量更新可视化地图配置表")
	@RequestMapping(method = RequestMethod.PUT, value = "/bladevisualmaps/batch")
    public ResponseEntity<Boolean> updateBatch(@RequestBody List<BladeVisualMapDTO> bladevisualmapdtos) {
        bladevisualmapService.updateBatch(bladevisualmapMapping.toDomain(bladevisualmapdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-BladeVisualMap-Remove-all')")
    @ApiOperation(value = "删除可视化地图配置表", tags = {"可视化地图配置表" },  notes = "删除可视化地图配置表")
	@RequestMapping(method = RequestMethod.DELETE, value = "/bladevisualmaps/{bladevisualmap_id}")
    public ResponseEntity<Boolean> remove(@PathVariable("bladevisualmap_id") Long bladevisualmap_id) {
         return ResponseEntity.status(HttpStatus.OK).body(bladevisualmapService.remove(bladevisualmap_id));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-BladeVisualMap-Remove-all')")
    @ApiOperation(value = "批量删除可视化地图配置表", tags = {"可视化地图配置表" },  notes = "批量删除可视化地图配置表")
	@RequestMapping(method = RequestMethod.DELETE, value = "/bladevisualmaps/batch")
    public ResponseEntity<Boolean> removeBatch(@RequestBody List<Long> ids) {
        bladevisualmapService.removeBatch(ids);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-BladeVisualMap-Get-all')")
    @ApiOperation(value = "获取可视化地图配置表", tags = {"可视化地图配置表" },  notes = "获取可视化地图配置表")
	@RequestMapping(method = RequestMethod.GET, value = "/bladevisualmaps/{bladevisualmap_id}")
    public ResponseEntity<BladeVisualMapDTO> get(@PathVariable("bladevisualmap_id") Long bladevisualmap_id) {
        BladeVisualMap domain = bladevisualmapService.get(bladevisualmap_id);
        BladeVisualMapDTO dto = bladevisualmapMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "获取可视化地图配置表草稿", tags = {"可视化地图配置表" },  notes = "获取可视化地图配置表草稿")
	@RequestMapping(method = RequestMethod.GET, value = "/bladevisualmaps/getdraft")
    public ResponseEntity<BladeVisualMapDTO> getDraft() {
        return ResponseEntity.status(HttpStatus.OK).body(bladevisualmapMapping.toDto(bladevisualmapService.getDraft(new BladeVisualMap())));
    }

    @ApiOperation(value = "检查可视化地图配置表", tags = {"可视化地图配置表" },  notes = "检查可视化地图配置表")
	@RequestMapping(method = RequestMethod.POST, value = "/bladevisualmaps/checkkey")
    public ResponseEntity<Boolean> checkKey(@RequestBody BladeVisualMapDTO bladevisualmapdto) {
        return  ResponseEntity.status(HttpStatus.OK).body(bladevisualmapService.checkKey(bladevisualmapMapping.toDomain(bladevisualmapdto)));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-BladeVisualMap-Save-all')")
    @ApiOperation(value = "保存可视化地图配置表", tags = {"可视化地图配置表" },  notes = "保存可视化地图配置表")
	@RequestMapping(method = RequestMethod.POST, value = "/bladevisualmaps/save")
    public ResponseEntity<Boolean> save(@RequestBody BladeVisualMapDTO bladevisualmapdto) {
        return ResponseEntity.status(HttpStatus.OK).body(bladevisualmapService.save(bladevisualmapMapping.toDomain(bladevisualmapdto)));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-BladeVisualMap-Save-all')")
    @ApiOperation(value = "批量保存可视化地图配置表", tags = {"可视化地图配置表" },  notes = "批量保存可视化地图配置表")
	@RequestMapping(method = RequestMethod.POST, value = "/bladevisualmaps/savebatch")
    public ResponseEntity<Boolean> saveBatch(@RequestBody List<BladeVisualMapDTO> bladevisualmapdtos) {
        bladevisualmapService.saveBatch(bladevisualmapMapping.toDomain(bladevisualmapdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-BladeVisualMap-searchDefault-all')")
	@ApiOperation(value = "获取数据集", tags = {"可视化地图配置表" } ,notes = "获取数据集")
    @RequestMapping(method= RequestMethod.GET , value="/bladevisualmaps/fetchdefault")
	public ResponseEntity<List<BladeVisualMapDTO>> fetchDefault(BladeVisualMapSearchContext context) {
        Page<BladeVisualMap> domains = bladevisualmapService.searchDefault(context) ;
        List<BladeVisualMapDTO> list = bladevisualmapMapping.toDto(domains.getContent());
        return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(list);
	}

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzlite-BladeVisualMap-searchDefault-all')")
	@ApiOperation(value = "查询数据集", tags = {"可视化地图配置表" } ,notes = "查询数据集")
    @RequestMapping(method= RequestMethod.POST , value="/bladevisualmaps/searchdefault")
	public ResponseEntity<Page<BladeVisualMapDTO>> searchDefault(@RequestBody BladeVisualMapSearchContext context) {
        Page<BladeVisualMap> domains = bladevisualmapService.searchDefault(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(bladevisualmapMapping.toDto(domains.getContent()), context.getPageable(), domains.getTotalElements()));
	}


}

