package cn.ibizlab.core.extensions.service;

import cn.ibizlab.core.lite.domain.MetaEntity;
import cn.ibizlab.core.lite.domain.MetaField;
import cn.ibizlab.core.lite.domain.MetaModel;
import cn.ibizlab.core.lite.domain.MetaRelationship;
import cn.ibizlab.core.lite.extensions.model.DataModel;
import cn.ibizlab.core.lite.extensions.model.LayerMapping;
import cn.ibizlab.core.lite.extensions.model.Property;
import cn.ibizlab.core.lite.extensions.model.PropertyMapping;
import cn.ibizlab.core.lite.extensions.service.LiteModelService;
import cn.ibizlab.core.lite.service.IMetaModelService;
import cn.ibizlab.core.lite.service.impl.MetaEntityServiceImpl;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.context.annotation.Primary;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;
import java.util.LinkedHashSet;
import java.util.List;

/**
 * 实体[实体] 自定义服务对象
 */
@Slf4j
@Primary
@Service("MetaEntityExService")
public class MetaEntityExService extends MetaEntityServiceImpl {

    private final int loop = 2;

    @Override
    protected Class currentModelClass() {
        return com.baomidou.mybatisplus.core.toolkit.ReflectionKit.getSuperClassGenericType(this.getClass().getSuperclass(), 1);
    }

    /**
     * [GetDefaultModel:GetDefaultModel] 行为扩展
     *
     * @param et
     * @return
     */
    @Override
    @Transactional
    public MetaEntity getDefaultModel(MetaEntity et) {
        return super.getDefaultModel(et);
    }

    @Autowired
    @Lazy
    private LiteModelService liteModelService;

    @Autowired
    @Lazy
    private IMetaModelService metaModelService;

    @Override
    public boolean update(MetaEntity et) {
        if (!super.update(et))
            return false;
        liteModelService.resetEntityModel(et.getSystemId(), et.getEntityName());
        return true;
    }

    @Override
    public MetaEntity initModels(MetaEntity et) {
        et = get(et.getEntityId());
        DataModel dm = getDataModel(et, loop, null);
        String json = JSON.toJSONString(dm);
        JSONObject modelCfg = JSON.parseObject(json);
        modelCfg.remove("dataModelId");

        MetaModel model = new MetaModel();
        model.setId(et.getEntityId());
        model.setName(et.getEntityName());
        model.setCodeName(et.getCodeName());
        model.setSystemId(et.getSystemId());
        model.setConfig(modelCfg.toJSONString());
        metaModelService.save(model);

        return et;
    }

    /**
     * 获取实体模型结构
     * @param et     实体
     * @param loop   关系嵌套层级（默认只取2层）,如：order --> orderDetail
     * @param parent 父关系
     * @return
     */
    private DataModel getDataModel(MetaEntity et, int loop, MetaRelationship parent) {
        DataModel model = new DataModel();
        model.setDataModelName(et.getEntityName());
        if (loop > 0) {
            loop--;
            model.setObjectProperties(getParentModel(et));  //获取主
            model.setNestedDataModels(getNestedModel(et, loop)); //获取从
        }
        if (!ObjectUtils.isEmpty(parent)) {
            model.setLayerMappings(getMapping(et, parent));//获取关系
        }
        return model;
    }

    /**
     * 获取子关系模型
     * @param et
     * @param loop
     * @return
     */
    private LinkedHashSet<DataModel> getNestedModel(MetaEntity et, int loop) {
        LinkedHashSet<DataModel> models = new LinkedHashSet<>();
        //主关系
        List<MetaRelationship> subList = et.getSubEntitys();
        if (!ObjectUtils.isEmpty(subList)) {
            for (MetaRelationship relation : subList) {
                DataModel model = null;
                MetaEntity subEntity = relation.getEntity();
                if (!ObjectUtils.isEmpty(subEntity)) {
                    model = getDataModel(subEntity, loop, relation);
                    models.add(model);
                }
                models.add(model);
            }
        }
        return models;
    }

    /**
     * 获取父关系模型
     * @param et
     * @return
     */
    public LinkedHashSet<Property> getParentModel(MetaEntity et) {
        LinkedHashSet<Property> properties = new LinkedHashSet<>();
        List<MetaField> fields = et.getFields();
        List<MetaRelationship> parentList = et.getParentEntitys();
        if (!ObjectUtils.isEmpty(parentList)) {
            Property master = new Property();
            master.setPropertyName(et.getEntityName());
            master.setSystem(et.getSystemId());
            master.setPropertyEntity(et.getEntityName());
            properties.add(master);
            for (MetaRelationship relation : parentList) {
                MetaField fkField = getForeignkey(fields, relation);
                if (fkField == null) {
                    log.error("未能找到实体关系{}对应的外键属性", relation.getName());
                    continue;
                }
                // 存储对象参数集合
                Property property = new Property();
                property.setPropertyName(relation.getRefEntityName() == null ? "" : relation.getRefEntityName());
                property.setSystem(et.getSystemId() == null ? "" : et.getSystemId());
                property.setPropertyEntity(relation.getRefEntityName() == null ? "" : relation.getRefEntityName());

                // 对应存储相同实体id关系
                PropertyMapping mapping = new PropertyMapping();
                mapping.setSelfPropertyColumn(fkField.getRefFieldName() == null ? "" : fkField.getRefFieldName());
                mapping.setJoinPropertyName(relation.getEntityName() == null ? "" : relation.getEntityName());
                mapping.setJoinPropertyColumn(fkField.getFieldName() == null ? "" : fkField.getFieldName());

                property.setPropertyMappings(new LinkedHashSet<PropertyMapping>() {{
                    add(mapping);
                }});
                properties.add(property);
            }
        }
        return properties;
    }

    /**
     * 获取父关系映射
     * @param et
     * @param relation
     * @return
     */
    private LinkedHashSet<LayerMapping> getMapping(MetaEntity et, MetaRelationship relation) {
        LinkedHashSet<LayerMapping> mappings = new LinkedHashSet<>();
        List<MetaField> fields = et.getFields();
        if (!ObjectUtils.isEmpty(fields)) {
            MetaField fkField = getForeignkey(fields, relation);
            if (fkField == null) {
                log.error("未能找到实体关系{}对应的外键属性", relation.getName());
                return mappings;
            }
            LayerMapping mapping = new LayerMapping();
            mapping.setSelfPropertyColumn(fkField.getFieldName());
            mapping.setParentPropertyColumn(fkField.getRefFieldName());
            mappings.add(mapping);
        }
        return mappings;
    }

    /**
     * 获取外键属性
     *
     * @param fields
     * @param relation
     * @return
     */
    private MetaField getForeignkey(List<MetaField> fields, MetaRelationship relation) {
        for (MetaField field : fields) {
            String relationName = field.getRelationName();
            if (!StringUtils.isEmpty(relationName) && relationName.equalsIgnoreCase(relation.getName()) && "PICKUP".equalsIgnoreCase(field.getFieldType())) {
                return field;
            }
        }
        return null;
    }
}

