package cn.ibizlab.core.lite.extensions.service;

import cn.ibizlab.core.lite.domain.*;
import cn.ibizlab.core.lite.extensions.domain.MetaEntityModel;
import cn.ibizlab.core.lite.extensions.domain.SysModel;
import cn.ibizlab.core.lite.extensions.mapping.MetaEntityMapping;
import cn.ibizlab.core.lite.service.*;
import com.alibaba.fastjson.JSONObject;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;
import org.springframework.util.ObjectUtils;
import java.util.*;

@Service
@Slf4j
public class LiteCoreService {

    @Autowired
    @Lazy
    IMetaEntityService entityService;

    @Autowired
    @Lazy
    IMetaFieldService fieldService;

    @Autowired
    @Lazy
    IMetaRelationshipService relationService;

    @Autowired
    @Lazy
    LiteCoreService proxyService;

    @Autowired
    @Lazy
    MetaEntityMapping entityMapping;

    @Autowired
    @Lazy
    IMetaModuleService moduleService;

    @Autowired
    @Lazy
    IDstDataSourceService dataSourceService;

    @Transactional
    public boolean syncSysModel(SysModel sysModel) {
        proxyService.asyncSysModel(sysModel);
        return true;
    }

    @Transactional
    @Async("asyncExecutor")
    public void asyncSysModel(SysModel sysModel) {
        long start = System.currentTimeMillis();
        log.info("开始同步[{}]系统模型",sysModel.getSystemid());
        syncDE(sysModel);
        syncDEField(sysModel);
        syncDERelation(sysModel);
        syncDataSource(sysModel);
        long end = System.currentTimeMillis();
        log.info("完成[{}]系统同步模型任务，总耗时[{}]秒",sysModel.getSystemid(),(end-start)/1000);
    }

    /**
     * 初始化master datasource
     * @param sysModel
     */
    private void syncDataSource(SysModel sysModel) {
        DstDataSource dataSource= new DstDataSource();
        dataSource.setDsId(String.format("%s-master",sysModel.getSystemid()));
        dataSource.setDsName(String.format("[%s]主数据源",sysModel.getSystemname()));
        dataSourceService.save(dataSource);
    }

    /**
     * 同步实体
     * @param sysModel
     */
    private void syncDE(SysModel sysModel) {
        Map<String,Integer> delDE = new HashMap<>();
        Map<String,Integer> delModule = new HashMap<>();
        entityService.list(new QueryWrapper<MetaEntity>().select("entityid").eq("systemid",sysModel.getSystemid())).forEach(entity -> delDE.put(entity.getEntityId(),1));
        moduleService.list(new QueryWrapper<MetaModule>().select("codename").eq("systemid",sysModel.getSystemid())).forEach(module -> delModule.put(module.getCodeName(),1));
        List<MetaEntityModel> metaEntityModels = sysModel.getEntities();
        Set<MetaEntity> entities=new HashSet<>();
        Map<Object,MetaModule> modulesMap = new HashMap<>();
        if(!ObjectUtils.isEmpty(metaEntityModels)){
            entities.addAll(entityMapping.toDomain(metaEntityModels));
        }
        entities.forEach(entity -> {
            delDE.remove(entity.getEntityId());
            String moduleId=entity.getModuleId();
            String moduleName=entity.getModuleName();
            if(!ObjectUtils.isEmpty(moduleId) && !ObjectUtils.isEmpty(moduleName)){
                delModule.remove(moduleId);
                if(!modulesMap.containsKey(moduleId)){
                    MetaModule module =new MetaModule();
                    module.setCodeName(moduleId);
                    module.setName(moduleName);
                    module.setSystemId(sysModel.getSystemid());
                    modulesMap.put(moduleId,module);
                }
                MetaModule module = modulesMap.get(moduleId);
                entity.setModuleId(module.getId());
                entity.setModuleName(module.getName());
            }
        });
        //移除无效资源
        if(delDE.size()>0)
            entityService.removeBatch(delDE.keySet());
        if(delModule.size()>0)
            moduleService.removeBatch(delModule.keySet());
        //存储或更新资源saveOrUpdate
        if(entities.size()>0)
            entityService.saveBatch(entities);
        if(modulesMap.size()>0)
            moduleService.saveBatch(modulesMap.values());
    }

    /**
     * 同步实体属性
     * @param sysModel
     */
    private void syncDEField(SysModel sysModel) {
        Map<String,Integer> delField = new HashMap<>();
        Map param=new HashMap();
        param.put("id",sysModel.getSystemid());
        List<JSONObject> oldField=fieldService.select("select fieldid from ibzfield t inner join ibzentity t1 on t.entityid =t1.entityid where t1.systemid = #{et.id}",param);
        for(JSONObject field: oldField){
            delField.put(field.getString("FIELDID"),1);
        }
        Set<MetaField> list = sysModel.getField();
        list.forEach(field -> {
            delField.remove(field.getFieldId());
        });
        //移除无效资源
        if(delField.size()>0)
            fieldService.removeBatch(delField.keySet());
        //存储或更新资源saveOrUpdate
        if(list.size()>0)
            fieldService.saveBatch(list);
    }

    /**
     * 同步实体关系
     * @param sysModel
     */
    private void syncDERelation(SysModel sysModel) {
        Map<String,Integer> delRelation = new HashMap<>();
        Map param=new HashMap();
        param.put("id",sysModel.getSystemid());
        List<JSONObject> oldRelation=relationService.select("select distinct(relationid) from ibzrelation t inner join ibzentity t1 on t.entityid=t1.entityid or t.refentityid=t1.entityid where t1.systemid=#{et.id} ",param);
        for(JSONObject relation: oldRelation){
            delRelation.put(relation.getString("RELATIONID"),1);
        }
        Collection<MetaRelationship> list = sysModel.getRelation();
        list.forEach(relation -> {
            delRelation.remove(relation.getId());
        });
        //移除无效资源
        if(delRelation.size()>0)
            relationService.removeBatch(delRelation.keySet());
        //存储或更新资源saveOrUpdate
        if(list.size()>0)
            relationService.saveBatch(list);
    }
}
