package cn.ibizlab.api.rest;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.math.BigInteger;
import java.util.HashMap;
import lombok.extern.slf4j.Slf4j;
import com.alibaba.fastjson.JSONObject;
import javax.servlet.ServletRequest;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cglib.beans.BeanCopier;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import org.springframework.http.HttpStatus;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.util.StringUtils;
import org.springframework.context.annotation.Lazy;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.security.access.prepost.PostAuthorize;
import org.springframework.validation.annotation.Validated;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import cn.ibizlab.api.dto.*;
import cn.ibizlab.api.mapping.*;
import cn.ibizlab.core.lite.domain.DstSystem;
import cn.ibizlab.core.lite.service.IDstSystemService;
import cn.ibizlab.core.lite.filter.DstSystemSearchContext;
import cn.ibizlab.util.annotation.VersionCheck;

@Slf4j
@Api(tags = {"系统" })
@RestController("api-dstsystem")
@RequestMapping("")
public class DstSystemResource {

    @Autowired
    public IDstSystemService dstsystemService;

    @Autowired
    @Lazy
    public DstSystemMapping dstsystemMapping;

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DstSystem-Create-all')")
    @ApiOperation(value = "新建系统", tags = {"系统" },  notes = "新建系统")
	@RequestMapping(method = RequestMethod.POST, value = "/dstsystems")
    public ResponseEntity<DstSystemDTO> create(@Validated @RequestBody DstSystemDTO dstsystemdto) {
        DstSystem domain = dstsystemMapping.toDomain(dstsystemdto);
		dstsystemService.create(domain);
        DstSystemDTO dto = dstsystemMapping.toDto(domain);
		return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DstSystem-Create-all')")
    @ApiOperation(value = "批量新建系统", tags = {"系统" },  notes = "批量新建系统")
	@RequestMapping(method = RequestMethod.POST, value = "/dstsystems/batch")
    public ResponseEntity<Boolean> createBatch(@RequestBody List<DstSystemDTO> dstsystemdtos) {
        dstsystemService.createBatch(dstsystemMapping.toDomain(dstsystemdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DstSystem-Update-all')")
    @ApiOperation(value = "更新系统", tags = {"系统" },  notes = "更新系统")
	@RequestMapping(method = RequestMethod.PUT, value = "/dstsystems/{dstsystem_id}")
    public ResponseEntity<DstSystemDTO> update(@PathVariable("dstsystem_id") String dstsystem_id, @RequestBody DstSystemDTO dstsystemdto) {
		DstSystem domain  = dstsystemMapping.toDomain(dstsystemdto);
        domain .setPssystemid(dstsystem_id);
		dstsystemService.update(domain );
		DstSystemDTO dto = dstsystemMapping.toDto(domain );
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DstSystem-Update-all')")
    @ApiOperation(value = "批量更新系统", tags = {"系统" },  notes = "批量更新系统")
	@RequestMapping(method = RequestMethod.PUT, value = "/dstsystems/batch")
    public ResponseEntity<Boolean> updateBatch(@RequestBody List<DstSystemDTO> dstsystemdtos) {
        dstsystemService.updateBatch(dstsystemMapping.toDomain(dstsystemdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DstSystem-Remove-all')")
    @ApiOperation(value = "删除系统", tags = {"系统" },  notes = "删除系统")
	@RequestMapping(method = RequestMethod.DELETE, value = "/dstsystems/{dstsystem_id}")
    public ResponseEntity<Boolean> remove(@PathVariable("dstsystem_id") String dstsystem_id) {
         return ResponseEntity.status(HttpStatus.OK).body(dstsystemService.remove(dstsystem_id));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DstSystem-Remove-all')")
    @ApiOperation(value = "批量删除系统", tags = {"系统" },  notes = "批量删除系统")
	@RequestMapping(method = RequestMethod.DELETE, value = "/dstsystems/batch")
    public ResponseEntity<Boolean> removeBatch(@RequestBody List<String> ids) {
        dstsystemService.removeBatch(ids);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DstSystem-Get-all')")
    @ApiOperation(value = "获取系统", tags = {"系统" },  notes = "获取系统")
	@RequestMapping(method = RequestMethod.GET, value = "/dstsystems/{dstsystem_id}")
    public ResponseEntity<DstSystemDTO> get(@PathVariable("dstsystem_id") String dstsystem_id) {
        DstSystem domain = dstsystemService.get(dstsystem_id);
        DstSystemDTO dto = dstsystemMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "获取系统草稿", tags = {"系统" },  notes = "获取系统草稿")
	@RequestMapping(method = RequestMethod.GET, value = "/dstsystems/getdraft")
    public ResponseEntity<DstSystemDTO> getDraft() {
        return ResponseEntity.status(HttpStatus.OK).body(dstsystemMapping.toDto(dstsystemService.getDraft(new DstSystem())));
    }

    @ApiOperation(value = "检查系统", tags = {"系统" },  notes = "检查系统")
	@RequestMapping(method = RequestMethod.POST, value = "/dstsystems/checkkey")
    public ResponseEntity<Boolean> checkKey(@RequestBody DstSystemDTO dstsystemdto) {
        return  ResponseEntity.status(HttpStatus.OK).body(dstsystemService.checkKey(dstsystemMapping.toDomain(dstsystemdto)));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DstSystem-Save-all')")
    @ApiOperation(value = "保存系统", tags = {"系统" },  notes = "保存系统")
	@RequestMapping(method = RequestMethod.POST, value = "/dstsystems/save")
    public ResponseEntity<Boolean> save(@RequestBody DstSystemDTO dstsystemdto) {
        return ResponseEntity.status(HttpStatus.OK).body(dstsystemService.save(dstsystemMapping.toDomain(dstsystemdto)));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DstSystem-Save-all')")
    @ApiOperation(value = "批量保存系统", tags = {"系统" },  notes = "批量保存系统")
	@RequestMapping(method = RequestMethod.POST, value = "/dstsystems/savebatch")
    public ResponseEntity<Boolean> saveBatch(@RequestBody List<DstSystemDTO> dstsystemdtos) {
        dstsystemService.saveBatch(dstsystemMapping.toDomain(dstsystemdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DstSystem-SyncSysModel-all')")
    @ApiOperation(value = "同步系统模型", tags = {"系统" },  notes = "同步系统模型")
	@RequestMapping(method = RequestMethod.POST, value = "/dstsystems/{dstsystem_id}/syncsysmodel")
    public ResponseEntity<DstSystemDTO> syncSysModel(@PathVariable("dstsystem_id") String dstsystem_id, @RequestBody DstSystemDTO dstsystemdto) {
        DstSystem domain = dstsystemMapping.toDomain(dstsystemdto);
        domain.setPssystemid(dstsystem_id);
        domain = dstsystemService.syncSysModel(domain);
        dstsystemdto = dstsystemMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dstsystemdto);
    }
    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DstSystem-SyncSysModel-all')")
    @ApiOperation(value = "批量处理[同步系统模型]", tags = {"系统" },  notes = "批量处理[同步系统模型]")
	@RequestMapping(method = RequestMethod.POST, value = "/dstsystems/{dstsystem_id}/syncsysmodelbatch")
    public ResponseEntity<Boolean> syncSysModelBatch(@RequestBody List<DstSystemDTO> dstsystemdtos) {
        return ResponseEntity.status(HttpStatus.OK).body(dstsystemService.syncSysModelBatch(dstsystemMapping.toDomain(dstsystemdtos)));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DstSystem-searchDefault-all')")
	@ApiOperation(value = "获取数据集", tags = {"系统" } ,notes = "获取数据集")
    @RequestMapping(method= RequestMethod.GET , value="/dstsystems/fetchdefault")
	public ResponseEntity<List<DstSystemDTO>> fetchDefault(DstSystemSearchContext context) {
        Page<DstSystem> domains = dstsystemService.searchDefault(context) ;
        List<DstSystemDTO> list = dstsystemMapping.toDto(domains.getContent());
        return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(list);
	}

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DstSystem-searchDefault-all')")
	@ApiOperation(value = "查询数据集", tags = {"系统" } ,notes = "查询数据集")
    @RequestMapping(method= RequestMethod.POST , value="/dstsystems/searchdefault")
	public ResponseEntity<Page<DstSystemDTO>> searchDefault(@RequestBody DstSystemSearchContext context) {
        Page<DstSystem> domains = dstsystemService.searchDefault(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(dstsystemMapping.toDto(domains.getContent()), context.getPageable(), domains.getTotalElements()));
	}


}

