package cn.ibizlab.api.rest;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.math.BigInteger;
import java.util.HashMap;
import lombok.extern.slf4j.Slf4j;
import com.alibaba.fastjson.JSONObject;
import javax.servlet.ServletRequest;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cglib.beans.BeanCopier;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import org.springframework.http.HttpStatus;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.util.StringUtils;
import org.springframework.context.annotation.Lazy;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.security.access.prepost.PostAuthorize;
import org.springframework.validation.annotation.Validated;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import cn.ibizlab.api.dto.*;
import cn.ibizlab.api.mapping.*;
import cn.ibizlab.core.analysis.domain.DAChart;
import cn.ibizlab.core.analysis.service.IDAChartService;
import cn.ibizlab.core.analysis.filter.DAChartSearchContext;
import cn.ibizlab.util.annotation.VersionCheck;

@Slf4j
@Api(tags = {"图表" })
@RestController("api-dachart")
@RequestMapping("")
public class DAChartResource {

    @Autowired
    public IDAChartService dachartService;

    @Autowired
    @Lazy
    public DAChartMapping dachartMapping;

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DAChart-Create-all')")
    @ApiOperation(value = "新建图表", tags = {"图表" },  notes = "新建图表")
	@RequestMapping(method = RequestMethod.POST, value = "/dacharts")
    public ResponseEntity<DAChartDTO> create(@Validated @RequestBody DAChartDTO dachartdto) {
        DAChart domain = dachartMapping.toDomain(dachartdto);
		dachartService.create(domain);
        DAChartDTO dto = dachartMapping.toDto(domain);
		return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DAChart-Create-all')")
    @ApiOperation(value = "批量新建图表", tags = {"图表" },  notes = "批量新建图表")
	@RequestMapping(method = RequestMethod.POST, value = "/dacharts/batch")
    public ResponseEntity<Boolean> createBatch(@RequestBody List<DAChartDTO> dachartdtos) {
        dachartService.createBatch(dachartMapping.toDomain(dachartdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DAChart-Update-all')")
    @ApiOperation(value = "更新图表", tags = {"图表" },  notes = "更新图表")
	@RequestMapping(method = RequestMethod.PUT, value = "/dacharts/{dachart_id}")
    public ResponseEntity<DAChartDTO> update(@PathVariable("dachart_id") String dachart_id, @RequestBody DAChartDTO dachartdto) {
		DAChart domain  = dachartMapping.toDomain(dachartdto);
        domain .setChartId(dachart_id);
		dachartService.update(domain );
		DAChartDTO dto = dachartMapping.toDto(domain );
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DAChart-Update-all')")
    @ApiOperation(value = "批量更新图表", tags = {"图表" },  notes = "批量更新图表")
	@RequestMapping(method = RequestMethod.PUT, value = "/dacharts/batch")
    public ResponseEntity<Boolean> updateBatch(@RequestBody List<DAChartDTO> dachartdtos) {
        dachartService.updateBatch(dachartMapping.toDomain(dachartdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DAChart-Remove-all')")
    @ApiOperation(value = "删除图表", tags = {"图表" },  notes = "删除图表")
	@RequestMapping(method = RequestMethod.DELETE, value = "/dacharts/{dachart_id}")
    public ResponseEntity<Boolean> remove(@PathVariable("dachart_id") String dachart_id) {
         return ResponseEntity.status(HttpStatus.OK).body(dachartService.remove(dachart_id));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DAChart-Remove-all')")
    @ApiOperation(value = "批量删除图表", tags = {"图表" },  notes = "批量删除图表")
	@RequestMapping(method = RequestMethod.DELETE, value = "/dacharts/batch")
    public ResponseEntity<Boolean> removeBatch(@RequestBody List<String> ids) {
        dachartService.removeBatch(ids);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DAChart-Get-all')")
    @ApiOperation(value = "获取图表", tags = {"图表" },  notes = "获取图表")
	@RequestMapping(method = RequestMethod.GET, value = "/dacharts/{dachart_id}")
    public ResponseEntity<DAChartDTO> get(@PathVariable("dachart_id") String dachart_id) {
        DAChart domain = dachartService.get(dachart_id);
        DAChartDTO dto = dachartMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "获取图表草稿", tags = {"图表" },  notes = "获取图表草稿")
	@RequestMapping(method = RequestMethod.GET, value = "/dacharts/getdraft")
    public ResponseEntity<DAChartDTO> getDraft() {
        return ResponseEntity.status(HttpStatus.OK).body(dachartMapping.toDto(dachartService.getDraft(new DAChart())));
    }

    @ApiOperation(value = "检查图表", tags = {"图表" },  notes = "检查图表")
	@RequestMapping(method = RequestMethod.POST, value = "/dacharts/checkkey")
    public ResponseEntity<Boolean> checkKey(@RequestBody DAChartDTO dachartdto) {
        return  ResponseEntity.status(HttpStatus.OK).body(dachartService.checkKey(dachartMapping.toDomain(dachartdto)));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DAChart-Save-all')")
    @ApiOperation(value = "保存图表", tags = {"图表" },  notes = "保存图表")
	@RequestMapping(method = RequestMethod.POST, value = "/dacharts/save")
    public ResponseEntity<Boolean> save(@RequestBody DAChartDTO dachartdto) {
        return ResponseEntity.status(HttpStatus.OK).body(dachartService.save(dachartMapping.toDomain(dachartdto)));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DAChart-Save-all')")
    @ApiOperation(value = "批量保存图表", tags = {"图表" },  notes = "批量保存图表")
	@RequestMapping(method = RequestMethod.POST, value = "/dacharts/savebatch")
    public ResponseEntity<Boolean> saveBatch(@RequestBody List<DAChartDTO> dachartdtos) {
        dachartService.saveBatch(dachartMapping.toDomain(dachartdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DAChart-searchDefault-all')")
	@ApiOperation(value = "获取数据集", tags = {"图表" } ,notes = "获取数据集")
    @RequestMapping(method= RequestMethod.GET , value="/dacharts/fetchdefault")
	public ResponseEntity<List<DAChartDTO>> fetchDefault(DAChartSearchContext context) {
        Page<DAChart> domains = dachartService.searchDefault(context) ;
        List<DAChartDTO> list = dachartMapping.toDto(domains.getContent());
        return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(list);
	}

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DAChart-searchDefault-all')")
	@ApiOperation(value = "查询数据集", tags = {"图表" } ,notes = "查询数据集")
    @RequestMapping(method= RequestMethod.POST , value="/dacharts/searchdefault")
	public ResponseEntity<Page<DAChartDTO>> searchDefault(@RequestBody DAChartSearchContext context) {
        Page<DAChart> domains = dachartService.searchDefault(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(dachartMapping.toDto(domains.getContent()), context.getPageable(), domains.getTotalElements()));
	}


}

