package cn.ibizlab.api.rest;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.math.BigInteger;
import java.util.HashMap;
import lombok.extern.slf4j.Slf4j;
import com.alibaba.fastjson.JSONObject;
import javax.servlet.ServletRequest;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cglib.beans.BeanCopier;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import org.springframework.http.HttpStatus;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.util.StringUtils;
import org.springframework.context.annotation.Lazy;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.security.access.prepost.PostAuthorize;
import org.springframework.validation.annotation.Validated;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import cn.ibizlab.api.dto.*;
import cn.ibizlab.api.mapping.*;
import cn.ibizlab.core.analysis.domain.DABuild;
import cn.ibizlab.core.analysis.service.IDABuildService;
import cn.ibizlab.core.analysis.filter.DABuildSearchContext;
import cn.ibizlab.util.annotation.VersionCheck;

@Slf4j
@Api(tags = {"分析" })
@RestController("api-dabuild")
@RequestMapping("")
public class DABuildResource {

    @Autowired
    public IDABuildService dabuildService;

    @Autowired
    @Lazy
    public DABuildMapping dabuildMapping;

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DABuild-Create-all')")
    @ApiOperation(value = "新建分析", tags = {"分析" },  notes = "新建分析")
	@RequestMapping(method = RequestMethod.POST, value = "/dabuilds")
    public ResponseEntity<DABuildDTO> create(@Validated @RequestBody DABuildDTO dabuilddto) {
        DABuild domain = dabuildMapping.toDomain(dabuilddto);
		dabuildService.create(domain);
        DABuildDTO dto = dabuildMapping.toDto(domain);
		return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DABuild-Create-all')")
    @ApiOperation(value = "批量新建分析", tags = {"分析" },  notes = "批量新建分析")
	@RequestMapping(method = RequestMethod.POST, value = "/dabuilds/batch")
    public ResponseEntity<Boolean> createBatch(@RequestBody List<DABuildDTO> dabuilddtos) {
        dabuildService.createBatch(dabuildMapping.toDomain(dabuilddtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DABuild-Update-all')")
    @ApiOperation(value = "更新分析", tags = {"分析" },  notes = "更新分析")
	@RequestMapping(method = RequestMethod.PUT, value = "/dabuilds/{dabuild_id}")
    public ResponseEntity<DABuildDTO> update(@PathVariable("dabuild_id") String dabuild_id, @RequestBody DABuildDTO dabuilddto) {
		DABuild domain  = dabuildMapping.toDomain(dabuilddto);
        domain .setBuildId(dabuild_id);
		dabuildService.update(domain );
		DABuildDTO dto = dabuildMapping.toDto(domain );
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DABuild-Update-all')")
    @ApiOperation(value = "批量更新分析", tags = {"分析" },  notes = "批量更新分析")
	@RequestMapping(method = RequestMethod.PUT, value = "/dabuilds/batch")
    public ResponseEntity<Boolean> updateBatch(@RequestBody List<DABuildDTO> dabuilddtos) {
        dabuildService.updateBatch(dabuildMapping.toDomain(dabuilddtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DABuild-Remove-all')")
    @ApiOperation(value = "删除分析", tags = {"分析" },  notes = "删除分析")
	@RequestMapping(method = RequestMethod.DELETE, value = "/dabuilds/{dabuild_id}")
    public ResponseEntity<Boolean> remove(@PathVariable("dabuild_id") String dabuild_id) {
         return ResponseEntity.status(HttpStatus.OK).body(dabuildService.remove(dabuild_id));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DABuild-Remove-all')")
    @ApiOperation(value = "批量删除分析", tags = {"分析" },  notes = "批量删除分析")
	@RequestMapping(method = RequestMethod.DELETE, value = "/dabuilds/batch")
    public ResponseEntity<Boolean> removeBatch(@RequestBody List<String> ids) {
        dabuildService.removeBatch(ids);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DABuild-Get-all')")
    @ApiOperation(value = "获取分析", tags = {"分析" },  notes = "获取分析")
	@RequestMapping(method = RequestMethod.GET, value = "/dabuilds/{dabuild_id}")
    public ResponseEntity<DABuildDTO> get(@PathVariable("dabuild_id") String dabuild_id) {
        DABuild domain = dabuildService.get(dabuild_id);
        DABuildDTO dto = dabuildMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "获取分析草稿", tags = {"分析" },  notes = "获取分析草稿")
	@RequestMapping(method = RequestMethod.GET, value = "/dabuilds/getdraft")
    public ResponseEntity<DABuildDTO> getDraft() {
        return ResponseEntity.status(HttpStatus.OK).body(dabuildMapping.toDto(dabuildService.getDraft(new DABuild())));
    }

    @ApiOperation(value = "检查分析", tags = {"分析" },  notes = "检查分析")
	@RequestMapping(method = RequestMethod.POST, value = "/dabuilds/checkkey")
    public ResponseEntity<Boolean> checkKey(@RequestBody DABuildDTO dabuilddto) {
        return  ResponseEntity.status(HttpStatus.OK).body(dabuildService.checkKey(dabuildMapping.toDomain(dabuilddto)));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DABuild-Run-all')")
    @ApiOperation(value = "运行", tags = {"分析" },  notes = "运行")
	@RequestMapping(method = RequestMethod.POST, value = "/dabuilds/{dabuild_id}/run")
    public ResponseEntity<DABuildDTO> run(@PathVariable("dabuild_id") String dabuild_id, @RequestBody DABuildDTO dabuilddto) {
        DABuild domain = dabuildMapping.toDomain(dabuilddto);
        domain.setBuildId(dabuild_id);
        domain = dabuildService.run(domain);
        dabuilddto = dabuildMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dabuilddto);
    }
    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DABuild-Run-all')")
    @ApiOperation(value = "批量处理[运行]", tags = {"分析" },  notes = "批量处理[运行]")
	@RequestMapping(method = RequestMethod.POST, value = "/dabuilds/{dabuild_id}/runbatch")
    public ResponseEntity<Boolean> runBatch(@RequestBody List<DABuildDTO> dabuilddtos) {
        return ResponseEntity.status(HttpStatus.OK).body(dabuildService.runBatch(dabuildMapping.toDomain(dabuilddtos)));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DABuild-Save-all')")
    @ApiOperation(value = "保存分析", tags = {"分析" },  notes = "保存分析")
	@RequestMapping(method = RequestMethod.POST, value = "/dabuilds/save")
    public ResponseEntity<Boolean> save(@RequestBody DABuildDTO dabuilddto) {
        return ResponseEntity.status(HttpStatus.OK).body(dabuildService.save(dabuildMapping.toDomain(dabuilddto)));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DABuild-Save-all')")
    @ApiOperation(value = "批量保存分析", tags = {"分析" },  notes = "批量保存分析")
	@RequestMapping(method = RequestMethod.POST, value = "/dabuilds/savebatch")
    public ResponseEntity<Boolean> saveBatch(@RequestBody List<DABuildDTO> dabuilddtos) {
        dabuildService.saveBatch(dabuildMapping.toDomain(dabuilddtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DABuild-searchDefault-all')")
	@ApiOperation(value = "获取数据集", tags = {"分析" } ,notes = "获取数据集")
    @RequestMapping(method= RequestMethod.GET , value="/dabuilds/fetchdefault")
	public ResponseEntity<List<DABuildDTO>> fetchDefault(DABuildSearchContext context) {
        Page<DABuild> domains = dabuildService.searchDefault(context) ;
        List<DABuildDTO> list = dabuildMapping.toDto(domains.getContent());
        return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(list);
	}

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DABuild-searchDefault-all')")
	@ApiOperation(value = "查询数据集", tags = {"分析" } ,notes = "查询数据集")
    @RequestMapping(method= RequestMethod.POST , value="/dabuilds/searchdefault")
	public ResponseEntity<Page<DABuildDTO>> searchDefault(@RequestBody DABuildSearchContext context) {
        Page<DABuild> domains = dabuildService.searchDefault(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(dabuildMapping.toDto(domains.getContent()), context.getPageable(), domains.getTotalElements()));
	}


}

