package cn.ibizlab.api.rest;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.math.BigInteger;
import java.util.HashMap;
import lombok.extern.slf4j.Slf4j;
import com.alibaba.fastjson.JSONObject;
import javax.servlet.ServletRequest;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cglib.beans.BeanCopier;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import org.springframework.http.HttpStatus;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.util.StringUtils;
import org.springframework.context.annotation.Lazy;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.security.access.prepost.PostAuthorize;
import org.springframework.validation.annotation.Validated;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import cn.ibizlab.api.dto.*;
import cn.ibizlab.api.mapping.*;
import cn.ibizlab.core.rule.domain.ExecLog;
import cn.ibizlab.core.rule.service.IExecLogService;
import cn.ibizlab.core.rule.filter.ExecLogSearchContext;
import cn.ibizlab.util.annotation.VersionCheck;

@Slf4j
@Api(tags = {"执行日志" })
@RestController("api-execlog")
@RequestMapping("")
public class ExecLogResource {

    @Autowired
    public IExecLogService execlogService;

    @Autowired
    @Lazy
    public ExecLogMapping execlogMapping;

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-ExecLog-Create-all')")
    @ApiOperation(value = "新建执行日志", tags = {"执行日志" },  notes = "新建执行日志")
	@RequestMapping(method = RequestMethod.POST, value = "/execlogs")
    public ResponseEntity<ExecLogDTO> create(@Validated @RequestBody ExecLogDTO execlogdto) {
        ExecLog domain = execlogMapping.toDomain(execlogdto);
		execlogService.create(domain);
        ExecLogDTO dto = execlogMapping.toDto(domain);
		return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-ExecLog-Create-all')")
    @ApiOperation(value = "批量新建执行日志", tags = {"执行日志" },  notes = "批量新建执行日志")
	@RequestMapping(method = RequestMethod.POST, value = "/execlogs/batch")
    public ResponseEntity<Boolean> createBatch(@RequestBody List<ExecLogDTO> execlogdtos) {
        execlogService.createBatch(execlogMapping.toDomain(execlogdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @VersionCheck(entity = "execlog" , versionfield = "updatedate")
    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-ExecLog-Update-all')")
    @ApiOperation(value = "更新执行日志", tags = {"执行日志" },  notes = "更新执行日志")
	@RequestMapping(method = RequestMethod.PUT, value = "/execlogs/{execlog_id}")
    public ResponseEntity<ExecLogDTO> update(@PathVariable("execlog_id") String execlog_id, @RequestBody ExecLogDTO execlogdto) {
		ExecLog domain  = execlogMapping.toDomain(execlogdto);
        domain .setId(execlog_id);
		execlogService.update(domain );
		ExecLogDTO dto = execlogMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-ExecLog-Update-all')")
    @ApiOperation(value = "批量更新执行日志", tags = {"执行日志" },  notes = "批量更新执行日志")
	@RequestMapping(method = RequestMethod.PUT, value = "/execlogs/batch")
    public ResponseEntity<Boolean> updateBatch(@RequestBody List<ExecLogDTO> execlogdtos) {
        execlogService.updateBatch(execlogMapping.toDomain(execlogdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-ExecLog-Remove-all')")
    @ApiOperation(value = "删除执行日志", tags = {"执行日志" },  notes = "删除执行日志")
	@RequestMapping(method = RequestMethod.DELETE, value = "/execlogs/{execlog_id}")
    public ResponseEntity<Boolean> remove(@PathVariable("execlog_id") String execlog_id) {
         return ResponseEntity.status(HttpStatus.OK).body(execlogService.remove(execlog_id));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-ExecLog-Remove-all')")
    @ApiOperation(value = "批量删除执行日志", tags = {"执行日志" },  notes = "批量删除执行日志")
	@RequestMapping(method = RequestMethod.DELETE, value = "/execlogs/batch")
    public ResponseEntity<Boolean> removeBatch(@RequestBody List<String> ids) {
        execlogService.removeBatch(ids);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-ExecLog-Get-all')")
    @ApiOperation(value = "获取执行日志", tags = {"执行日志" },  notes = "获取执行日志")
	@RequestMapping(method = RequestMethod.GET, value = "/execlogs/{execlog_id}")
    public ResponseEntity<ExecLogDTO> get(@PathVariable("execlog_id") String execlog_id) {
        ExecLog domain = execlogService.get(execlog_id);
        ExecLogDTO dto = execlogMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "获取执行日志草稿", tags = {"执行日志" },  notes = "获取执行日志草稿")
	@RequestMapping(method = RequestMethod.GET, value = "/execlogs/getdraft")
    public ResponseEntity<ExecLogDTO> getDraft(ExecLogDTO dto) {
        ExecLog domain = execlogMapping.toDomain(dto);
        return ResponseEntity.status(HttpStatus.OK).body(execlogMapping.toDto(execlogService.getDraft(domain)));
    }

    @ApiOperation(value = "检查执行日志", tags = {"执行日志" },  notes = "检查执行日志")
	@RequestMapping(method = RequestMethod.POST, value = "/execlogs/checkkey")
    public ResponseEntity<Boolean> checkKey(@RequestBody ExecLogDTO execlogdto) {
        return  ResponseEntity.status(HttpStatus.OK).body(execlogService.checkKey(execlogMapping.toDomain(execlogdto)));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-ExecLog-Save-all')")
    @ApiOperation(value = "保存执行日志", tags = {"执行日志" },  notes = "保存执行日志")
	@RequestMapping(method = RequestMethod.POST, value = "/execlogs/save")
    public ResponseEntity<ExecLogDTO> save(@RequestBody ExecLogDTO execlogdto) {
        ExecLog domain = execlogMapping.toDomain(execlogdto);
        execlogService.save(domain);
        return ResponseEntity.status(HttpStatus.OK).body(execlogMapping.toDto(domain));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-ExecLog-Save-all')")
    @ApiOperation(value = "批量保存执行日志", tags = {"执行日志" },  notes = "批量保存执行日志")
	@RequestMapping(method = RequestMethod.POST, value = "/execlogs/savebatch")
    public ResponseEntity<Boolean> saveBatch(@RequestBody List<ExecLogDTO> execlogdtos) {
        execlogService.saveBatch(execlogMapping.toDomain(execlogdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-ExecLog-searchDefault-all')")
	@ApiOperation(value = "获取数据集", tags = {"执行日志" } ,notes = "获取数据集")
    @RequestMapping(method= RequestMethod.GET , value="/execlogs/fetchdefault")
	public ResponseEntity<List<ExecLogDTO>> fetchDefault(ExecLogSearchContext context) {
        Page<ExecLog> domains = execlogService.searchDefault(context) ;
        List<ExecLogDTO> list = execlogMapping.toDto(domains.getContent());
        return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(list);
	}

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-ExecLog-searchDefault-all')")
	@ApiOperation(value = "查询数据集", tags = {"执行日志" } ,notes = "查询数据集")
    @RequestMapping(method= RequestMethod.POST , value="/execlogs/searchdefault")
	public ResponseEntity<Page<ExecLogDTO>> searchDefault(@RequestBody ExecLogSearchContext context) {
        Page<ExecLog> domains = execlogService.searchDefault(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(execlogMapping.toDto(domains.getContent()), context.getPageable(), domains.getTotalElements()));
	}



}

