package cn.ibizlab.api.rest;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.math.BigInteger;
import java.util.HashMap;
import lombok.extern.slf4j.Slf4j;
import com.alibaba.fastjson.JSONObject;
import javax.servlet.ServletRequest;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cglib.beans.BeanCopier;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import org.springframework.http.HttpStatus;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.util.StringUtils;
import org.springframework.context.annotation.Lazy;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.security.access.prepost.PostAuthorize;
import org.springframework.validation.annotation.Validated;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import cn.ibizlab.api.dto.*;
import cn.ibizlab.api.mapping.*;
import cn.ibizlab.core.lite.domain.TableSync;
import cn.ibizlab.core.lite.service.ITableSyncService;
import cn.ibizlab.core.lite.filter.TableSyncSearchContext;
import cn.ibizlab.util.annotation.VersionCheck;

@Slf4j
@Api(tags = {"ETL表同步" })
@RestController("api-tablesync")
@RequestMapping("")
public class TableSyncResource {

    @Autowired
    public ITableSyncService tablesyncService;

    @Autowired
    @Lazy
    public TableSyncMapping tablesyncMapping;

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-TableSync-Create-all')")
    @ApiOperation(value = "新建ETL表同步", tags = {"ETL表同步" },  notes = "新建ETL表同步")
	@RequestMapping(method = RequestMethod.POST, value = "/tablesyncs")
    public ResponseEntity<TableSyncDTO> create(@Validated @RequestBody TableSyncDTO tablesyncdto) {
        TableSync domain = tablesyncMapping.toDomain(tablesyncdto);
		tablesyncService.create(domain);
        TableSyncDTO dto = tablesyncMapping.toDto(domain);
		return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-TableSync-Create-all')")
    @ApiOperation(value = "批量新建ETL表同步", tags = {"ETL表同步" },  notes = "批量新建ETL表同步")
	@RequestMapping(method = RequestMethod.POST, value = "/tablesyncs/batch")
    public ResponseEntity<Boolean> createBatch(@RequestBody List<TableSyncDTO> tablesyncdtos) {
        tablesyncService.createBatch(tablesyncMapping.toDomain(tablesyncdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @VersionCheck(entity = "tablesync" , versionfield = "updatedate")
    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-TableSync-Update-all')")
    @ApiOperation(value = "更新ETL表同步", tags = {"ETL表同步" },  notes = "更新ETL表同步")
	@RequestMapping(method = RequestMethod.PUT, value = "/tablesyncs/{tablesync_id}")
    public ResponseEntity<TableSyncDTO> update(@PathVariable("tablesync_id") String tablesync_id, @RequestBody TableSyncDTO tablesyncdto) {
		TableSync domain  = tablesyncMapping.toDomain(tablesyncdto);
        domain .setId(tablesync_id);
		tablesyncService.update(domain );
		TableSyncDTO dto = tablesyncMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-TableSync-Update-all')")
    @ApiOperation(value = "批量更新ETL表同步", tags = {"ETL表同步" },  notes = "批量更新ETL表同步")
	@RequestMapping(method = RequestMethod.PUT, value = "/tablesyncs/batch")
    public ResponseEntity<Boolean> updateBatch(@RequestBody List<TableSyncDTO> tablesyncdtos) {
        tablesyncService.updateBatch(tablesyncMapping.toDomain(tablesyncdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-TableSync-Remove-all')")
    @ApiOperation(value = "删除ETL表同步", tags = {"ETL表同步" },  notes = "删除ETL表同步")
	@RequestMapping(method = RequestMethod.DELETE, value = "/tablesyncs/{tablesync_id}")
    public ResponseEntity<Boolean> remove(@PathVariable("tablesync_id") String tablesync_id) {
         return ResponseEntity.status(HttpStatus.OK).body(tablesyncService.remove(tablesync_id));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-TableSync-Remove-all')")
    @ApiOperation(value = "批量删除ETL表同步", tags = {"ETL表同步" },  notes = "批量删除ETL表同步")
	@RequestMapping(method = RequestMethod.DELETE, value = "/tablesyncs/batch")
    public ResponseEntity<Boolean> removeBatch(@RequestBody List<String> ids) {
        tablesyncService.removeBatch(ids);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-TableSync-Get-all')")
    @ApiOperation(value = "获取ETL表同步", tags = {"ETL表同步" },  notes = "获取ETL表同步")
	@RequestMapping(method = RequestMethod.GET, value = "/tablesyncs/{tablesync_id}")
    public ResponseEntity<TableSyncDTO> get(@PathVariable("tablesync_id") String tablesync_id) {
        TableSync domain = tablesyncService.get(tablesync_id);
        TableSyncDTO dto = tablesyncMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "获取ETL表同步草稿", tags = {"ETL表同步" },  notes = "获取ETL表同步草稿")
	@RequestMapping(method = RequestMethod.GET, value = "/tablesyncs/getdraft")
    public ResponseEntity<TableSyncDTO> getDraft(TableSyncDTO dto) {
        TableSync domain = tablesyncMapping.toDomain(dto);
        return ResponseEntity.status(HttpStatus.OK).body(tablesyncMapping.toDto(tablesyncService.getDraft(domain)));
    }

    @ApiOperation(value = "检查ETL表同步", tags = {"ETL表同步" },  notes = "检查ETL表同步")
	@RequestMapping(method = RequestMethod.POST, value = "/tablesyncs/checkkey")
    public ResponseEntity<Boolean> checkKey(@RequestBody TableSyncDTO tablesyncdto) {
        return  ResponseEntity.status(HttpStatus.OK).body(tablesyncService.checkKey(tablesyncMapping.toDomain(tablesyncdto)));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-TableSync-Save-all')")
    @ApiOperation(value = "保存ETL表同步", tags = {"ETL表同步" },  notes = "保存ETL表同步")
	@RequestMapping(method = RequestMethod.POST, value = "/tablesyncs/save")
    public ResponseEntity<TableSyncDTO> save(@RequestBody TableSyncDTO tablesyncdto) {
        TableSync domain = tablesyncMapping.toDomain(tablesyncdto);
        tablesyncService.save(domain);
        return ResponseEntity.status(HttpStatus.OK).body(tablesyncMapping.toDto(domain));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-TableSync-Save-all')")
    @ApiOperation(value = "批量保存ETL表同步", tags = {"ETL表同步" },  notes = "批量保存ETL表同步")
	@RequestMapping(method = RequestMethod.POST, value = "/tablesyncs/savebatch")
    public ResponseEntity<Boolean> saveBatch(@RequestBody List<TableSyncDTO> tablesyncdtos) {
        tablesyncService.saveBatch(tablesyncMapping.toDomain(tablesyncdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-TableSync-Sync-all')")
    @ApiOperation(value = "Sync", tags = {"ETL表同步" },  notes = "Sync")
	@RequestMapping(method = RequestMethod.POST, value = "/tablesyncs/{tablesync_id}/sync")
    public ResponseEntity<TableSyncDTO> sync(@PathVariable("tablesync_id") String tablesync_id, @RequestBody TableSyncDTO tablesyncdto) {
        TableSync domain = tablesyncMapping.toDomain(tablesyncdto);
        domain.setId(tablesync_id);
        domain = tablesyncService.sync(domain);
        tablesyncdto = tablesyncMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(tablesyncdto);
    }
    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-TableSync-Sync-all')")
    @ApiOperation(value = "批量处理[Sync]", tags = {"ETL表同步" },  notes = "批量处理[Sync]")
	@RequestMapping(method = RequestMethod.POST, value = "/tablesyncs/syncbatch")
    public ResponseEntity<Boolean> syncBatch(@RequestBody List<TableSyncDTO> tablesyncdtos) {
        List<TableSync> domains = tablesyncMapping.toDomain(tablesyncdtos);
        boolean result = tablesyncService.syncBatch(domains);
        return ResponseEntity.status(HttpStatus.OK).body(result);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-TableSync-searchDefault-all')")
	@ApiOperation(value = "获取数据集", tags = {"ETL表同步" } ,notes = "获取数据集")
    @RequestMapping(method= RequestMethod.GET , value="/tablesyncs/fetchdefault")
	public ResponseEntity<List<TableSyncDTO>> fetchDefault(TableSyncSearchContext context) {
        Page<TableSync> domains = tablesyncService.searchDefault(context) ;
        List<TableSyncDTO> list = tablesyncMapping.toDto(domains.getContent());
        return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(list);
	}

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-TableSync-searchDefault-all')")
	@ApiOperation(value = "查询数据集", tags = {"ETL表同步" } ,notes = "查询数据集")
    @RequestMapping(method= RequestMethod.POST , value="/tablesyncs/searchdefault")
	public ResponseEntity<Page<TableSyncDTO>> searchDefault(@RequestBody TableSyncSearchContext context) {
        Page<TableSync> domains = tablesyncService.searchDefault(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(tablesyncMapping.toDto(domains.getContent()), context.getPageable(), domains.getTotalElements()));
	}



}

