package cn.ibizlab.api.rest;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.math.BigInteger;
import java.util.HashMap;
import lombok.extern.slf4j.Slf4j;
import com.alibaba.fastjson.JSONObject;
import javax.servlet.ServletRequest;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cglib.beans.BeanCopier;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import org.springframework.http.HttpStatus;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.util.StringUtils;
import org.springframework.context.annotation.Lazy;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.security.access.prepost.PostAuthorize;
import org.springframework.validation.annotation.Validated;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import cn.ibizlab.api.dto.*;
import cn.ibizlab.api.mapping.*;
import cn.ibizlab.core.analysis.domain.DAReport;
import cn.ibizlab.core.analysis.service.IDAReportService;
import cn.ibizlab.core.analysis.filter.DAReportSearchContext;
import cn.ibizlab.util.annotation.VersionCheck;

@Slf4j
@Api(tags = {"报表" })
@RestController("api-dareport")
@RequestMapping("")
public class DAReportResource {

    @Autowired
    public IDAReportService dareportService;

    @Autowired
    @Lazy
    public DAReportMapping dareportMapping;

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DAReport-Create-all')")
    @ApiOperation(value = "新建报表", tags = {"报表" },  notes = "新建报表")
	@RequestMapping(method = RequestMethod.POST, value = "/dareports")
    public ResponseEntity<DAReportDTO> create(@Validated @RequestBody DAReportDTO dareportdto) {
        DAReport domain = dareportMapping.toDomain(dareportdto);
		dareportService.create(domain);
        DAReportDTO dto = dareportMapping.toDto(domain);
		return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DAReport-Create-all')")
    @ApiOperation(value = "批量新建报表", tags = {"报表" },  notes = "批量新建报表")
	@RequestMapping(method = RequestMethod.POST, value = "/dareports/batch")
    public ResponseEntity<Boolean> createBatch(@RequestBody List<DAReportDTO> dareportdtos) {
        dareportService.createBatch(dareportMapping.toDomain(dareportdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DAReport-Update-all')")
    @ApiOperation(value = "更新报表", tags = {"报表" },  notes = "更新报表")
	@RequestMapping(method = RequestMethod.PUT, value = "/dareports/{dareport_id}")
    public ResponseEntity<DAReportDTO> update(@PathVariable("dareport_id") String dareport_id, @RequestBody DAReportDTO dareportdto) {
		DAReport domain  = dareportMapping.toDomain(dareportdto);
        domain .setReportId(dareport_id);
		dareportService.update(domain );
		DAReportDTO dto = dareportMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DAReport-Update-all')")
    @ApiOperation(value = "批量更新报表", tags = {"报表" },  notes = "批量更新报表")
	@RequestMapping(method = RequestMethod.PUT, value = "/dareports/batch")
    public ResponseEntity<Boolean> updateBatch(@RequestBody List<DAReportDTO> dareportdtos) {
        dareportService.updateBatch(dareportMapping.toDomain(dareportdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DAReport-Remove-all')")
    @ApiOperation(value = "删除报表", tags = {"报表" },  notes = "删除报表")
	@RequestMapping(method = RequestMethod.DELETE, value = "/dareports/{dareport_id}")
    public ResponseEntity<Boolean> remove(@PathVariable("dareport_id") String dareport_id) {
         return ResponseEntity.status(HttpStatus.OK).body(dareportService.remove(dareport_id));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DAReport-Remove-all')")
    @ApiOperation(value = "批量删除报表", tags = {"报表" },  notes = "批量删除报表")
	@RequestMapping(method = RequestMethod.DELETE, value = "/dareports/batch")
    public ResponseEntity<Boolean> removeBatch(@RequestBody List<String> ids) {
        dareportService.removeBatch(ids);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DAReport-Get-all')")
    @ApiOperation(value = "获取报表", tags = {"报表" },  notes = "获取报表")
	@RequestMapping(method = RequestMethod.GET, value = "/dareports/{dareport_id}")
    public ResponseEntity<DAReportDTO> get(@PathVariable("dareport_id") String dareport_id) {
        DAReport domain = dareportService.get(dareport_id);
        DAReportDTO dto = dareportMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "获取报表草稿", tags = {"报表" },  notes = "获取报表草稿")
	@RequestMapping(method = RequestMethod.GET, value = "/dareports/getdraft")
    public ResponseEntity<DAReportDTO> getDraft(DAReportDTO dto) {
        DAReport domain = dareportMapping.toDomain(dto);
        return ResponseEntity.status(HttpStatus.OK).body(dareportMapping.toDto(dareportService.getDraft(domain)));
    }

    @ApiOperation(value = "检查报表", tags = {"报表" },  notes = "检查报表")
	@RequestMapping(method = RequestMethod.POST, value = "/dareports/checkkey")
    public ResponseEntity<Boolean> checkKey(@RequestBody DAReportDTO dareportdto) {
        return  ResponseEntity.status(HttpStatus.OK).body(dareportService.checkKey(dareportMapping.toDomain(dareportdto)));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DAReport-Save-all')")
    @ApiOperation(value = "保存报表", tags = {"报表" },  notes = "保存报表")
	@RequestMapping(method = RequestMethod.POST, value = "/dareports/save")
    public ResponseEntity<DAReportDTO> save(@RequestBody DAReportDTO dareportdto) {
        DAReport domain = dareportMapping.toDomain(dareportdto);
        dareportService.save(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dareportMapping.toDto(domain));
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DAReport-Save-all')")
    @ApiOperation(value = "批量保存报表", tags = {"报表" },  notes = "批量保存报表")
	@RequestMapping(method = RequestMethod.POST, value = "/dareports/savebatch")
    public ResponseEntity<Boolean> saveBatch(@RequestBody List<DAReportDTO> dareportdtos) {
        dareportService.saveBatch(dareportMapping.toDomain(dareportdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DAReport-searchDefault-all')")
	@ApiOperation(value = "获取数据集", tags = {"报表" } ,notes = "获取数据集")
    @RequestMapping(method= RequestMethod.GET , value="/dareports/fetchdefault")
	public ResponseEntity<List<DAReportDTO>> fetchDefault(DAReportSearchContext context) {
        Page<DAReport> domains = dareportService.searchDefault(context) ;
        List<DAReportDTO> list = dareportMapping.toDto(domains.getContent());
        return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(list);
	}

    @PreAuthorize("hasAnyAuthority('ROLE_SUPERADMIN','ibzdst-DAReport-searchDefault-all')")
	@ApiOperation(value = "查询数据集", tags = {"报表" } ,notes = "查询数据集")
    @RequestMapping(method= RequestMethod.POST , value="/dareports/searchdefault")
	public ResponseEntity<Page<DAReportDTO>> searchDefault(@RequestBody DAReportSearchContext context) {
        Page<DAReport> domains = dareportService.searchDefault(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(dareportMapping.toDto(domains.getContent()), context.getPageable(), domains.getTotalElements()));
	}



}

