package cn.ibizlab.api.rest;

import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.math.BigInteger;
import java.util.HashMap;
import lombok.extern.slf4j.Slf4j;
import com.alibaba.fastjson.JSONObject;
import javax.servlet.ServletRequest;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cglib.beans.BeanCopier;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;
import org.springframework.http.HttpStatus;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.util.StringUtils;
import org.springframework.context.annotation.Lazy;
import org.springframework.security.access.prepost.PreAuthorize;
import org.springframework.security.access.prepost.PostAuthorize;
import org.springframework.validation.annotation.Validated;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import io.swagger.annotations.ApiResponse;
import io.swagger.annotations.ApiResponses;
import cn.ibizlab.api.dto.*;
import cn.ibizlab.api.mapping.*;
import cn.ibizlab.core.lite.domain.DstApp;
import cn.ibizlab.core.lite.service.IDstAppService;
import cn.ibizlab.core.lite.filter.DstAppSearchContext;
import cn.ibizlab.util.annotation.VersionCheck;

@Slf4j
@Api(tags = {"应用" })
@RestController("api-dstapp")
@RequestMapping("")
public class DstAppResource {

    @Autowired
    public IDstAppService dstappService;

    @Autowired
    @Lazy
    public DstAppMapping dstappMapping;

    @ApiOperation(value = "新建应用", tags = {"应用" },  notes = "新建应用")
	@RequestMapping(method = RequestMethod.POST, value = "/dstapps")
    public ResponseEntity<DstAppDTO> create(@Validated @RequestBody DstAppDTO dstappdto) {
        DstApp domain = dstappMapping.toDomain(dstappdto);
		dstappService.create(domain);
        DstAppDTO dto = dstappMapping.toDto(domain);
		return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "批量新建应用", tags = {"应用" },  notes = "批量新建应用")
	@RequestMapping(method = RequestMethod.POST, value = "/dstapps/batch")
    public ResponseEntity<Boolean> createBatch(@RequestBody List<DstAppDTO> dstappdtos) {
        dstappService.createBatch(dstappMapping.toDomain(dstappdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "获取应用", tags = {"应用" },  notes = "获取应用")
	@RequestMapping(method = RequestMethod.GET, value = "/dstapps/{dstapp_id}")
    public ResponseEntity<DstAppDTO> get(@PathVariable("dstapp_id") String dstapp_id) {
        DstApp domain = dstappService.get(dstapp_id);
        DstAppDTO dto = dstappMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "删除应用", tags = {"应用" },  notes = "删除应用")
	@RequestMapping(method = RequestMethod.DELETE, value = "/dstapps/{dstapp_id}")
    public ResponseEntity<Boolean> remove(@PathVariable("dstapp_id") String dstapp_id) {
         return ResponseEntity.status(HttpStatus.OK).body(dstappService.remove(dstapp_id));
    }

    @ApiOperation(value = "批量删除应用", tags = {"应用" },  notes = "批量删除应用")
	@RequestMapping(method = RequestMethod.DELETE, value = "/dstapps/batch")
    public ResponseEntity<Boolean> removeBatch(@RequestBody List<String> ids) {
        dstappService.removeBatch(ids);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "更新应用", tags = {"应用" },  notes = "更新应用")
	@RequestMapping(method = RequestMethod.PUT, value = "/dstapps/{dstapp_id}")
    public ResponseEntity<DstAppDTO> update(@PathVariable("dstapp_id") String dstapp_id, @RequestBody DstAppDTO dstappdto) {
		DstApp domain  = dstappMapping.toDomain(dstappdto);
        domain .setId(dstapp_id);
		dstappService.update(domain );
		DstAppDTO dto = dstappMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "批量更新应用", tags = {"应用" },  notes = "批量更新应用")
	@RequestMapping(method = RequestMethod.PUT, value = "/dstapps/batch")
    public ResponseEntity<Boolean> updateBatch(@RequestBody List<DstAppDTO> dstappdtos) {
        dstappService.updateBatch(dstappMapping.toDomain(dstappdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "检查应用", tags = {"应用" },  notes = "检查应用")
	@RequestMapping(method = RequestMethod.POST, value = "/dstapps/checkkey")
    public ResponseEntity<Boolean> checkKey(@RequestBody DstAppDTO dstappdto) {
        return  ResponseEntity.status(HttpStatus.OK).body(dstappService.checkKey(dstappMapping.toDomain(dstappdto)));
    }

    @ApiOperation(value = "获取应用草稿", tags = {"应用" },  notes = "获取应用草稿")
	@RequestMapping(method = RequestMethod.GET, value = "/dstapps/getdraft")
    public ResponseEntity<DstAppDTO> getDraft(DstAppDTO dto) {
        DstApp domain = dstappMapping.toDomain(dto);
        return ResponseEntity.status(HttpStatus.OK).body(dstappMapping.toDto(dstappService.getDraft(domain)));
    }

    @ApiOperation(value = "保存应用", tags = {"应用" },  notes = "保存应用")
	@RequestMapping(method = RequestMethod.POST, value = "/dstapps/save")
    public ResponseEntity<DstAppDTO> save(@RequestBody DstAppDTO dstappdto) {
        DstApp domain = dstappMapping.toDomain(dstappdto);
        dstappService.save(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dstappMapping.toDto(domain));
    }

    @ApiOperation(value = "批量保存应用", tags = {"应用" },  notes = "批量保存应用")
	@RequestMapping(method = RequestMethod.POST, value = "/dstapps/savebatch")
    public ResponseEntity<Boolean> saveBatch(@RequestBody List<DstAppDTO> dstappdtos) {
        dstappService.saveBatch(dstappMapping.toDomain(dstappdtos));
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

	@ApiOperation(value = "获取数据集", tags = {"应用" } ,notes = "获取数据集")
    @RequestMapping(method= RequestMethod.GET , value="/dstapps/fetchdefault")
	public ResponseEntity<List<DstAppDTO>> fetchDefault(DstAppSearchContext context) {
        Page<DstApp> domains = dstappService.searchDefault(context) ;
        List<DstAppDTO> list = dstappMapping.toDto(domains.getContent());
        return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(list);
	}

	@ApiOperation(value = "查询数据集", tags = {"应用" } ,notes = "查询数据集")
    @RequestMapping(method= RequestMethod.POST , value="/dstapps/searchdefault")
	public ResponseEntity<Page<DstAppDTO>> searchDefault(@RequestBody DstAppSearchContext context) {
        Page<DstApp> domains = dstappService.searchDefault(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(dstappMapping.toDto(domains.getContent()), context.getPageable(), domains.getTotalElements()));
	}



    @ApiOperation(value = "根据系统建立应用", tags = {"应用" },  notes = "根据系统建立应用")
	@RequestMapping(method = RequestMethod.POST, value = "/dstsystems/{dstsystem_id}/dstapps")
    public ResponseEntity<DstAppDTO> createByDstSystem(@PathVariable("dstsystem_id") String dstsystem_id, @RequestBody DstAppDTO dstappdto) {
        DstApp domain = dstappMapping.toDomain(dstappdto);
        domain.setSystemid(dstsystem_id);
		dstappService.create(domain);
        DstAppDTO dto = dstappMapping.toDto(domain);
		return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "根据系统批量建立应用", tags = {"应用" },  notes = "根据系统批量建立应用")
	@RequestMapping(method = RequestMethod.POST, value = "/dstsystems/{dstsystem_id}/dstapps/batch")
    public ResponseEntity<Boolean> createBatchByDstSystem(@PathVariable("dstsystem_id") String dstsystem_id, @RequestBody List<DstAppDTO> dstappdtos) {
        List<DstApp> domainlist=dstappMapping.toDomain(dstappdtos);
        for(DstApp domain:domainlist){
            domain.setSystemid(dstsystem_id);
        }
        dstappService.createBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "根据系统获取应用", tags = {"应用" },  notes = "根据系统获取应用")
	@RequestMapping(method = RequestMethod.GET, value = "/dstsystems/{dstsystem_id}/dstapps/{dstapp_id}")
    public ResponseEntity<DstAppDTO> getByDstSystem(@PathVariable("dstsystem_id") String dstsystem_id, @PathVariable("dstapp_id") String dstapp_id) {
        DstApp domain = dstappService.get(dstapp_id);
        DstAppDTO dto = dstappMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "根据系统删除应用", tags = {"应用" },  notes = "根据系统删除应用")
	@RequestMapping(method = RequestMethod.DELETE, value = "/dstsystems/{dstsystem_id}/dstapps/{dstapp_id}")
    public ResponseEntity<Boolean> removeByDstSystem(@PathVariable("dstsystem_id") String dstsystem_id, @PathVariable("dstapp_id") String dstapp_id) {
		return ResponseEntity.status(HttpStatus.OK).body(dstappService.remove(dstapp_id));
    }

    @ApiOperation(value = "根据系统批量删除应用", tags = {"应用" },  notes = "根据系统批量删除应用")
	@RequestMapping(method = RequestMethod.DELETE, value = "/dstsystems/{dstsystem_id}/dstapps/batch")
    public ResponseEntity<Boolean> removeBatchByDstSystem(@RequestBody List<String> ids) {
        dstappService.removeBatch(ids);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "根据系统更新应用", tags = {"应用" },  notes = "根据系统更新应用")
	@RequestMapping(method = RequestMethod.PUT, value = "/dstsystems/{dstsystem_id}/dstapps/{dstapp_id}")
    public ResponseEntity<DstAppDTO> updateByDstSystem(@PathVariable("dstsystem_id") String dstsystem_id, @PathVariable("dstapp_id") String dstapp_id, @RequestBody DstAppDTO dstappdto) {
        DstApp domain = dstappMapping.toDomain(dstappdto);
        domain.setSystemid(dstsystem_id);
        domain.setId(dstapp_id);
		dstappService.update(domain);
        DstAppDTO dto = dstappMapping.toDto(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dto);
    }

    @ApiOperation(value = "根据系统批量更新应用", tags = {"应用" },  notes = "根据系统批量更新应用")
	@RequestMapping(method = RequestMethod.PUT, value = "/dstsystems/{dstsystem_id}/dstapps/batch")
    public ResponseEntity<Boolean> updateBatchByDstSystem(@PathVariable("dstsystem_id") String dstsystem_id, @RequestBody List<DstAppDTO> dstappdtos) {
        List<DstApp> domainlist=dstappMapping.toDomain(dstappdtos);
        for(DstApp domain:domainlist){
            domain.setSystemid(dstsystem_id);
        }
        dstappService.updateBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

    @ApiOperation(value = "根据系统检查应用", tags = {"应用" },  notes = "根据系统检查应用")
	@RequestMapping(method = RequestMethod.POST, value = "/dstsystems/{dstsystem_id}/dstapps/checkkey")
    public ResponseEntity<Boolean> checkKeyByDstSystem(@PathVariable("dstsystem_id") String dstsystem_id, @RequestBody DstAppDTO dstappdto) {
        return  ResponseEntity.status(HttpStatus.OK).body(dstappService.checkKey(dstappMapping.toDomain(dstappdto)));
    }

    @ApiOperation(value = "根据系统获取应用草稿", tags = {"应用" },  notes = "根据系统获取应用草稿")
    @RequestMapping(method = RequestMethod.GET, value = "/dstsystems/{dstsystem_id}/dstapps/getdraft")
    public ResponseEntity<DstAppDTO> getDraftByDstSystem(@PathVariable("dstsystem_id") String dstsystem_id, DstAppDTO dto) {
        DstApp domain = dstappMapping.toDomain(dto);
        domain.setSystemid(dstsystem_id);
        return ResponseEntity.status(HttpStatus.OK).body(dstappMapping.toDto(dstappService.getDraft(domain)));
    }

    @ApiOperation(value = "根据系统保存应用", tags = {"应用" },  notes = "根据系统保存应用")
	@RequestMapping(method = RequestMethod.POST, value = "/dstsystems/{dstsystem_id}/dstapps/save")
    public ResponseEntity<DstAppDTO> saveByDstSystem(@PathVariable("dstsystem_id") String dstsystem_id, @RequestBody DstAppDTO dstappdto) {
        DstApp domain = dstappMapping.toDomain(dstappdto);
        domain.setSystemid(dstsystem_id);
        dstappService.save(domain);
        return ResponseEntity.status(HttpStatus.OK).body(dstappMapping.toDto(domain));
    }

    @ApiOperation(value = "根据系统批量保存应用", tags = {"应用" },  notes = "根据系统批量保存应用")
	@RequestMapping(method = RequestMethod.POST, value = "/dstsystems/{dstsystem_id}/dstapps/savebatch")
    public ResponseEntity<Boolean> saveBatchByDstSystem(@PathVariable("dstsystem_id") String dstsystem_id, @RequestBody List<DstAppDTO> dstappdtos) {
        List<DstApp> domainlist=dstappMapping.toDomain(dstappdtos);
        for(DstApp domain:domainlist){
             domain.setSystemid(dstsystem_id);
        }
        dstappService.saveBatch(domainlist);
        return  ResponseEntity.status(HttpStatus.OK).body(true);
    }

	@ApiOperation(value = "根据系统获取数据集", tags = {"应用" } ,notes = "根据系统获取数据集")
    @RequestMapping(method= RequestMethod.GET , value="/dstsystems/{dstsystem_id}/dstapps/fetchdefault")
	public ResponseEntity<List<DstAppDTO>> fetchDstAppDefaultByDstSystem(@PathVariable("dstsystem_id") String dstsystem_id,DstAppSearchContext context) {
        context.setN_pssystemid_eq(dstsystem_id);
        Page<DstApp> domains = dstappService.searchDefault(context) ;
        List<DstAppDTO> list = dstappMapping.toDto(domains.getContent());
	    return ResponseEntity.status(HttpStatus.OK)
                .header("x-page", String.valueOf(context.getPageable().getPageNumber()))
                .header("x-per-page", String.valueOf(context.getPageable().getPageSize()))
                .header("x-total", String.valueOf(domains.getTotalElements()))
                .body(list);
	}

	@ApiOperation(value = "根据系统查询数据集", tags = {"应用" } ,notes = "根据系统查询数据集")
    @RequestMapping(method= RequestMethod.POST , value="/dstsystems/{dstsystem_id}/dstapps/searchdefault")
	public ResponseEntity<Page<DstAppDTO>> searchDstAppDefaultByDstSystem(@PathVariable("dstsystem_id") String dstsystem_id, @RequestBody DstAppSearchContext context) {
        context.setN_pssystemid_eq(dstsystem_id);
        Page<DstApp> domains = dstappService.searchDefault(context) ;
	    return ResponseEntity.status(HttpStatus.OK)
                .body(new PageImpl(dstappMapping.toDto(domains.getContent()), context.getPageable(), domains.getTotalElements()));
	}
}

