package cn.ibizlab.core.data.service.impl;

import cn.ibizlab.core.data.dto.BaseData;
import cn.ibizlab.core.data.dto.FilterData;
import cn.ibizlab.core.data.model.DSLink;
import cn.ibizlab.core.data.model.POSchema;
import cn.ibizlab.core.data.repository.MongoDataRepository;
import cn.ibizlab.core.data.service.IDSSettingService;
import cn.ibizlab.core.data.service.IPersistentService;
import cn.ibizlab.core.data.service.ModelService;
import cn.ibizlab.util.errors.BadRequestAlertException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.util.ObjectUtils;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

@Service
@Slf4j
public class MongoPersistentServiceImpl implements IPersistentService {


    @Autowired
    private MongoDataRepository repository;
    @Autowired
    private ModelService modelService;
    
    @Override
    public BaseData create(DSLink link, POSchema poSchema, BaseData et, boolean bGet) {
        repository.insertData(link.getName(),poSchema,et);
        if(bGet) et.setAll(getByMap(link,poSchema,et));
        return et;
    }

    @Override
    public List<BaseData> createBatch(DSLink link, POSchema poSchema, List<BaseData> list, boolean bGet) {
        repository.insertBathData(link.getName(),poSchema,list);
        if(bGet) return getByMapBatch(link,poSchema,list);
        return list;
    }

    @Override
    public BaseData update(DSLink link, POSchema poSchema, BaseData et, boolean bGet) {
        repository.updateData(link.getName(),poSchema,et);
        if(bGet) et.setAll(getByMap(link,poSchema,et));
        return et;
    }

    @Override
    public List<BaseData> updateBatch(DSLink link, POSchema poSchema, List<BaseData> list, boolean bGet) {
        repository.updateBathData(link.getName(),poSchema,list);
        if(bGet) return getByMapBatch(link,poSchema,list);
        return list;
    }

    @Override
    public boolean remove(DSLink link, POSchema poSchema, Serializable key) {
        BaseData et=poSchema.newData(key);
        Assert.notNull(et,"未找到主键");
        repository.removeData(link.getName(),poSchema,et);
        return true;
    }

    @Override
    public boolean removeBatch(DSLink link, POSchema poSchema, List<Serializable> idList) {
        List<BaseData> batch=new ArrayList<>();
        idList.forEach(key->{
            BaseData et = poSchema.newData(key);
            Assert.notNull(et,"未找到主键");
            batch.add(et);
        });
        if(batch.size()>0)
        {
            repository.removeBathData(link.getName(),poSchema,batch);
            batch.clear();
        }
        return true;
    }

    @Override
    public boolean removeByMap(DSLink link, POSchema poSchema, BaseData et) {
        repository.removeData(link.getName(),poSchema,et);
        return true;
    }

    @Override
    public boolean removeByMapBatch(DSLink link, POSchema poSchema, List<BaseData> list) {
        repository.removeBathData(link.getName(),poSchema,list);
        return true;
    }

    @Override
    public BaseData get(DSLink link, POSchema poSchema, Serializable key) {
        BaseData et=poSchema.newData(key);
        Assert.notNull(et,"未找到主键");
        List<BaseData> list = repository.getData(link.getName(),poSchema,et);
        if(ObjectUtils.isEmpty(list)||list.size()>1)
            throw new IllegalArgumentException("未找到数据"+poSchema.getName()+":"+key);
        return list.get(0);
    }

    @Override
    public List<BaseData> getBatch(DSLink link, POSchema poSchema, List<Serializable> idList) {
        List<BaseData> batch=new ArrayList<>();
        idList.forEach(key->{
            BaseData et = poSchema.newData(key);
            Assert.notNull(et,"未找到主键");
            batch.add(et);
        });

        return repository.getBatchData(link.getName(),poSchema,batch);
    }

    @Override
    public BaseData getByMap(DSLink link, POSchema poSchema, BaseData et) {
        List<BaseData> list = repository.getData(link.getName(),poSchema,et);
        if(ObjectUtils.isEmpty(list)||list.size()>1)
            throw new IllegalArgumentException("未找到数据"+poSchema.getName()+":"+et.getKey());
        return list.get(0);
    }

    @Override
    public List<BaseData> getByMapBatch(DSLink link, POSchema poSchema, List<BaseData> list) {
        return repository.getBatchData(link.getName(),poSchema,list);
    }

    @Override
    public List<BaseData> getBatchKey(DSLink link, POSchema poSchema, List<BaseData> list) {
        return repository.getBatchKey(link.getName(),poSchema,list);
    }

    @Override
    public Long count(DSLink link, POSchema poSchema, BaseData et) {
        return repository.countData(link.getName(),poSchema,et);
    }

    @Override
    public boolean checkKey(DSLink link, POSchema poSchema, BaseData et) {
        Long cnt=repository.countData(link.getName(),poSchema,et);
        return cnt==1;
    }

    @Override
    public BaseData save(DSLink link, POSchema poSchema, BaseData et, boolean bGet) {
        if(checkKey(link,poSchema,et))
            return update(link,poSchema,et,bGet);
        else
            return create(link,poSchema,et,bGet);
    }

    @Override
    public List<BaseData> saveBatch(DSLink link, POSchema poSchema, List<BaseData> list, boolean bGet) {
        List<BaseData> rt=this.getBatchKey(link,poSchema,list);

        Map<Serializable,Integer> keys=new LinkedHashMap<>();
        rt.forEach(et->{
            Serializable key=poSchema.getKeyValue(et,false);
            if(!ObjectUtils.isEmpty(key))
                keys.put(key,1);
        });

        List<BaseData> create=new ArrayList<>();
        List<BaseData> update=new ArrayList<>();

        list.forEach(et-> {
            Serializable key = poSchema.getKeyValue(et, false);
            if(keys.containsKey(key))
                update.add(et);
            else
                create.add(et);
        });
        List rtList=new ArrayList<>();
        if(update.size()>0)
            rtList.addAll(this.updateBatch(link,poSchema,update,bGet));
        if(create.size()>0)
            rtList.addAll(this.createBatch(link,poSchema,create,bGet));

        return rtList;
    }

    @Override
    public List<BaseData> select(DSLink link, POSchema poSchema, BaseData et) {
        return repository.selectData(link.getName(),poSchema,et);
    }

    @Override
    public Page<BaseData> select(DSLink link, POSchema poSchema, BaseData et, Pageable pageable) {
        Page<BaseData> pages= repository.selectData(link.getName(),poSchema,poSchema.trans2PO(et),pageable);
        return pages;
    }

    @Override
    public List<BaseData> query(DSLink link, POSchema poSchema, String dataQuery, FilterData context) {
        context.setPOSchema(poSchema);
        return repository.queryData(link.getName(),poSchema,"",context);
    }

    @Override
    public Page<BaseData> query(DSLink link, POSchema poSchema, String dataQuery, FilterData context, Pageable pageable) {
        context.setPOSchema(poSchema);
        if(pageable!=null)
            context.setPageable(pageable);
        else
            pageable=context.getPageable();
        Page<BaseData> pages=repository.queryData(link.getName(),poSchema,"",context,pageable);
        return pages;

    }

    @Override
    public List<BaseData> selectRaw(DSLink link, POSchema poSchema, String sql, FilterData context) {
        return repository.queryData(link.getName(),poSchema,sql,context);
    }

    @Override
    public boolean execRaw(DSLink link, POSchema poSchema, String sql, BaseData param) {
        throw new IllegalArgumentException(String.format("没有实现方法execRaw,%1$s,%2$s",poSchema.getName(),sql));
    }

    @Override
    public List<BaseData> fetch(DSLink link, POSchema poSchema, String dataSet, FilterData context) {
        context.setPOSchema(poSchema);
        return repository.queryData(link.getName(),poSchema,"",context);
    }

    @Override
    public Page<BaseData> fetch(DSLink link, POSchema poSchema, String dataSet, FilterData context, Pageable pageable) {
        context.setPOSchema(poSchema);
        if(pageable!=null)
            context.setPageable(pageable);
        else
            pageable=context.getPageable();
        Page<BaseData> pages=repository.queryData(link.getName(),poSchema,"",context,pageable);
        return pages;
    }

    @Autowired
    private IDSSettingService dsSettingService;
    @Override
    public DSLink getDSLink(String datasource) {
        return dsSettingService.getDataSource(datasource);
    }

    @Override
    public POSchema getSchema(DSLink link, String system, String table) {
        return modelService.getPOSchema(system,table,link.getType());
    }
}
