package cn.ibizlab.core.data.service.impl;

import cn.ibizlab.core.data.domain.DOModel;
import cn.ibizlab.core.data.dto.*;
import cn.ibizlab.core.data.mapper.DbDataMapper;
import cn.ibizlab.core.data.model.POSchema;
import cn.ibizlab.core.data.service.IDataService;
import cn.ibizlab.core.data.service.ModelService;
import cn.ibizlab.util.errors.BadRequestAlertException;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.stereotype.Service;
import org.springframework.util.ObjectUtils;

import java.io.Serializable;
import java.util.*;

/**
 * 实体[业务实体] 无存储服务对象接口实现
 */
@Slf4j
@Service
public class DbDataServiceImpl implements IDataService {

    @Value("${ibiz.data.batchsize:500}")
    private int BATCH_SIZE;

    @Autowired
    private DbDataMapper dbDataMapper;

    @Override
    public ResponseData call(DOModel model, String scope, String datasource, String method, RequestData requestData)
    {
        return null;
    }

    @Override
    public boolean create(DOModel model, String scope, String datasource, BaseData et)
    {
        Serializable key=model.getKeyValue(et,true);
        if(ObjectUtils.isEmpty(key))
            throw new BadRequestAlertException("未找到主键",model.getName(),null);
        if(dbDataMapper.insertData(model.getPOSchema(datasource),et)==1)
            et.setAll(get(model,scope,datasource,key));
        return true;
    }

    @Override
    public void createBatch(DOModel model, String scope, String datasource, List<BaseData> list)
    {
        List<BaseData> batch=new ArrayList<>();
        POSchema poSchema=model.getPOSchema(datasource);
        list.forEach(et->{
            Serializable key=model.getKeyValue(et,true);
            if(ObjectUtils.isEmpty(key))
                throw new BadRequestAlertException("未找到主键",model.getName(),null);
            batch.add(et);
            if(batch.size()>=BATCH_SIZE)
            {
                dbDataMapper.insertBathData(poSchema,batch);
                batch.clear();
            }
        });
        if(batch.size()>=0)
        {
            dbDataMapper.insertBathData(poSchema,batch);
            batch.clear();
        }
    }

    @Override
    public boolean update(DOModel model, String scope, String datasource, BaseData et)
    {
        Serializable key=model.getKeyValue(et,false);
        if(ObjectUtils.isEmpty(key))
            throw new BadRequestAlertException("未找到主键",model.getName(),null);
        dbDataMapper.updateData(model.getPOSchema(datasource),et);
        et.setAll(get(model,scope,datasource,(Serializable)key));
        return true;
    }

    @Override
    public void updateBatch(DOModel model, String scope, String datasource, List<BaseData> list)
    {
        List<BaseData> batch=new ArrayList<>();
        POSchema poSchema=model.getPOSchema(datasource);
        list.forEach(et->{
            Serializable key=model.getKeyValue(et,false);
            if(ObjectUtils.isEmpty(key))
                throw new BadRequestAlertException("未找到主键",model.getName(),null);
            batch.add(et);
            if(batch.size()>=BATCH_SIZE)
            {
                dbDataMapper.updateBathData(poSchema,batch);
                batch.clear();
            }
        });
        if(batch.size()>=0)
        {
            dbDataMapper.updateBathData(poSchema,batch);
            batch.clear();
        }
    }

    @Override
    public boolean remove(DOModel model, String scope, String datasource, Serializable key)
    {
        if(ObjectUtils.isEmpty(key))
            throw new BadRequestAlertException("未找到主键",model.getName(),null);
        BaseData et = model.newData(key);
        if(et==null)
            throw new BadRequestAlertException("未找到主键",model.getName(),null);
        dbDataMapper.removeData(model.getPOSchema(datasource),et);
        return true;
    }

    @Override
    public void removeBatch(DOModel model, String scope, String datasource, List<Serializable> idList)
    {
        List<BaseData> batch=new ArrayList<>();
        POSchema poSchema=model.getPOSchema(datasource);
        idList.forEach(key->{
            BaseData et = model.newData(key);
            if(et==null)
                throw new BadRequestAlertException("未找到主键",model.getName(),null);
            batch.add(et);
            if(batch.size()>=BATCH_SIZE)
            {
                dbDataMapper.removeBathData(poSchema,batch);
                batch.clear();
            }
        });
        if(batch.size()>=0)
        {
            dbDataMapper.removeBathData(poSchema,batch);
            batch.clear();
        }
    }

    @Override
    public boolean removeByMap(DOModel model, String scope, String datasource, BaseData et)
    {
        Serializable key=model.getKeyValue(et,false);
        if(ObjectUtils.isEmpty(key))
            throw new BadRequestAlertException("未找到主键",model.getName(),null);
        POSchema poSchema=model.getPOSchema(datasource);
        dbDataMapper.removeData(poSchema,et);
        return true;
    }

    @Override
    public BaseData get(DOModel model, String scope, String datasource, Serializable key)
    {
        if(ObjectUtils.isEmpty(key))
            throw new BadRequestAlertException("未找到主键",model.getName(),null);
        BaseData et = model.newData(key);
        if(et==null)
            throw new BadRequestAlertException("未找到主键",model.getName(),null);
        POSchema poSchema=model.getPOSchema(datasource);
        List<BaseData> list=dbDataMapper.getData(poSchema,et);
        if(ObjectUtils.isEmpty(list)||list.size()>1)
            throw new BadRequestAlertException("未找到数据",model.getName(),key.toString());
        return poSchema.trans(list.get(0));
    }

    @Override
    public List<BaseData> getBatch(DOModel model, String scope, String datasource, List<Serializable> idList)
    {
        List<BaseData> rt=new ArrayList<>();
        List<BaseData> batch=new ArrayList<>();
        POSchema poSchema=model.getPOSchema(datasource);
        idList.forEach(key->{
            BaseData et = model.newData(key);
            if(et==null)
                throw new BadRequestAlertException("未找到主键",model.getName(),null);
            batch.add(et);
            if(batch.size()>=BATCH_SIZE)
            {
                rt.addAll(dbDataMapper.getBatchData(poSchema,batch));
                batch.clear();
            }
        });
        if(batch.size()>=0)
        {
            rt.addAll(dbDataMapper.getBatchData(poSchema,batch));
            batch.clear();
        }
        return rt;
    }

    @Override
    public BaseData getByMap(DOModel model, String scope, String datasource, BaseData et)
    {
        Serializable key=model.getKeyValue(et,false);
        if(ObjectUtils.isEmpty(key))
            throw new BadRequestAlertException("未找到主键",model.getName(),null);
        POSchema poSchema=model.getPOSchema(datasource);
        List<BaseData> list=dbDataMapper.getData(poSchema,et);
        if(ObjectUtils.isEmpty(list)||list.size()>1)
            throw new BadRequestAlertException("未找到数据",model.getName(),key.toString());
        return poSchema.trans(list.get(0));
    }

    @Override
    public BaseData getDraft(DOModel model, String scope, String datasource, BaseData et)
    {
        return null;
    }

    @Override
    public boolean checkKey(DOModel model, String scope, String datasource, BaseData et)
    {
        Serializable key=model.getKeyValue(et,false);
        if(ObjectUtils.isEmpty(key))
            return false;

        int cnt=dbDataMapper.countData(model.getPOSchema(datasource),et);
        return cnt==1;
    }

    @Override
    public boolean save(DOModel model, String scope, String datasource, BaseData et)
    {
        if(checkKey(model,scope,datasource,et))
            return update(model,scope,datasource,et);
        else
            return create(model,scope,datasource,et);
    }

    @Override
    public void saveBatch(DOModel model, String scope, String datasource, List<BaseData> list)
    {

        List<BaseData> rt=new ArrayList<>();
        List<BaseData> batch=new ArrayList<>();
        POSchema poSchema=model.getPOSchema(datasource);
        list.forEach(et->{
            Serializable key=model.getKeyValue(et,true);
            if(ObjectUtils.isEmpty(key))
                throw new BadRequestAlertException("未找到主键",model.getName(),null);
            batch.add(et);
            if(batch.size()>=BATCH_SIZE)
            {
                rt.addAll(dbDataMapper.getBatchKey(poSchema,batch));
                batch.clear();
            }
        });
        if(batch.size()>=0)
        {
            rt.addAll(dbDataMapper.getBatchKey(poSchema,batch));
            batch.clear();
        }
        Map<Serializable,Integer> keys=new LinkedHashMap<>();
        rt.forEach(et->{
            Serializable key=model.getKeyValue(et,false);
            if(!ObjectUtils.isEmpty(key))
                keys.put(key,1);
        });

        List<BaseData> create=new ArrayList<>();
        List<BaseData> update=new ArrayList<>();

        list.forEach(et-> {
            Serializable key = model.getKeyValue(et, false);
            if(keys.containsKey(key))
                update.add(et);
            else
                create.add(et);
        });
        if(update.size()>0)
            this.updateBatch(model,scope,datasource,update);
        if(create.size()>0)
            this.createBatch(model,scope,datasource,create);

    }

    @Override
    public List<BaseData> select(DOModel model, String scope, String datasource, BaseData et)
    {
        POSchema poSchema=model.getPOSchema(datasource);
        return dbDataMapper.selectData(poSchema,et);
    }

    @Override
    public Page<BaseData> select(DOModel model, String scope, String datasource, BaseData et, Pageable pageable)
    {
        POSchema poSchema=model.getPOSchema(datasource);
        com.baomidou.mybatisplus.extension.plugins.pagination.Page<BaseData> pages=dbDataMapper.selectData(poSchema,et,FilterData.getPages(poSchema,pageable));
        return new PageImpl<BaseData>(pages.getRecords(), pageable, pages.getTotal());
    }

    @Override
    public List<BaseData> query(DOModel model, String scope, String datasource, String dataQuery, FilterData context)
    {
        POSchema poSchema=model.getPOSchema(datasource);
        context.setPOSchema(poSchema);
        return dbDataMapper.queryData(poSchema,context.getSql("-dq-"+dataQuery+"-"),context,context.getSearchCond());
    }

    @Override
    public Page<BaseData> query(DOModel model, String scope, String datasource, String dataQuery, FilterData context, Pageable pageable)
    {
        POSchema poSchema=model.getPOSchema(datasource);
        context.setPOSchema(poSchema);
        if(pageable!=null)
            context.setPageable(pageable);
        else
            pageable=context.getPageable();
        com.baomidou.mybatisplus.extension.plugins.pagination.Page<BaseData> pages=
                dbDataMapper.queryData(poSchema,context.getSql("-dq-"+dataQuery+"-"),context,context.getSearchCond(),context.getPages());
        return new PageImpl<BaseData>(pages.getRecords(), pageable, pages.getTotal());
    }

    @Override
    public List<BaseData> selectRaw(DOModel model, String scope, String datasource, String sql, FilterData context)
    {
        POSchema poSchema=model.getPOSchema(datasource);
        return dbDataMapper.queryData(poSchema,sql,context,context.getSearchCond());
    }

    @Override
    public boolean execRaw(DOModel model, String scope, String datasource, String sql, BaseData param)
    {
        POSchema poSchema=model.getPOSchema(datasource);
        return dbDataMapper.executeRaw(poSchema,sql,param);
    }

    @Override
    public List<BaseData> fetch(DOModel model, String scope, String datasource, String dataSet, FilterData context)
    {
        POSchema poSchema=model.getPOSchema(datasource);
        context.setPOSchema(poSchema);
        return dbDataMapper.queryData(poSchema,context.getSql("-ds-"+dataSet+"-"),context,context.getSearchCond());
    }

    @Override
    public Page<BaseData> fetch(DOModel model, String scope, String datasource, String dataSet, FilterData context, Pageable pageable)
    {
        POSchema poSchema=model.getPOSchema(datasource);
        context.setPOSchema(poSchema);
        if(pageable!=null)
            context.setPageable(pageable);
        else
            pageable=context.getPageable();
        com.baomidou.mybatisplus.extension.plugins.pagination.Page<BaseData> pages=
                dbDataMapper.queryData(poSchema,context.getSql("-ds-"+dataSet+"-"),context,context.getSearchCond(),context.getPages());
        return new PageImpl<BaseData>(pages.getRecords(), pageable, pages.getTotal());
    }

    @Override
    public DOModel getDOModel(String system, String entity)
    {
        return modelService.getDOModel(system,entity);
    }


    @Autowired
    private ModelService modelService;

}


