package cn.ibizlab.core.data.mongodb;

import cn.ibizlab.core.data.dto.BaseData;
import cn.ibizlab.core.data.dto.FilterData;
import cn.ibizlab.core.data.model.POSchema;
import cn.ibizlab.util.errors.BadRequestAlertException;
import com.alibaba.fastjson.JSON;
import com.baomidou.dynamic.datasource.toolkit.DynamicDataSourceContextHolder;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import lombok.extern.slf4j.Slf4j;
import org.apache.ibatis.annotations.Param;
import org.apache.ibatis.annotations.Select;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.*;
import org.springframework.data.mongodb.core.MongoTemplate;
import org.springframework.data.mongodb.core.query.BasicQuery;
import org.springframework.data.mongodb.core.query.Criteria;
import org.springframework.data.mongodb.core.query.Query;
import org.springframework.data.mongodb.core.query.Update;
import org.springframework.util.ObjectUtils;
import org.springframework.util.StringUtils;

import java.util.*;

@Slf4j
public abstract class MongoDataRepository  {

	@Autowired
	private MongoTemplate mongoTemplate;

	/**
	 * 反射获取泛型类型
	 *
	 * @return
	 */
	protected abstract Class<BaseData> getEntityClass();

	/**
	 * 保存
	 *
	 * @param t
	 * @param collectionName 集合名称
	 */
	public void save(BaseData t, String collectionName) {
		 
		this.mongoTemplate.save(t, collectionName);
	}


	/**
	 * 批量保存
	 * @param objectsToSave
	 * @param collectionName
	 */
	public Collection<BaseData> batchSave(Collection<BaseData> objectsToSave, String collectionName) {
		log.info("-------------->MongoDB batch save start");
		return this.mongoTemplate.insert(objectsToSave,collectionName);
	}


	/**
	 * 根据id从几何中查询对象
	 * @param id
	 * @param collectionName
	 * @return
	 */
	public BaseData queryById(String id,String collectionName) {
		Query query = new Query(Criteria.where("_id").is(id));
		log.info("-------------->MongoDB find start");
		return this.mongoTemplate.findOne(query, this.getEntityClass(),collectionName);
	}


	/**
	 * 根据条件查询集合
	 * @param object
	 * @param collectionName
	 * @return
	 */
	public List<BaseData> queryList(BaseData object,String collectionName) {
		Query query = getQueryByObject(object);
		log.info("-------------->MongoDB find start");
		return mongoTemplate.find(query, this.getEntityClass(),collectionName);
	}



	/**
	 * 根据条件查询只返回一个文档
	 * @param object
	 * @param collectionName
	 * @return
	 */
	public BaseData queryOne(BaseData object,String collectionName) {
		Query query = getQueryByObject(object);
		log.info("-------------->MongoDB find start");
		return mongoTemplate.findOne(query, this.getEntityClass(),collectionName);
	}



	/**
	 * 根据条件分页查询
	 * @param object
	 * @param start
	 * @param size
	 * @param collectionName
	 * @return
	 */
	public List<BaseData> getPage(BaseData object, int start, int size,String collectionName) {
		Query query = getQueryByObject(object);
		if(start >0)
		{
			start --;
		}
		query.skip(start);
		query.limit(size);
		log.info("-------------->MongoDB queryPage start");
		return this.mongoTemplate.find(query, this.getEntityClass(),collectionName);
	}


	/**
	 * 根据条件查询库中符合条件的记录数量
	 * @param object
	 * @param collectionName
	 * @return
	 */
	public Long getCount(BaseData object,String collectionName) {
		Query query = getQueryByObject(object);
		log.info("-------------->MongoDB Count start");
		return this.mongoTemplate.count(query, this.getEntityClass(),collectionName);
	}

	/*MongoDB中更新操作分为三种
	 * 1：updateFirst     修改第一条
	 * 2：updateMulti     修改所有匹配的记录
	 * 3：upsert  修改时如果不存在则进行添加操作
	 * */



	/**
	 * 删除对象
	 *
	 * @param t
	 * @param collectionName
	 * @return
	 */
	public int delete(BaseData t, String collectionName) {
		log.info("-------------->MongoDB delete start");
		return (int) this.mongoTemplate.remove(t, collectionName).getDeletedCount();
	}


	/**
	 * 根据id列表批量删除
	 * @param ids
	 * @param collectionName
	 * @return
	 */
	public int delete(List<String> ids,String collectionName) {
		Criteria criteria = Criteria.where("_id").in(ids);
		Query query = new Query(criteria);
		return (int) this.mongoTemplate.remove(query, this.getEntityClass(),collectionName).getDeletedCount();
	}



	/**
	 * 根据id删除
	 *
	 * @param id
	 * @param collectionName 集合名称
	 */
	public void deleteById(String id, String collectionName) {
		Criteria criteria = Criteria.where("_id").is(id);
		if (null != criteria) {
			Query query = new Query(criteria);
			BaseData obj = this.mongoTemplate.findOne(query, this.getEntityClass(),collectionName);
			log.info("-------------->MongoDB deleteById start");
			if (obj != null) {
				this.delete(obj, collectionName);
			}
		}
	}

	/**
	 * 修改匹配到的第一条记录
	 *
	 * @param srcObj
	 * @param targetObj
	 * @param collectionName 集合名称
	 */
	public void updateFirst(BaseData srcObj, BaseData targetObj, String collectionName) {
		Query query = getQueryByObject(srcObj);
		Update update = getUpdateByObject(targetObj);
		log.info("-------------->MongoDB updateFirst start");
		this.mongoTemplate.updateFirst(query, update, collectionName);
	}


	/**
	 * 修改匹配到的所有记录
	 *
	 * @param srcObj
	 * @param targetObj
	 * @param collectionName 集合名称
	 */
	public void updateMulti(BaseData srcObj, BaseData targetObj, String collectionName) {
		Query query = getQueryByObject(srcObj);
		Update update = getUpdateByObject(targetObj);
		log.info("-------------->MongoDB updateFirst start");
		this.mongoTemplate.updateMulti(query, update, collectionName);
	}


	/**
	 * 修改匹配到的记录，若不存在该记录则进行添加
	 *
	 * @param srcObj
	 * @param targetObj
	 * @param collectionName 集合名字
	 */
	public void updateInsert(BaseData srcObj, BaseData targetObj, String collectionName) {
		Query query = getQueryByObject(srcObj);
		Update update = getUpdateByObject(targetObj);
		log.info("-------------->MongoDB updateInsert start");
		this.mongoTemplate.upsert(query, update, collectionName);
	}

	/**
	 * 将查询条件对象转换为query
	 *
	 * @param object
	 * @return
	 * @author Jason
	 */
	private Query getQueryByObject(BaseData object) {
		Query query = new Query();
		Map<String, Object> dataMap = (Map)object;
		Criteria criteria = new Criteria();
		for (Map.Entry<String, Object> entry : dataMap.entrySet()) {
			criteria.and(entry.getKey()).is(entry.getValue());
		}
		query.addCriteria(criteria);
		return query;
	}

	/**
	 * 将查询条件对象转换为update
	 *
	 * @param object
	 * @return
	 * @author Jason
	 */
	private Update getUpdateByObject(BaseData object) {
		Update update = new Update();
		Map<String, Object> dataMap = (Map)object;
		for (Map.Entry<String,Object> entry : dataMap.entrySet()) {
			update.set(entry.getKey(), entry.getValue());
		}
		return update;
	}


	public Query getKeyQuery(POSchema schema,BaseData data)
	{
		Query query = new Query();
		if(!ObjectUtils.isEmpty(data.getKey()))
			return query.addCriteria(new Criteria().and("_id").is(data.getKey()));
		Criteria criteria = new Criteria();
		if(!ObjectUtils.isEmpty(schema.getKeyMap()))
		{
			for(Map.Entry<String,String> entry:schema.getKeyMap().entrySet())
			{
				Object obj=data.get(entry.getValue());
				if(!ObjectUtils.isEmpty(obj))
					criteria.and(entry.getKey()).is(obj);
				else
					throw new BadRequestAlertException("未找到主键",schema.getName(),entry.getValue());
			}
		}
		query.addCriteria(criteria);
		return query;
	}

	public Query getKeyQuery(POSchema schema,List list)
	{
		Query query = new Query();
		if(ObjectUtils.isEmpty(list))
			return query;
		List<String> ids=new ArrayList<>();
		if(list.get(0) instanceof BaseData)
		{
			list.forEach(item -> {
				BaseData data = (BaseData) item;
				if (ObjectUtils.isEmpty(data.getKey()))
				{
					if (!ObjectUtils.isEmpty(schema.getKeyMap()))
					{
						for (Map.Entry<String, String> entry : schema.getKeyMap().entrySet())
						{
							Object obj = data.get(entry.getValue());
							if (ObjectUtils.isEmpty(obj))
								throw new BadRequestAlertException("未找到主键", schema.getName(), entry.getValue());
						}
					}
				}
				if (!ObjectUtils.isEmpty(data.getKey()))
					ids.add(data.getKey().toString());
			});
		}
		else
			ids.addAll(list);
		return query.addCriteria(new Criteria().and("_id").in(ids));
	}


	public void insertData(String ds,POSchema schema, BaseData data){

		try
		{
			DynamicDataSourceContextHolder.push(ds);
			this.mongoTemplate.save(data, schema.getName());
		}
		finally {
			DynamicDataSourceContextHolder.poll();
		}
	}

	public void insertBathData(String ds,POSchema schema, List<BaseData> list){

		try
		{
			DynamicDataSourceContextHolder.push(ds);
			this.mongoTemplate.insert(list, schema.getName());
		}
		finally {
			DynamicDataSourceContextHolder.poll();
		}
	}

	public int updateData(String ds,POSchema schema, BaseData data){
		try
		{
			DynamicDataSourceContextHolder.push(ds);
			Query query = getKeyQuery(schema,data);
			Update update = getUpdateByObject(data);
			return (int)this.mongoTemplate.updateFirst(query, update, schema.getName()).getModifiedCount();
		}
		finally {
			DynamicDataSourceContextHolder.poll();
		}
	}

	public void updateBathData(String ds,POSchema schema, List<BaseData> list){
		try
		{
			DynamicDataSourceContextHolder.push(ds);
			list.forEach(data -> {
				Query query = getKeyQuery(schema,data);
				Update update = getUpdateByObject(data);
				this.mongoTemplate.updateFirst(query, update, schema.getName());
			});
		}
		finally {
			DynamicDataSourceContextHolder.poll();
		}
	}

	public int removeData(String ds,POSchema schema, BaseData data){
		try
		{
			DynamicDataSourceContextHolder.push(ds);
			return (int)this.mongoTemplate.remove(getKeyQuery(schema,data),schema.getName()).getDeletedCount();
		}
		finally {
			DynamicDataSourceContextHolder.poll();
		}
	}

	public int removeBathData(String ds,POSchema schema, List list){
		try
		{
			DynamicDataSourceContextHolder.push(ds);
			return (int)mongoTemplate.remove(getKeyQuery(schema,list),schema.getName()).getDeletedCount();
		}
		finally {
			DynamicDataSourceContextHolder.poll();
		}
	}

	public int save(String ds,POSchema schema, BaseData data){
		try
		{
			DynamicDataSourceContextHolder.push(ds);
			Query query = getKeyQuery(schema,data);
			Update update = getUpdateByObject(data);
			return (int)this.mongoTemplate.upsert(query, update, schema.getName()).getModifiedCount();
		}
		finally {
			DynamicDataSourceContextHolder.poll();
		}
	}

	public void saveBatch(String ds,POSchema schema, List<BaseData> list){
		try
		{
			DynamicDataSourceContextHolder.push(ds);
			list.forEach(data -> {
				Query query = getKeyQuery(schema,data);
				Update update = getUpdateByObject(data);
				mongoTemplate.upsert(query, update, schema.getName());
			});
		}
		finally {
			DynamicDataSourceContextHolder.poll();
		}
	}

	public int countData(String ds,POSchema schema, BaseData data){
		try
		{
			DynamicDataSourceContextHolder.push(ds);
			return (int)mongoTemplate.count(getQueryByObject(data),schema.getName());
		}
		finally {
			DynamicDataSourceContextHolder.poll();
		}
	}

	public List<BaseData> getData(String ds,POSchema schema, BaseData data){
		try
		{
			DynamicDataSourceContextHolder.push(ds);
			return mongoTemplate.find(getKeyQuery(schema,data),getEntityClass(),schema.getName());
		}
		finally {
			DynamicDataSourceContextHolder.poll();
		}
	}

	public List<BaseData> getBatchData(String ds,POSchema schema, List<BaseData> list){
		try
		{
			DynamicDataSourceContextHolder.push(ds);
			return mongoTemplate.find(getKeyQuery(schema,list),getEntityClass(),schema.getName());
		}
		finally {
			DynamicDataSourceContextHolder.poll();
		}
	}

	public List<BaseData> getBatchKey(String ds,POSchema schema, List<BaseData> list){
		try
		{
			DynamicDataSourceContextHolder.push(ds);
			return mongoTemplate.find(getKeyQuery(schema,list),getEntityClass(),schema.getName());
		}
		finally {
			DynamicDataSourceContextHolder.poll();
		}
	}

	public List<BaseData> selectData(String ds,POSchema schema, BaseData data){
		try
		{
			DynamicDataSourceContextHolder.push(ds);
			return mongoTemplate.find(getQueryByObject(data),getEntityClass(),schema.getName());
		}
		finally {
			DynamicDataSourceContextHolder.poll();
		}
	}

	public Page<BaseData> selectData(String ds,POSchema schema, BaseData data, Pageable page){
		try
		{
			DynamicDataSourceContextHolder.push(ds);
			if(page==null)
				page= PageRequest.of(0,20, Sort.unsorted());
			Query query = getQueryByObject(data);
			long total = mongoTemplate.count(query, BaseData.class,schema.getName());
			List<BaseData> list=mongoTemplate.find(query.with(page),BaseData.class,schema.getName());
			return new PageImpl<BaseData>(list,page,total);
		}
		finally {
			DynamicDataSourceContextHolder.poll();
		}
	}

	public List<BaseData> queryData(String ds,POSchema schema, String sql, FilterData context){
		try
		{
			DynamicDataSourceContextHolder.push(ds);
			Query query = new BasicQuery(context.getQueryBuilder().get().toString());
			return mongoTemplate.find(query,BaseData.class,schema.getName());
		}
		finally {
			DynamicDataSourceContextHolder.poll();
		}
	}

	public Page<BaseData> queryData(String ds,POSchema schema, String sql, FilterData context,  Pageable page){
		try
		{
			DynamicDataSourceContextHolder.push(ds);
			if(page==null)
				page=context.getPageable();
			else
				context.setPageable(page);
			Query query = new BasicQuery(context.getQueryBuilder().get().toString());
			long total = mongoTemplate.count(query, BaseData.class,schema.getName());
			List<BaseData> list=mongoTemplate.find(query.with(page),BaseData.class,schema.getName());
			return new PageImpl<BaseData>(list,page,total);
		}
		finally {
			DynamicDataSourceContextHolder.poll();
		}
	}

}

