package cn.ibizlab.codegen.cmd;

import cn.ibizlab.codegen.CodegenConstants;
import cn.ibizlab.codegen.DefaultGenerator;
import cn.ibizlab.codegen.Generator;
import cn.ibizlab.codegen.GeneratorNotFoundException;
import cn.ibizlab.codegen.config.CodegenConfigurator;
import cn.ibizlab.codegen.config.CodegenConfiguratorUtils;
import cn.ibizlab.codegen.*;
import io.airlift.airline.Command;
import io.airlift.airline.Option;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;

import java.io.File;
import java.util.ArrayList;
import java.util.List;

import static org.apache.commons.lang3.StringUtils.isNotEmpty;

@SuppressWarnings({"java:S106"})
@Command(name = "generate", description = "Generate code with the specified generator.")
public class Generate extends IbizLabGeneratorCommand {

    CodegenConfigurator configurator;
    Generator generator;



    @Option(name = {"-o", "--output"}, title = "output directory",
            description = "where to write the generated files (current dir by default)")
    private String output = "";

    @Option(name = {"-i", "--input-spec"}, title = "spec file",
            description = "location of the ibizlab-Model spec, as URL or file (required if not loaded via config using -c)")
    private String spec;

    @Option(name = {"-f", "--input-spec-filter"}, title = "model's filter regex",
            description = "input spec model file's filter regex, multiple regex are supported, the format of page:EntityAEditView,page:EntityAEditView,appEntity:EntityA,ctrl:main " +
                    "(or multiple options, each with -f page:EntityAEditView -f page:EntityAEditView -f appEntity:EntityA -f ctrl@FORM:main)")
    private List<String> filters;

    @Option(name = {"-t", "--template-dir"}, title = "template directories",
            description = "folder containing the template files, multiple paths are supported, the format of /templateA,/tmp/templateB " +
                    "(or multiple options, each with -t /templateA -t /tmp/templateB)")
    private List<String> templateDirs;


    @Option(name = {"--template-path"}, title = "template files relative path",
            description = "special template file's relative path, multiple paths are supported, the format of /folderA/README.md.hbs,/folderB/sub/file.json.hbs " +
                    "(or multiple options, each with --template-path /folderA/README.md.hbs --template-path /folderB/sub/file.json.hbs)")
    private List<String> templatePaths;


    @Option(name = {"--template-filter"}, title = "template file's filter regex",
            description = "template file's filter regex, multiple regex are supported, the format of /folderA,/folderB/.*/{{entities}}.java.hbs " +
                    "(or multiple options, each with --template-filter /folderA --template-filter /folderB/.*/{{entities}}.java.hbs)")
    private List<String> templateFilters;

    @Option(
            name = {"-a", "--auth"},
            title = "authorization",
            description = "adds authorization headers when fetching the definitions remotely. "
                    + "Pass in a URL-encoded string of name:header with a comma separating multiple values")
    private String auth;

    @Option(
            name = {"--global-property"},
            title = "global properties",
            description = "sets specified global properties (previously called 'system properties') in "
                    + "the format of name=value,name=value (or multiple options, each with name=value)")
    private List<String> globalProperties = new ArrayList<>();

    @Option(
            name = {"-c", "--config"},
            title = "configuration file",
            description = "Path to configuration file. It can be JSON or YAML. "
                    + "If file is JSON, the content should have the format {\"optionKey\":\"optionValue\", \"optionKey1\":\"optionValue1\"...}. "
                    + "If file is YAML, the content should have the format optionKey: optionValue. "
                    + "Supported options can be different for each language. Run config-help -g {generator name} command for language-specific config options.")
    private String configFile;

    @Option(name = {"-n","--project-name","--system-name"}, title = "package name",
            description = CodegenConstants.PROJECT_NAME)
    private String name;

    @Option(name = {"--package-name"}, title = "package name",
            description = CodegenConstants.PACKAGE_NAME_DESC)
    private String packageName;



    @Option(
            name = {"-p", "--additional-properties"},
            title = "additional properties",
            description = "sets additional properties that can be referenced by the mustache templates in the format of name=value,name=value."
                    + " You can also have multiple occurrences of this option.")
    private List<String> additionalProperties = new ArrayList<>();




    @Option(name = {"--git-host"}, title = "git host",
            description = CodegenConstants.GIT_HOST_DESC)
    private String gitHost;

    @Option(name = {"--git-user-id"}, title = "git user id",
            description = CodegenConstants.GIT_USER_ID_DESC)
    private String gitUserId;

    @Option(name = {"--git-repo-id"}, title = "git repo id",
            description = CodegenConstants.GIT_REPO_ID_DESC)
    private String gitRepoId;

    @Option(name = {"--release-note"}, title = "release note",
            description = CodegenConstants.RELEASE_NOTE_DESC)
    private String releaseNote;

    @Option(name = {"--http-user-agent"}, title = "http user agent",
            description = CodegenConstants.HTTP_USER_AGENT_DESC)
    private String httpUserAgent;


    @Override
    public void execute() {


        // this initial check allows for field-level package private injection (for unit testing)
        if (configurator == null) {
            if (configFile != null && configFile.length() > 0) {
                // attempt to load from configFile
                configurator = CodegenConfigurator.fromFile(configFile);
            }
//            else if (StringUtils.isEmpty(spec)) {
//                // if user doesn't pass configFile and does not pass spec, we can fail immediately because one of these two is required to run.
//                System.err.println("[error] Required option '-i' is missing");
//                System.exit(1);
//            }

            // if a config file wasn't specified, or we were unable to read it
            if (configurator == null) {
                // create a fresh configurator
                configurator = new CodegenConfigurator();
            }
        }



        if (isNotEmpty(spec)) {
            if (!spec.matches("^http(s)?://.*") && !new File(spec).exists()) {
                System.err.println("[error] The spec file is not found: " + spec);
                System.err.println("[error] Check the path of the ibizlab-Model spec and try again.");
                System.exit(1);
            }
            configurator.setInputSpec(spec);
        }



        if (isNotEmpty(output)) {
            configurator.setOutputDir(output);
        }

        if (isNotEmpty(auth)) {
            configurator.setAuth(auth);
        }

        if (!ObjectUtils.isEmpty(templateDirs)) {
            configurator.setTemplateDirs(templateDirs);
        }

        if (!ObjectUtils.isEmpty(templatePaths)) {
            configurator.setTemplatePaths(templatePaths);
        }

        if (!ObjectUtils.isEmpty(templateFilters)) {
            configurator.setTemplateFilters(templateFilters);
        }

        if (!ObjectUtils.isEmpty(filters)) {
            configurator.setFilters(filters);
        }

        if (isNotEmpty(packageName)) {
            configurator.setPackageName(packageName);
        }

        if (isNotEmpty(name)) {
            configurator.setProjectName(name);
        }


        if (isNotEmpty(gitHost)) {
            configurator.setGitHost(gitHost);
        }

        if (isNotEmpty(gitUserId)) {
            configurator.setGitUserId(gitUserId);
        }

        if (isNotEmpty(gitRepoId)) {
            configurator.setGitRepoId(gitRepoId);
        }

        if (isNotEmpty(releaseNote)) {
            configurator.setReleaseNote(releaseNote);
        }

        if (isNotEmpty(httpUserAgent)) {
            configurator.setHttpUserAgent(httpUserAgent);
        }



        if (globalProperties != null && !globalProperties.isEmpty()) {
            CodegenConfiguratorUtils.applyGlobalPropertiesKvpList(globalProperties, configurator);
        }

        CodegenConfiguratorUtils.applyAdditionalPropertiesKvpList(additionalProperties, configurator);


        try {

            // this null check allows us to inject for unit testing.
            if (generator == null) {
                generator = new DefaultGenerator();
            }

            generator.opts(configurator.toClientOptInput());
            generator.generate();
        } catch (GeneratorNotFoundException e) {
            System.err.println(e.getMessage());
            System.err.println("[error] Check the spelling of the generator's name and try again.");
            System.exit(1);
        }
    }
}
