package cn.ibizlab.codegen;

import cn.ibizlab.codegen.config.GlobalSettings;
import cn.ibizlab.codegen.templating.MustacheEngineAdapter;
import cn.ibizlab.codegen.templating.TemplateDefinition;
import cn.ibizlab.codegen.templating.TemplatingEngineAdapter;
import cn.ibizlab.codegen.templating.mustache.*;
import cn.ibizlab.codegen.utils.StringAdvUtils;
import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.Ticker;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.Setter;
import lombok.experimental.Accessors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.StringUtils;

import java.io.File;
import java.nio.file.Paths;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.regex.Pattern;

@Getter
@Setter
@Accessors(chain = true)
public class CodegenConfig {

    private final Logger LOGGER = LoggerFactory.getLogger(CodegenConfig.class);


    private String inputSpec;
    private String outputDir;
    private String templateDir;
    private String auth;

    public CodegenConfig setInputSpec(String inputSpec) {
        if(!StringUtils.isEmpty(inputSpec))
        {
            this.inputSpec=inputSpec.replace("\\", "/");
        }
        return this;
    }

    public CodegenConfig setOutputDir(String outputDir) {
        if(!StringUtils.isEmpty(outputDir))
        {
            this.outputDir=outputDir.replace("\\", "/");
        }
        return this;
    }

    public CodegenConfig setTemplateDir(String templateDir) {
        if(!StringUtils.isEmpty(templateDir))
        {
            this.templateDir=templateDir.replace("\\", "/");
        }
        return this;
    }

    private Map<String, Object> additionalProperties = new HashMap<>();

    private Map<String, String> typeMappings;

    private TemplatingEngineAdapter templatingEngine;

    public boolean isEnableMinimalUpdate()
    {
        return Boolean.valueOf(additionalProperties.getOrDefault("enableMinimalUpdate",false).toString());
    }
    public boolean isSkipOverwrite()
    {
        return Boolean.valueOf(additionalProperties.getOrDefault("skipOverwrite",false).toString());
    }

    public String getIgnoreFilePathOverride()
    {
        return additionalProperties.getOrDefault(CodegenConstants.IGNORE_FILE_OVERRIDE,"").toString();
    }

    private List<TemplateDefinition> templateDefinitions;

    public List<TemplateDefinition> getTemplateDefinitions()
    {
        if(templateDefinitions==null&& (!StringUtils.isEmpty(templateDir)))
        {
            templateDefinitions=new ArrayList<>();
            scanTemplate(new File(templateDir),templateDefinitions);
        }
        return templateDefinitions;
    }

    public CodegenConfig addLambda()
    {
        if(templatingEngine instanceof MustacheEngineAdapter)
        {

            additionalProperties.put("lowercase", new LowercaseLambda().generator(this));
            additionalProperties.put("uppercase", new UppercaseLambda());
            additionalProperties.put("snakecase", new SnakecaseLambda());
            additionalProperties.put("spinalcase", new SpinalcaseLambda());
            additionalProperties.put("titlecase", new TitlecaseLambda());
            additionalProperties.put("pluralize", new PluralizeLambda());
            additionalProperties.put("camelcase", new CamelCaseLambda(true).generator(this));
            additionalProperties.put("pascalcase", new CamelCaseLambda(false).generator(this));
            additionalProperties.put("indented", new IndentedLambda());
            additionalProperties.put("indented_8", new IndentedLambda(8, " "));
            additionalProperties.put("indented_12", new IndentedLambda(12, " "));
            additionalProperties.put("indented_16", new IndentedLambda(16, " "));
        }
        return this;
    }

    private void scanTemplate(File file,List<TemplateDefinition> templateDefinitions)
    {
        if(file.isDirectory())
        {
            for(File sub:file.listFiles())
                scanTemplate(sub,templateDefinitions);
        }
        else {
            String path=file.getPath().replace("\\","/");
            templateDefinitions.add(new TemplateDefinition(Paths.get("/",path.replace(this.getTemplateDir(),"")).toString()));
        }
    }



    @SuppressWarnings("static-method")
    public String sanitizeName(String name) {
        return sanitizeName(name, "\\W");
    }


    private static Cache<SanitizeNameOptions, String> sanitizedNameCache;
    static {


        int cacheSize = Integer.parseInt(GlobalSettings.getProperty(StringAdvUtils.NAME_CACHE_SIZE_PROPERTY, "500"));
        int cacheExpiry = Integer.parseInt(GlobalSettings.getProperty(StringAdvUtils.NAME_CACHE_EXPIRY_PROPERTY, "10"));
        sanitizedNameCache = Caffeine.newBuilder()
                .maximumSize(cacheSize)
                .expireAfterAccess(cacheExpiry, TimeUnit.SECONDS)
                .ticker(Ticker.systemTicker())
                .build();
    }
    /**
     * Sanitize name (parameter, property, method, etc)
     *
     * @param name            string to be sanitize
     * @param removeCharRegEx a regex containing all char that will be removed
     * @return sanitized string
     */
    public String sanitizeName(String name, String removeCharRegEx) {
        return sanitizeName(name, removeCharRegEx, new ArrayList<>());
    }

    /**
     * Sanitize name (parameter, property, method, etc)
     *
     * @param name            string to be sanitize
     * @param removeCharRegEx a regex containing all char that will be removed
     * @param exceptionList   a list of matches which should not be sanitized (i.e exception)
     * @return sanitized string
     */
    @SuppressWarnings("static-method")
    public String sanitizeName(final String name, String removeCharRegEx, ArrayList<String> exceptionList) {
        // NOTE: performance wise, we should have written with 2 replaceAll to replace desired
        // character with _ or empty character. Below aims to spell out different cases we've
        // encountered so far and hopefully make it easier for others to add more special
        // cases in the future.

        // better error handling when map/array type is invalid
        if (name == null) {
            LOGGER.error("String to be sanitized is null. Default to ERROR_UNKNOWN");
            return "ERROR_UNKNOWN";
        }

        // if the name is just '$', map it to 'value' for the time being.
        if ("$".equals(name)) {
            return "value";
        }

        SanitizeNameOptions opts = new SanitizeNameOptions(name, removeCharRegEx, exceptionList);

        return sanitizedNameCache.get(opts, sanitizeNameOptions -> {
            String modifiable = sanitizeNameOptions.getName();
            List<String> exceptions = sanitizeNameOptions.getExceptions();
            // input[] => input
            modifiable = this.sanitizeValue(modifiable, "\\[\\]", "", exceptions);

            // input[a][b] => input_a_b
            modifiable = this.sanitizeValue(modifiable, "\\[", "_", exceptions);
            modifiable = this.sanitizeValue(modifiable, "\\]", "", exceptions);

            // input(a)(b) => input_a_b
            modifiable = this.sanitizeValue(modifiable, "\\(", "_", exceptions);
            modifiable = this.sanitizeValue(modifiable, "\\)", "", exceptions);

            // input.name => input_name
            modifiable = this.sanitizeValue(modifiable, "\\.", "_", exceptions);

            // input-name => input_name
            modifiable = this.sanitizeValue(modifiable, "-", "_", exceptions);

            // a|b => a_b
            modifiable = this.sanitizeValue(modifiable, "\\|", "_", exceptions);

            // input name and age => input_name_and_age
            modifiable = this.sanitizeValue(modifiable, " ", "_", exceptions);

            // /api/films/get => _api_films_get
            // \api\films\get => _api_films_get
            modifiable = modifiable.replaceAll("/", "_");
            modifiable = modifiable.replaceAll("\\\\", "_");

            // remove everything else other than word, number and _
            // $php_variable => php_variable
            modifiable = modifiable.replaceAll(sanitizeNameOptions.getRemoveCharRegEx(), "");
            return modifiable;
        });
    }

    private String sanitizeValue(String value, String replaceMatch, String replaceValue, List<String> exceptionList) {
        if (exceptionList.size() == 0 || !exceptionList.contains(replaceMatch)) {
            return value.replaceAll(replaceMatch, replaceValue);
        }
        return value;
    }

    /**
     * Sanitize tag
     *
     * @param tag Tag
     * @return Sanitized tag
     */
    public String sanitizeTag(String tag) {
        tag = StringAdvUtils.camelize(sanitizeName(tag));

        // tag starts with numbers
        if (tag.matches("^\\d.*")) {
            tag = "Class" + tag;
        }

        return tag;
    }


    private static class SanitizeNameOptions {
        public SanitizeNameOptions(String name, String removeCharRegEx, List<String> exceptions) {
            this.name = name;
            this.removeCharRegEx = removeCharRegEx;
            if (exceptions != null) {
                this.exceptions = Collections.unmodifiableList(exceptions);
            } else {
                this.exceptions = Collections.emptyList();
            }
        }

        public String getName() {
            return name;
        }

        public String getRemoveCharRegEx() {
            return removeCharRegEx;
        }

        public List<String> getExceptions() {
            return exceptions;
        }

        private String name;
        private String removeCharRegEx;
        private List<String> exceptions;

        @Override
        public boolean equals(Object o) {
            if (this == o) return true;
            if (o == null || getClass() != o.getClass()) return false;
            SanitizeNameOptions that = (SanitizeNameOptions) o;
            return Objects.equals(getName(), that.getName()) &&
                    Objects.equals(getRemoveCharRegEx(), that.getRemoveCharRegEx()) &&
                    Objects.equals(getExceptions(), that.getExceptions());
        }

        @Override
        public int hashCode() {
            return Objects.hash(getName(), getRemoveCharRegEx(), getExceptions());
        }
    }

}