package cn.ibizlab.codegen.templating;

import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.*;
import java.nio.charset.StandardCharsets;
import java.nio.file.*;
import java.util.*;
import java.util.regex.Pattern;

/**
 * Manages the lookup, compilation, and writing of template files
 */
public class TemplateManager implements TemplatingExecutor, TemplateProcessor {
    private final TemplateManagerOptions options;
    private final TemplatingEngineAdapter engineAdapter;
    private final List<TemplatePathLocator> templateLoaders;

    private final Logger LOGGER = LoggerFactory.getLogger(TemplateManager.class);

 
    public TemplateManager(
            TemplateManagerOptions options,
            TemplatingEngineAdapter engineAdapter,
            List<TemplatePathLocator> templateLoaders) {
        this.options = options;
        this.engineAdapter = engineAdapter;
        this.templateLoaders = templateLoaders;
    }

    private String getFullTemplateFile(String name) {
        String template = templateLoaders.stream()
                .map(i -> i.getFullTemplatePath(name))
                .filter(Objects::nonNull)
                .findFirst()
                .orElse("");

        if (StringUtils.isEmpty(template)) {
            throw new TemplateNotFoundException(name);
        }

        if (name == null || name.contains("..")) {
            throw new IllegalArgumentException("Template location must be constrained to template directory.");
        }

        return template;
    }

    /**
     * returns the template content by name
     *
     * @param name the template name (e.g. model.mustache)
     * @return the contents of that template
     */
    @Override
    public String getFullTemplateContents(String name) {
        return readTemplate(getFullTemplateFile(name));
    }

    public Reader getFullTemplateReader(String name) {
        return getTemplateReader(getFullTemplateFile(name));
    }

    /**
     * Returns the path of a template, allowing access to the template where consuming literal contents aren't desirable or possible.
     *
     * @param name the template name (e.g. model.mustache)
     * @return The {@link Path} to the template
     */
    @Override
    public Path getFullTemplatePath(String name) {
        return Paths.get(getFullTemplateFile(name));
    }

    /**
     * Gets a normalized classpath resource location according to OS-specific file separator
     *
     * @param name The name of the resource file/directory to find
     *
     * @return A normalized string according to OS-specific file separator
     */
    public static String getCPResourcePath(final String name) {
        if (!"/".equals(File.separator)) {
            return name.replaceAll(Pattern.quote(File.separator), "/");
        }
        return name;
    }

    /**
     * Reads a template's contents from the specified location
     *
     * @param name The location of the template
     * @return The raw template contents
     */
    @SuppressWarnings("java:S112")
    // ignored rule java:S112 as RuntimeException is used to match previous exception type
    public String readTemplate(String name) {
        if (name == null || name.contains("..")) {
            throw new IllegalArgumentException("Template location must be constrained to template directory.");
        }
        try (Reader reader = getTemplateReader(name)) {
            if (reader == null) {
                throw new RuntimeException("no file found");
            }
//            try (Scanner s = new Scanner(reader).useDelimiter("\\A")) {
//                return s.hasNext() ? s.next() : "";
//            }
            BufferedReader br = new BufferedReader(reader);
            String line;
            List<String> list=new ArrayList<>();
            while ( (line = br.readLine()) != null )
            {
                list.add(line);

            }  return String.join(System.lineSeparator(),list);
        } catch (Exception e) {
            LOGGER.error("{}", e.getMessage(), e);
        }
        throw new RuntimeException("can't load template " + name);
    }

    @SuppressWarnings({"squid:S2095", "java:S112"})
    // ignored rule squid:S2095 as used in the CLI and it's required to return a reader
    // ignored rule java:S112 as RuntimeException is used to match previous exception type
    public Reader getTemplateReader(String name) {
        try {
            InputStream is = getInputStream(name);
            return new InputStreamReader(is, StandardCharsets.UTF_8);
        } catch (FileNotFoundException e) {
            LOGGER.error(e.getMessage());
            throw new RuntimeException("can't load template " + name);
        }
    }

    private InputStream getInputStream(String name) throws FileNotFoundException {
        InputStream is;
        is = this.getClass().getClassLoader().getResourceAsStream(getCPResourcePath(name));
        if (is == null) {
            if (name == null || name.contains("..")) {
                throw new IllegalArgumentException("Template location must be constrained to template directory.");
            }
            is = new FileInputStream(name); // May throw but never return a null value
        }
        return is;
    }

    /**
     * Writes data to a compiled template
     *
     * @param data     Input data
     * @param template Input template location
     * @param target The targeted file output location
     *
     * @return The actual file
     */
    @Override
    public File write(Map<String, Object> data, String template, File target) throws IOException {
        String templateContent="";
        if(!this.engineAdapter.handlesFile(template))
        {
            File dir=target.getParentFile();
            if(!dir.exists())
                dir.mkdirs();
            templateContent=getFullTemplateContents(template);
            if(StringUtils.isEmpty(templateContent))
            {
                Path srcPath=this.getFullTemplatePath(template);
                if(srcPath.toFile().exists()) {
                    Files.copy(srcPath, Paths.get(target.getPath()), StandardCopyOption.REPLACE_EXISTING);
                    return target;
                }
            }
        }
        else
            templateContent = this.engineAdapter.compileTemplate(this, data, template);
        if(StringUtils.isEmpty(templateContent))
            return null;
        return writeToFile(target.getPath(), templateContent);
    }


    @Override
    public String targetPath(Map<String, Object> data, String template) throws IOException {
        template=template.replace("{{packageName}}","{{packagePath}}");
        String templateContent = this.engineAdapter.compilePath(this, data, template);
        return templateContent;
    }

    @Override
    public void ignore(Path path, String context) {
        LOGGER.info("Ignored {} ({})", path, context);
    }

    @Override
    public void skip(Path path, String context) {
        LOGGER.info("Skipped {} ({})", path, context);
    }

    /**
     * Write String to a file, formatting as UTF-8
     *
     * @param filename The name of file to write
     * @param contents The contents string.
     * @return File representing the written file.
     * @throws IOException If file cannot be written.
     */
    public File writeToFile(String filename, String contents) throws IOException {

        return writeToFile(filename, contents.getBytes(StandardCharsets.UTF_8));
    }

    /**
     * Write bytes to a file
     *
     * @param filename The name of file to write
     * @param contents The contents bytes.  Typically, this is a UTF-8 formatted string.
     * @return File representing the written file.
     * @throws IOException If file cannot be written.
     */
    @Override
    public File writeToFile(String filename, byte[] contents) throws IOException {
        // Use Paths.get here to normalize path (for Windows file separator, space escaping on Linux/Mac, etc)
        File outputFile = Paths.get(filename).toFile();

        if (this.options.isMinimalUpdate()) {
            String tempFilename = filename + ".tmp";
            File tempFile = null;
            try {
                tempFile = writeToFileRaw(tempFilename, contents);
                if (!filesEqual(tempFile, outputFile)) {
                    LOGGER.info("writing file {}", filename);
                    Files.move(tempFile.toPath(), outputFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
                    tempFile = null;
                } else {
                    LOGGER.info("skipping unchanged file {}", filename);
                }
            } finally {
                if (tempFile != null && tempFile.exists()) {
                    try {
                        Files.delete(tempFile.toPath());
                    } catch (Exception ex) {
                        LOGGER.error("Error removing temporary file {}", tempFile, ex);
                    }
                }
            }
        } else {
            LOGGER.info("writing file {}", filename);
            outputFile = writeToFileRaw(filename, contents);
        }

        return outputFile;
    }

    private File writeToFileRaw(String filename, byte[] contents) throws IOException {
        // Use Paths.get here to normalize path (for Windows file separator, space escaping on Linux/Mac, etc)
        File output = Paths.get(filename).toFile();
        if (this.options.isSkipOverwrite() && output.exists()) {
            LOGGER.info("skip overwrite of file {}", filename);
            return output;
        }

        if (output.getParent() != null && !new File(output.getParent()).exists()) {
            File parent = Paths.get(output.getParent()).toFile();
            parent.mkdirs();
        }
        Files.write(output.toPath(), contents);

        return output;
    }

    private boolean filesEqual(File file1, File file2) throws IOException {
        return file1.exists() && file2.exists() && Arrays.equals(Files.readAllBytes(file1.toPath()), Files.readAllBytes(file2.toPath()));
    }
}
