package cn.ibizlab.util.service;

import cn.ibizlab.util.domain.IBZUSER;
import cn.ibizlab.util.errors.BadRequestAlertException;
import cn.ibizlab.util.helper.CachedBeanCopier;
import cn.ibizlab.util.mapper.IBZUSERMapper;
import cn.ibizlab.util.security.AuthTokenUtil;
import cn.ibizlab.util.security.AuthenticationInfo;
import cn.ibizlab.util.security.AuthenticationUser;
import com.baomidou.mybatisplus.core.conditions.query.QueryWrapper;
import com.baomidou.mybatisplus.extension.service.impl.ServiceImpl;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnExpression;
import org.springframework.cache.annotation.CacheEvict;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.security.core.authority.AuthorityUtils;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import org.springframework.stereotype.Service;
import org.springframework.util.DigestUtils;
import org.springframework.util.StringUtils;

/**
 * 实体[IBZUSER] 服务对象接口实现
 */
@Service("IBZUSERService")
@ConditionalOnExpression("(!${ibiz.enablePermissionValid:false})&&'${ibiz.auth.service:IBZUAAUserService}'.equals('IBZUSERService')")
public class IBZUSERServiceImpl extends ServiceImpl<IBZUSERMapper, IBZUSER> implements IBZUSERService, AuthenticationUserService {

	@Value("${ibiz.auth.pwencrymode:0}")
	private int pwencrymode;


	@Value("${ibiz.enablePermissionValid:false}")
	boolean enablePermissionValid;  //是否开启权限校验

	@Value("${ibiz.systemid}")
	private String systemId;

	@Override
	public boolean isEnablePermissionValid() {
		return enablePermissionValid;
	}

	@Override
	public String getSystemId() {
		return systemId;
	}


	@Override
	@Cacheable( value="ibzuaa_users", key = "#root.target.systemId+':getByUsername:'+#p0")
	public AuthenticationUser loadUserByUsername(String username) {
		if(StringUtils.isEmpty(username)) {
			throw new UsernameNotFoundException("用户名为空");
		}
		QueryWrapper<IBZUSER> conds = new QueryWrapper<IBZUSER>();
		String[] data = username.split("[|]");
		String loginname = "";
		String domains = "";
		if(data.length>0) {
			loginname = data[0].trim();
		}
		if(data.length>1) {
			domains = data[1].trim();
		}
		if(!StringUtils.isEmpty(loginname)) {
			conds.eq("loginname",loginname);
		}
		if(!StringUtils.isEmpty(domains)) {
			conds.eq("domains",domains);
		}
		IBZUSER user = this.getOne(conds);
		if(user == null) {
			throw new BadRequestAlertException("登录失败","IBZUSER",username);
		}
		else {
			user.setUsername(username);
			return createUserDetails(user);
		}
	}


	@Autowired
	private AuthTokenUtil jwtTokenUtil;


	@Override
	public AuthenticationInfo loadUserByLogin(String username, String password){
		AuthenticationUser authuserdetail = loadUserByUsername(username);
		if(pwencrymode == 1){
			password = DigestUtils.md5DigestAsHex(password.getBytes());
		}
		else if(pwencrymode == 2){
			password = DigestUtils.md5DigestAsHex(String.format("%1$s||%2$s", username, password).getBytes());
		}
		if(!authuserdetail.getPassword().equals(password)) {
			throw new BadRequestAlertException("用户名密码错误","IBZUSER",username);
		}
		final String token = jwtTokenUtil.generateToken(authuserdetail);
		// 返回 token
		return new AuthenticationInfo(token,authuserdetail);
	}


	@CacheEvict( value="ibzuaa_users", key = "'glob:*getByUsername:'+#p0")
	public void resetByUsername(String username) {

	}

	public AuthenticationUser createUserDetails(IBZUSER user) {
		AuthenticationUser userdatail = new AuthenticationUser();
		CachedBeanCopier.copy(user,userdatail);
		userdatail.setSuperuser(user.getSuperuser());
    	if(userdatail.getSuperuser() == 1){
    		userdatail.setAuthorities(AuthorityUtils.createAuthorityList("ROLE_USERS","ROLE_SUPERADMIN"));
    	}
    	else {
			userdatail.setAuthorities(AuthorityUtils.createAuthorityList("ROLE_USERS"));
		}
		return userdatail;
	}
}