import { Util } from "@/utils";

/**
 * 面板成员模型
 *
 * @export
 * @class PanelDetailModel
 */
export class PanelDetailModel {

    /**
     * 成员标题
     *
     * @type {string}
     * @memberof PanelDetailModel
     */
    public caption: string = '';

    /**
     * 成员类型
     *
     * @type {string}
     * @memberof PanelDetailModel
     */
    public itemType: string = '';

    /**
     * 面板对象
     *
     * @type {*}
     * @memberof PanelDetailModel
     */
    public panel: any = null;

    /**
     * 成员名称
     *
     * @type {string}
     * @memberof PanelDetailModel
     */
    public name: string = '';

    /**
     * 成员是否显示
     *
     * @type {boolean}
     * @memberof PanelDetailModel
     */
    public visible: boolean = true;

    /**
     * 布局模式
     *
     * @type {string}
     * @memberof PanelDetailModel
     */
    public layout: string = '';

    /**
     *  布局占位（边缘布局使用）
     *
     * @type {string}
     * @memberof PanelDetailModel
     */
    public layoutPos: string = '';

    /**
     * 布局高度
     *
     * @type {number}
     * @memberof PanelDetailModel
     */
    public layoutHeight: number = 0;

    /**
     * 高度模式
     *
     * @type {string}
     * @memberof PanelDetailModel
     */
    public heightMode: string = '';

    /**
     * 布局宽度
     *
     * @type {number}
     * @memberof PanelDetailModel
     */
    public layoutWidth: number = 0;

    /**
     * 宽度模式
     *
     * @type {string}
     * @memberof PanelDetailModel
     */
    public widthMode: string = '';

    /**
     * 下档间隔
     *
     * @type {string}
     * @memberof PanelDetailModel
     */
    public spacingBottom: string = '';

    /**
     * 左侧间隔
     *
     * @type {string}
     * @memberof PanelDetailModel
     */
    public spacingLeft: string = '';

    /**
     * 右侧间隔
     *
     * @type {string}
     * @memberof PanelDetailModel
     */
    public spacingRight: string = '';

    /**
     * 顶部间隔
     *
     * @type {string}
     * @memberof PanelDetailModel
     */
    public spacingTop: string = '';

    /**
     * 自身水平对齐模式
     *
     * @type {string}
     * @memberof PanelDetailModel
     */
    public hAlignSelf: string = '';

    /**
     * 自身垂直对齐模式
     *
     * @type {string}
     * @memberof PanelDetailModel
     */
    public vAlignSelf: string = '';

    /**
     * Flex延伸
     *
     * @type {number}
     * @memberof PanelDetailModel
     */
    public flexGrow: number = 0;

    /**
     * flex布局参数
     *
     * @type {*}
     * @memberof PanelDetailModel
     */
    public flexParams: any = {};

    /**
     * 是否显示标题
     *
     * @type {boolean}
     * @memberof PanelDetailModel
     */
    public isShowCaption: boolean = false;

    /**
     * 界面样式表
     *
     * @type {string}
     * @memberof PanelDetailModel
     */
    public sysCss: string = ''

    /**
     * 预置类型
     *
     * @type {string}
     * @memberof PanelDetailModel
     */
    public predefinedType: string = '';

    /**
     * Creates an instance of PanelDetailModel.
     * PanelDetailModel 实例
     * 
     * @param {*} [opts={}]
     * @memberof PanelDetailModel
     */
    constructor(opts: any = {}) {
        this.caption = opts.caption;
        this.itemType = opts.itemType;
        this.panel = opts.panel;
        this.name = opts.name;
        this.visible = opts.visible;
        this.layout = opts.layout;
        this.layoutPos = opts.layoutPos;
        this.layoutHeight = opts.layoutHeight;
        this.heightMode = opts.heightMode;
        this.layoutWidth = opts.layoutWidth;
        this.widthMode = opts.widthMode;
        this.spacingBottom = opts.spacingBottom;
        this.spacingLeft = opts.spacingLeft;
        this.spacingRight = opts.spacingRight;
        this.spacingTop = opts.spacingTop;
        this.hAlignSelf = opts.hAlignSelf;
        this.vAlignSelf = opts.vAlignSelf;
        this.flexGrow = opts.flexGrow;
        this.flexParams = opts.flexParams;
        this.isShowCaption = opts.isShowCaption;
        this.sysCss = opts.sysCss;
        this.predefinedType = opts.predefinedType;
    }

    /**
     * 获取元素样式（元素，包含内容盒子 大小/边距/内容 的样式）
     *
     * @memberof PanelDetailModel
     */
    public getElementStyle() {
        const elementStyle = {};
        Object.assign(elementStyle, this.getBoxSizeStyle());
        Object.assign(elementStyle, this.getBoxSpacingStyle());
        Object.assign(elementStyle, this.getBoxContentStyle());
        return elementStyle;
    }

    /**
     * 获取盒子间隔样式（元素）
     *
     * @memberof PanelDetailModel
     */
    public getBoxSpacingStyle() {
        const boxStyle = {};
        // 上方间隔模式
        if (this.spacingTop) {
            Object.assign(boxStyle, Util.getItemSpacingStyle(this.spacingTop, 'top'));
        }
        // 下方间隔模式
        if (this.spacingBottom) {
            Object.assign(boxStyle, Util.getItemSpacingStyle(this.spacingBottom, 'bottom'));
        }
        // 左侧间隔模式
        if (this.spacingLeft) {
            Object.assign(boxStyle, Util.getItemSpacingStyle(this.spacingLeft, 'left'));
        }
        // 右侧间隔模式
        if (this.spacingRight) {
            Object.assign(boxStyle, Util.getItemSpacingStyle(this.spacingRight, 'right'));
        }
        return boxStyle;
    }

    /**
     * 获取盒子大小样式（元素）
     *
     * @memberof PanelDetailModel
     */
    public getBoxSizeStyle() {
        const boxStyle = {};
        if (this.widthMode || this.layoutWidth) {
            Object.assign(boxStyle, Util.getBoxSize("WIDTH", this.widthMode, this.layoutWidth));
        }
        if (this.heightMode || this.layoutHeight) {
            Object.assign(boxStyle, Util.getBoxSize("HEIGHT", this.heightMode, this.layoutHeight));
        }
        return boxStyle;
    }

    /**
     * 获取盒子内容样式（元素，包含内容换行模式，内容水平对齐，内容垂直对齐）
     *
     * @memberof PanelDetailModel
     */
    public getBoxContentStyle() {
        const boxStyle = {};
        return boxStyle;
    }

    /**
     * 获取盒子边框样式（元素）
     *
     * @memberof PanelDetailModel
     */
    public getBoxBorderStyle() {
        const boxStyle = {};
        return boxStyle;
    }

    /**
     * 获取通用布局样式(SAMPLEFLEX/FLEX)（布局，包含约束内容区布局的样式，包含内容区的对齐方式）
     *
     * @return {*} 
     * @memberof PanelDetailModel
     */
     public getCommonLayoutStyle() {
        const layoutStyle = {};
        Object.assign(layoutStyle, this.getBoxSelfAlignStyle());
        Object.assign(layoutStyle, this.getBoxLayOutStyle());
        Object.assign(layoutStyle, this.getBoxSelfAlignStyle());
        return layoutStyle;
    }

    /**
     * 获取边缘布局样式
     *
     * @memberof PanelDetailModel
     */
    public getBorderLayoutStyle() {
        const borderLayoutStyle: any = { width: '100%', height: '100%' };
        if (this.layoutHeight) {
            Object.assign(borderLayoutStyle, Util.getBoxSize("HEIGHT", this.heightMode, this.layoutHeight));
        }
        // TODO 合入样式代码
        return borderLayoutStyle;
    }

    /**
     * 获取自身对齐模式(布局)
     *
     * @memberof PanelDetailModel
     */
    public getBoxSelfAlignStyle() {
        const boxStyle = {};
        // 自身对齐方式
        if (this.vAlignSelf || this.hAlignSelf) {
            Object.assign(boxStyle, { 'display': 'flex' });
            // 自身垂直对齐模式
            switch (this.vAlignSelf) {
                case 'TOP':
                    Object.assign(boxStyle, { 'align-items': 'flex-start' });
                    break;
                case 'MIDDLE':
                    Object.assign(boxStyle, { 'align-items': 'center' });
                    break;
                case 'BOTTOM':
                    Object.assign(boxStyle, { 'align-items': 'flex-end' });
                    break;
                default:
                    break;
            }
            // 自身水平对齐模式
            switch (this.hAlignSelf) {
                case 'LEFT':
                    Object.assign(boxStyle, { 'justify-content': 'flex-start' });
                    break;
                case 'CENTER':
                    Object.assign(boxStyle, { 'justify-content': 'center' });
                    break;
                case 'RIGHT':
                    Object.assign(boxStyle, { 'justify-content': 'flex-end' });
                    break;
                case 'JUSTIFY':
                    Object.assign(boxStyle, { 'justify-content': 'space-between' });
                    break;
                default:
                    break;
            }
        }
        return boxStyle;
    }

    /**
     * 获取盒子布局样式(布局)
     *
     * @memberof PanelDetailModel
     */
    public getBoxLayOutStyle() {
        const boxStyle = {};
        // 识别FLEX
        if (this.layout == 'FLEX') {
            Object.assign(boxStyle, { 'display': 'flex', 'flex-grow': this.flexGrow ? this.flexGrow : 0 });
        }
        // 识别flex布局方向，横轴对齐，纵轴对齐
        if (this.flexParams.align || this.flexParams.dir || this.flexParams.vAlign) {
            Object.assign(boxStyle, { 'display': 'flex' });
            if (this.flexParams.dir) {
                Object.assign(boxStyle, { 'flex-direction': this.flexParams.dir });
            }
            if (this.flexParams.align) {
                Object.assign(boxStyle, { 'justify-content': this.flexParams.align });
            }
            if (this.flexParams.vAlign) {
                Object.assign(boxStyle, { 'align-items': this.flexParams.vAlign });
            }
        }
        // 识别SIMPLEFLEX
        if (this.layout == 'SIMPLEFLEX') {
            if (this.flexGrow) {
                Object.assign(boxStyle, { 'width': `${(100 / 12) * this.flexGrow}%`, 'height': '100%' });
            } else {
                // 简单FLEX布局自适应
                Object.assign(boxStyle, { 'flex-grow': 1, 'min-width': `${(100 / 12)}%`, 'height': '100%' });
            }
        }
        // 识别边缘布局
        if (this.layout == 'BORDER') {
            Object.assign(boxStyle, { 'display': 'flex' });
        }
        return boxStyle;
    }

}